/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.dataaccess.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.Query;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.kuali.kfs.gl.Constant;
import org.kuali.kfs.gl.OJBUtility;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.businessobject.AccountStatusBaseFunds;
import org.kuali.kfs.module.ld.businessobject.EmployeeFunding;
import org.kuali.kfs.module.ld.businessobject.LaborCalculatedSalaryFoundationTracker;
import org.kuali.kfs.module.ld.dataaccess.LaborCalculatedSalaryFoundationTrackerDao;
import org.kuali.kfs.module.ld.util.ConsolidationUtil;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * This is the data access object for calculated salary foundation tracker
 *
 * @see org.kuali.kfs.module.ld.businessobject.LaborCalculatedSalaryFoundationTracker
 */
public class LaborCalculatedSalaryFoundationTrackerDaoOjb extends PlatformAwareDaoBaseOjb implements
        LaborCalculatedSalaryFoundationTrackerDao {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public List<LaborCalculatedSalaryFoundationTracker> findCSFTrackers(final Map fieldValues, final boolean isConsolidated) {
        LOG.debug("Start findCSFTrackers()");

        final List<LaborCalculatedSalaryFoundationTracker> csfTrackerCollection = new ArrayList<>();
        if (isConsolidated) {
            final List<String> groupByList = getGroupByList(isConsolidated);
            final List<String> attributeList = getAttributeListForCSFTracker(isConsolidated, false);

            final Iterator<Object[]> queryResults = findConsolidatedCSFTrackerRawData(fieldValues, groupByList,
                    attributeList);

            while (queryResults != null && queryResults.hasNext()) {
                csfTrackerCollection.add(marshalCSFTracker(queryResults.next()));
            }
        } else {
            csfTrackerCollection.addAll(findDetailedCSFTrackerRawData(fieldValues));
        }
        return csfTrackerCollection;
    }

    @Override
    public List<AccountStatusBaseFunds> findCSFTrackersAsAccountStatusBaseFunds(
            final Map fieldValues,
            final boolean isConsolidated) {
        LOG.debug("Start findCSFTrackersAsAccountStatusBaseFunds()");

        final List<String> groupByList = getGroupByList(isConsolidated);
        final List<String> attributeList = getAttributeListForCSFTracker(isConsolidated, false);

        final Iterator<Object[]> queryResults = findConsolidatedCSFTrackerRawData(fieldValues, groupByList,
                attributeList);
        final List<AccountStatusBaseFunds> baseFundsCollection = new ArrayList<>();
        while (queryResults != null && queryResults.hasNext()) {
            baseFundsCollection.add(marshalCSFTrackerAsAccountStatusBaseFunds(queryResults.next()));
        }
        return baseFundsCollection;
    }

    @Override
    public List<EmployeeFunding> findCSFTrackersAsEmployeeFunding(final Map fieldValues, final boolean isConsolidated) {
        LOG.debug("Start findCSFTrackersAsEmployeeFunding()");

        final List<LaborCalculatedSalaryFoundationTracker> csfTrackerCollection = findCSFTrackers(fieldValues,
                isConsolidated);
        final List<EmployeeFunding> employeeFundingCollection = new ArrayList<>();
        for (final LaborCalculatedSalaryFoundationTracker csfTracker : csfTrackerCollection) {
            final EmployeeFunding employeeFunding = new EmployeeFunding();
            ObjectUtil.buildObject(employeeFunding, csfTracker);
            employeeFundingCollection.add(employeeFunding);
        }
        return employeeFundingCollection;
    }

    // get the Consolidated CSF trackers according to the given criteria
    protected Iterator<Object[]> findConsolidatedCSFTrackerRawData(
            final Map fieldValues, final List<String> groupByList,
            final List<String> attributeList) {
        final Criteria tempCriteria1 = new Criteria();
        tempCriteria1.addEqualTo(KFSPropertyConstants.CSF_DELETE_CODE, LaborConstants.DASHES_DELETE_CODE);

        final Criteria tempCriteria2 = new Criteria();
        tempCriteria2.addIsNull(KFSPropertyConstants.CSF_DELETE_CODE);

        /* KFSPropertyConstants.CSF_DELETE_CODE = "-" OR is null */
        tempCriteria2.addOrCriteria(tempCriteria1);

        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LaborCalculatedSalaryFoundationTracker());
        criteria.addAndCriteria(tempCriteria2);

        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LaborCalculatedSalaryFoundationTracker.class,
                criteria);

        final String[] groupBy = groupByList.toArray(new String[groupByList.size()]);
        query.addGroupBy(groupBy);

        final String[] attributes = attributeList.toArray(new String[attributeList.size()]);
        query.setAttributes(attributes);

        return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
    }

    // get the detailed CSF trackers according to the given criteria
    protected Collection<LaborCalculatedSalaryFoundationTracker> findDetailedCSFTrackerRawData(final Map fieldValues) {
        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LaborCalculatedSalaryFoundationTracker());
        final Query query = QueryFactory.newQuery(LaborCalculatedSalaryFoundationTracker.class, criteria);
        return getPersistenceBrokerTemplate().getCollectionByQuery(query);
    }

    // marshal into CalculatedSalaryFoundationTracker from the query result
    protected LaborCalculatedSalaryFoundationTracker marshalCSFTracker(final Object[] queryResult) {
        final LaborCalculatedSalaryFoundationTracker CSFTracker = new LaborCalculatedSalaryFoundationTracker();
        final List<String> keyFields = getAttributeListForCSFTracker(false, true);

        ObjectUtil.buildObject(CSFTracker, queryResult, keyFields);
        return CSFTracker;
    }

    // marshal into AccountStatusBaseFunds from the query results
    protected AccountStatusBaseFunds marshalCSFTrackerAsAccountStatusBaseFunds(final Object[] queryResult) {
        final AccountStatusBaseFunds baseFunds = new AccountStatusBaseFunds();
        final List<String> keyFields = getAttributeListForCSFTracker(false, true);

        ObjectUtil.buildObject(baseFunds, queryResult, keyFields);
        return baseFunds;
    }

    // define a list of attributes that are used as the grouping criteria
    protected List<String> getGroupByList(final boolean isConsolidated) {
        final List<String> groupByList = new ArrayList<>();
        groupByList.add(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR);
        groupByList.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        groupByList.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        groupByList.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);

        if (!isConsolidated) {
            groupByList.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
            groupByList.add(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        }
        return groupByList;
    }

    // define the return attribute list
    protected List<String> getAttributeList(final boolean isConsolidated) {
        final List<String> attributeList = getGroupByList(isConsolidated);

        if (isConsolidated) {
            attributeList.add("'" + Constant.CONSOLIDATED_SUB_ACCOUNT_NUMBER + "'");
            attributeList.add("'" + Constant.CONSOLIDATED_SUB_OBJECT_CODE + "'");
        }
        return attributeList;
    }

    // define the return attribute list for CSF tracker query
    protected List<String> getAttributeListForCSFTracker(final boolean isConsolidated, final boolean isAttributeNameNeeded) {
        final List<String> attributeList = getAttributeList(isConsolidated);

        if (!isAttributeNameNeeded) {
            attributeList.add(ConsolidationUtil.sum(KFSPropertyConstants.CSF_FULL_TIME_EMPLOYMENT_QUANTITY));
            attributeList.add(ConsolidationUtil.sum(KFSPropertyConstants.CSF_AMOUNT));
        } else {
            attributeList.add(KFSPropertyConstants.CSF_FULL_TIME_EMPLOYMENT_QUANTITY);
            attributeList.add(KFSPropertyConstants.CSF_AMOUNT);
        }
        return attributeList;
    }
}
