/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntryFieldUtil;
import org.kuali.kfs.sys.KFSPropertyConstants;

import java.util.Comparator;
import java.util.Map;

/**
 * Special sort comparator used for sorting the input origin entry file.  It sorts by the same keys as the generated
 * balance file.
 */
public class LaborEncumbranceBalanceOriginEntrySortComparator implements Comparator<String> {

    private static final Logger LOG = LogManager.getLogger();

    private final LaborOriginEntryFieldUtil laborOriginEntryFieldUtil = new LaborOriginEntryFieldUtil();
    private final Map<String, Integer> positionMap = laborOriginEntryFieldUtil.getFieldBeginningPositionMap();
    private final Map<String, Integer> lengthMap = laborOriginEntryFieldUtil.getFieldLengthMap();

    // Decode all of these up front to avoid the decoding of the information on
    // *every* comparison, which could happen quite a number of times
    private final int EMPLID_POS = positionMap.get(KFSPropertyConstants.EMPLID);
    private final int EMPLID_LEN = lengthMap.get(KFSPropertyConstants.EMPLID);
    private final int POSITION_NUMBER_POS = positionMap.get(KFSPropertyConstants.POSITION_NUMBER);
    private final int POSITION_NUMBER_LEN = lengthMap.get(KFSPropertyConstants.POSITION_NUMBER);
    private final int CHART_OF_ACCOUNTS_CODE_POS = positionMap.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
    private final int CHART_OF_ACCOUNTS_CODE_LEN = lengthMap.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
    private final int ACCOUNT_NUMBER_POS = positionMap.get(KFSPropertyConstants.ACCOUNT_NUMBER);
    private final int ACCOUNT_NUMBER_LEN = lengthMap.get(KFSPropertyConstants.ACCOUNT_NUMBER);
    private final int SUB_ACCOUNT_NUMBER_POS = positionMap.get(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
    private final int SUB_ACCOUNT_NUMBER_LEN = lengthMap.get(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
    private final int FINANCIAL_OBJECT_TYPE_CODE_POS = positionMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);
    private final int FINANCIAL_OBJECT_TYPE_CODE_LEN = lengthMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);
    private final int FINANCIAL_OBJECT_CODE_POS = positionMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
    private final int FINANCIAL_OBJECT_CODE_LEN = lengthMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
    private final int FINANCIAL_SUB_OBJECT_CODE_POS = positionMap.get(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
    private final int FINANCIAL_SUB_OBJECT_CODE_LEN = lengthMap.get(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);

    private final StringBuilder sb = new StringBuilder(EMPLID_LEN + POSITION_NUMBER_LEN +
                                                     CHART_OF_ACCOUNTS_CODE_LEN + ACCOUNT_NUMBER_LEN + SUB_ACCOUNT_NUMBER_LEN + FINANCIAL_OBJECT_TYPE_CODE_LEN +
                                                     FINANCIAL_OBJECT_CODE_LEN + FINANCIAL_SUB_OBJECT_CODE_LEN);

    @Override
    public int compare(String originEntryLine1, String originEntryLine2) {
        // clear out the string buffer for the next use
        sb.setLength(0);

        try {
            sb.append(originEntryLine1.substring(EMPLID_POS, EMPLID_POS + EMPLID_LEN).trim());
            sb.append(originEntryLine1, POSITION_NUMBER_POS, POSITION_NUMBER_POS + POSITION_NUMBER_LEN);
            sb.append(originEntryLine1, CHART_OF_ACCOUNTS_CODE_POS,
                    CHART_OF_ACCOUNTS_CODE_POS + CHART_OF_ACCOUNTS_CODE_LEN);
            sb.append(originEntryLine1, ACCOUNT_NUMBER_POS, ACCOUNT_NUMBER_POS + ACCOUNT_NUMBER_LEN);
            sb.append(originEntryLine1, SUB_ACCOUNT_NUMBER_POS, SUB_ACCOUNT_NUMBER_POS + SUB_ACCOUNT_NUMBER_LEN);
            sb.append(originEntryLine1, FINANCIAL_OBJECT_CODE_POS,
                    FINANCIAL_OBJECT_CODE_POS + FINANCIAL_OBJECT_CODE_LEN);
            sb.append(originEntryLine1, FINANCIAL_SUB_OBJECT_CODE_POS,
                    FINANCIAL_SUB_OBJECT_CODE_POS + FINANCIAL_SUB_OBJECT_CODE_LEN);
            sb.append(originEntryLine1, FINANCIAL_OBJECT_TYPE_CODE_POS,
                    FINANCIAL_OBJECT_TYPE_CODE_POS + FINANCIAL_OBJECT_TYPE_CODE_LEN);
            originEntryLine1 = sb.toString();
        } catch (final Exception ex) {
            LOG.error("Exception while attempting to sort Encumbrance origin entry record: \n{}", originEntryLine1,
                    ex);
            return 1;
        }

        try {
            sb.setLength(0);
            sb.append(originEntryLine2.substring(EMPLID_POS, EMPLID_POS + EMPLID_LEN).trim());
            sb.append(originEntryLine2, POSITION_NUMBER_POS, POSITION_NUMBER_POS + POSITION_NUMBER_LEN);
            sb.append(originEntryLine2, CHART_OF_ACCOUNTS_CODE_POS,
                    CHART_OF_ACCOUNTS_CODE_POS + CHART_OF_ACCOUNTS_CODE_LEN);
            sb.append(originEntryLine2, ACCOUNT_NUMBER_POS, ACCOUNT_NUMBER_POS + ACCOUNT_NUMBER_LEN);
            sb.append(originEntryLine2, SUB_ACCOUNT_NUMBER_POS, SUB_ACCOUNT_NUMBER_POS + SUB_ACCOUNT_NUMBER_LEN);
            sb.append(originEntryLine2, FINANCIAL_OBJECT_CODE_POS,
                    FINANCIAL_OBJECT_CODE_POS + FINANCIAL_OBJECT_CODE_LEN);
            sb.append(originEntryLine2, FINANCIAL_SUB_OBJECT_CODE_POS,
                    FINANCIAL_SUB_OBJECT_CODE_POS + FINANCIAL_SUB_OBJECT_CODE_LEN);
            sb.append(originEntryLine2, FINANCIAL_OBJECT_TYPE_CODE_POS,
                    FINANCIAL_OBJECT_TYPE_CODE_POS + FINANCIAL_OBJECT_TYPE_CODE_LEN);
            originEntryLine2 = sb.toString();
        } catch (final Exception ex) {
            LOG.error("Exception while attempting to sort Encumbrance origin entry record: \n{}", originEntryLine2,
                    ex);
            return -1;
        }

        return originEntryLine1.compareTo(originEntryLine2);
    }
}
