/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.batch.BatchSortUtil;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.sys.batch.AbstractStep;

import java.io.File;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * Sorts the incoming encumbrance balance.
 */
public class LaborSortEncumbranceFileStep extends AbstractStep {

    private static final Logger LOG = LogManager.getLogger();
    private String batchFileDirectoryName;

    /**
     * Sorts the incoming encumbrance balance file.
     *
     * @param jobName       the name of the job this step is being run as part of
     * @param jobRunDate    the time/date the job was started
     * @return              true if the job completed successfully, false if otherwise
     */
    @Override
    public boolean execute(final String jobName, final Date jobRunDate) {
        LOG.debug("execute() started");

        boolean succeeded = true;

        final File batchFileDirectory = new File(batchFileDirectoryName);
        final List<File> doneFileList = Arrays.asList(batchFileDirectory.listFiles((batchFileDirectory1, name) ->
                name.startsWith(LaborConstants.Encumbrance.ENCUMBRANCE_INPUT_FILE) && name.endsWith(
                        GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION)));

        final String outputFileName = batchFileDirectoryName + File.separator
                                      + LaborConstants.Encumbrance.SORTED_ENCUMBRANCE_INPUT_FILE
                                      + GeneralLedgerConstants.BatchFileSystem.EXTENSION;

        for (final File doneFile : doneFileList) {
            final File inputFile = getDataFile(doneFile);
            if (inputFile != null && inputFile.exists()) {
                LOG.debug("Sorting: \n   {}\nto\n   {}", inputFile::getAbsolutePath, () -> outputFileName);

                BatchSortUtil.sortTextFileWithFields(inputFile.getAbsolutePath(),
                        outputFileName,
                        new LaborEncumbranceBalanceOriginEntrySortComparator()
                );
                inputFile.setLastModified(jobRunDate.getTime());
            } else {
                LOG.warn("Unable to find {}, sorting skipped.  No output file created.", inputFile::getAbsolutePath);
                succeeded = false;
            }
        }

        return succeeded;
    }

    /**
     * Retrieve the dat file that matches the passed in done file.
     * @param doneFile      The done file used to determine the dat file
     * @return              Returns the dat file
     */
    private File getDataFile(final File doneFile) {
        final String doneFileAbsPath = doneFile.getAbsolutePath();
        if (!doneFileAbsPath.endsWith(GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION)) {
            LOG.error("Done file name must end with {}", GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION);
            throw new IllegalArgumentException(
                    "Done file name must end with " + GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION);
        }
        final String dataFileAbsPath =
                StringUtils.removeEnd(doneFileAbsPath, GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION)
                + LaborConstants.Encumbrance.DAT_FILE_EXTENSION;
        final File dataFile = new File(dataFileAbsPath);
        if (!dataFile.exists() || !dataFile.canRead()) {
            LOG.error("Cannot find/read data file {}", dataFileAbsPath);
            return null;
        }
        return dataFile;
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }
}
