/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.batch.LaborCalculateEncumbranceAdjustmentsStep;
import org.kuali.kfs.module.ld.batch.dataaccess.LaborEncumbranceAdjustmentDao;
import org.kuali.kfs.module.ld.batch.report.LaborEncumbranceReportWriterService;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.batch.LaborOriginEntryBalanceRecordComparator;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntryFieldUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.sys.batch.service.WrappingBatchService;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.sql.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Main service which calculates the needed adjustments.  Used by the related batch steps.
 */
public class LaborEncumbranceAdjustmentService {

    private static final Logger LOG = LogManager.getLogger();
    private LaborEncumbranceAdjustmentDao laborEncumbranceAdjustmentDao;
    private DateTimeService dateTimeService;
    private LaborEncumbranceReportWriterService reportWriterService;
    private PersonService personService;
    private ParameterService parameterService;
    private int errorCount;
    private int inputLinesProcessed;
    private int balanceLinesProcessed;
    private int newBalanceCount;
    private int removedBalanceCount;
    private int increasedBalanceCount;
    private int decreasedBalanceCount;
    private int matchingBalanceCount;
    private int outputCount;
    private KualiDecimal transactionLedgerEntryAmountBalance;
    private LinkedList<String> errorMessages;

    /**
     * Builds a sorted text version of the balance table (IE encumbrances only) for use by the encumbrance balance
     * file comparison.
     */
    public void buildBalanceFile(final Integer fiscalYear, final File outputFile) {
        LOG.debug("buildBalanceFile() started");
        laborEncumbranceAdjustmentDao.buildFileForEncumbranceBalances(fiscalYear, outputFile);
    }

    /**
     * Outputs an origin entry file with the needed transactions to adjust the existing balances in the balance table
     * to the values from the input file.
     *
     * Note: The input and balance files *MUST* be sorted identically. They are walked over in order to detect
     * additions/removals.
     */
    public boolean buildEncumbranceDifferenceFile(final File inputFile, final File balanceFile, final File outputFile,
            final File errorFile) {
        LOG.debug("buildEncumbranceDifferenceFile() started");

        final Date currentDate = dateTimeService.getCurrentSqlDate();
        LaborOriginEntry lastOriginEntry = null;
        final LaborOriginEntryBalanceRecordComparator lineComparator = new LaborOriginEntryBalanceRecordComparator();
        final Map<String, Integer> lMap = new LaborOriginEntryFieldUtil().getFieldLengthMap();
        setupReportTrackingMetaData();

        int lineNumber = 1;

        try (BufferedReader inputReader = new BufferedReader(new FileReader(inputFile)); BufferedReader balanceReader
                = new BufferedReader(new FileReader(balanceFile)); PrintStream outputStream =
                     new PrintStream(outputFile); PrintStream errorStream = new PrintStream(errorFile)) {
            LOG.info("All Files Opened - starting comparisons");

            String encumbranceLine;
            String balanceLine;
            LaborOriginEntry encumbranceEntryLine = new LaborOriginEntry();

            encumbranceLine = inputReader.readLine();
            balanceLine = balanceReader.readLine();

            while (encumbranceLine != null || balanceLine != null) {
                LOG.debug("Processing Lines:\nHCM: {}\nBAL: {}", encumbranceLine, balanceLine);

                boolean encumbranceLineReadError = false;
                boolean advanceEncumbranceFile = false;
                boolean advanceBalanceFile = false;
                if (encumbranceLine != null) {
                    try {
                        final List<Message> parsingMessages =
                                encumbranceEntryLine.setFromTextFileForBatch(encumbranceLine, lineNumber);
                        if (!parsingMessages.isEmpty()) {
                            errorMessages.add("Line parsing returned error messages\n**LINE: " + encumbranceLine
                                              + "\n**MESSAGES: " + parsingMessages);
                            errorCount++;
                            LOG.error("{}", errorMessages::getLast);
                            errorStream.println(encumbranceLine);
                            encumbranceLineReadError = true;
                            advanceEncumbranceFile = true;
                        } else {
                            encumbranceEntryLine.setFinancialSystemOriginationCode(parameterService.getParameterValueAsString(
                                    LaborCalculateEncumbranceAdjustmentsStep.class,
                                    LaborParameterConstants.ORIGINATION_CODE
                            ));
                        }
                    } catch (final Exception ex) {
                        errorMessages.add(
                                "Unable to parse line - writing to error file.\n**LINE: " + encumbranceLine
                                + "\n**Exception: " + ex.getClass().getName() + " : " + ex.getMessage());
                        errorCount++;
                        LOG.error("{}", errorMessages::getLast);
                        errorStream.println(encumbranceLine);
                        encumbranceLineReadError = true;
                        advanceEncumbranceFile = true;
                    }
                } else {
                    if (encumbranceEntryLine != null) {
                        encumbranceEntryLine = null;
                        LOG.debug("Encumbrance file exhausted, blanking out entry line variable for rest of process");
                    }
                }
                if (!encumbranceLineReadError) {
                    if (encumbranceEntryLine != null) {
                        lastOriginEntry = encumbranceEntryLine;
                    }
                    final int compareResult;
                    compareResult = getCompareResult(lineComparator, balanceLine, encumbranceEntryLine);

                    if (compareResult == 0) {
                        //Compare amounts
                        processBalances(currentDate, lineComparator, outputStream, balanceLine, encumbranceEntryLine);

                        advanceBalanceFile = true;
                        advanceEncumbranceFile = true;
                    } else if (compareResult < 0) {
                        // Occurs when the encumbranceLine does not have a matching balance line
                        createMatchingBalanceLine(currentDate, outputStream, encumbranceEntryLine);

                        outputCount++;
                        newBalanceCount++;
                        advanceEncumbranceFile = true;
                    } else {
                        // Occurs when the balanceLine does not have a matching encumbranceLine
                        if (lastOriginEntry != null) {
                            createMatchingEncumbranceLine(currentDate,
                                    lastOriginEntry,
                                    lMap,
                                    outputStream,
                                    encumbranceLine,
                                    balanceLine
                            );

                            outputCount++;
                            removedBalanceCount++;
                            advanceBalanceFile = true;
                        } else {
                            LOG.info("The system had no valid lines, breaking out of the look.");
                            break;
                        }
                    }
                }
                if (advanceEncumbranceFile) {
                    inputLinesProcessed++;
                    encumbranceLine = inputReader.readLine();
                    lineNumber++;
                }
                if (advanceBalanceFile) {
                    balanceLinesProcessed++;
                    balanceLine = balanceReader.readLine();
                }
            }
            outputStream.flush();

            createDoneFile(outputFile);

            createReconFile(outputFile);
        } catch (final FileNotFoundException ex) {
            LOG.warn("Missing Input file: {}", ex::getMessage);
            return false;
        } catch (final IOException ex) {
            LOG.warn("Problem reading input files: {}", ex::getMessage);
            throw new RuntimeException("Error processing encumbrance balance file", ex);
        } finally {
            writeReport();
        }
        return true;
    }

    private void writeReport() {
        if (reportWriterService != null) {
            ((WrappingBatchService) reportWriterService).initialize();
            reportWriterService.writeStatisticLine("File Input Lines          %,20d", inputLinesProcessed);
            reportWriterService.writeStatisticLine("Balance File Lines        %,20d", balanceLinesProcessed);
            reportWriterService.writeStatisticLine("Output Lines              %,20d", outputCount);
            reportWriterService.writeStatisticLine("Input Line Errors         %,20d", errorCount);
            reportWriterService.writeNewLines(2);
            reportWriterService.writeStatisticLine("New Encumbrance Balances  %,20d", newBalanceCount);
            reportWriterService.writeStatisticLine("Increased Balances        %,20d", increasedBalanceCount);
            reportWriterService.writeStatisticLine("Decreased Balances        %,20d", decreasedBalanceCount);
            reportWriterService.writeStatisticLine("Removed Balances          %,20d", removedBalanceCount);
            reportWriterService.writeStatisticLine("Matching Balances         %,20d", matchingBalanceCount);
            if (!errorMessages.isEmpty()) {
                for (final String errorMessage : errorMessages) {
                    reportWriterService.writeErrorLine(errorMessage);
                }
            }
            ((WrappingBatchService) reportWriterService).destroy();
        }
    }

    private void createReconFile(final File outputFile) {
        try {
            final String reconFileName = StringUtils.removeEnd(outputFile.getAbsolutePath(),
                    GeneralLedgerConstants.BatchFileSystem.EXTENSION
            ) + GeneralLedgerConstants.BatchFileSystem.RECON_FILE_EXTENSION;

            final File reconFile = new File(reconFileName);
            reconFile.createNewFile();
            buildReconFile(reconFile, outputCount, transactionLedgerEntryAmountBalance);
        } catch (final Exception ex) {
            LOG.error("Unable to create recon file for labor encumbrance adjustment output file.", ex);
        }
    }

    private static void createDoneFile(final File outputFile) {
        try {
            final String doneFileName = StringUtils.removeEnd(outputFile.getAbsolutePath(),
                    GeneralLedgerConstants.BatchFileSystem.EXTENSION
            ) + GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION;
            final File doneFile = new File(doneFileName);
            doneFile.createNewFile();
        } catch (final Exception ex) {
            LOG.error("Unable to create done file for labor encumbrance adjustment output file.", ex);
        }
    }

    private void createMatchingBalanceLine(
            final Date currentDate, final PrintStream outputStream, final LaborOriginEntry encumbranceEntryLine
    ) {
        LOG.debug("Encr < Bal Keys: Create new encumbrance for encumbrance line");
        adjustEncumbranceOriginEntryLine(encumbranceEntryLine,
                encumbranceEntryLine.getTransactionLedgerEntryAmount(),
                KFSConstants.GL_DEBIT_CODE,
                currentDate
        );

        encumbranceEntryLine.setFinancialSystemOriginationCode(parameterService.getParameterValueAsString(LaborCalculateEncumbranceAdjustmentsStep.class,
                LaborParameterConstants.ORIGINATION_CODE
        ));

        outputStream.println(encumbranceEntryLine.getLine());
        transactionLedgerEntryAmountBalance =
                transactionLedgerEntryAmountBalance.add(encumbranceEntryLine.getTransactionLedgerEntryAmount());
    }

    private void createMatchingEncumbranceLine(
            final Date currentDate,
            final LaborOriginEntry lastOriginEntry,
            final Map<String, Integer> lMap,
            final PrintStream outputStream,
            final String encumbranceLine,
            final String balanceLine
    ) {
        LOG.info("Encr > Bal Keys: Reverse balance line");
        final LaborOriginEntry reversingOriginEntry =
                createOriginEntryToReverseBalanceEntry(balanceLine, lastOriginEntry, currentDate, lMap);

        // the description on the previous entry may be misleading, so we need to replace it.
        String newDescription = "Unknown Employee ID: " + reversingOriginEntry.getEmplid();
        try {
            final Person employeePerson = personService.getPersonByEmployeeId(reversingOriginEntry.getEmplid());
            if (employeePerson == null) {
                LOG.warn("No Person found for employee ID: {}", reversingOriginEntry::getEmplid);
            } else {
                newDescription = (employeePerson.getLastName() + ", " + employeePerson.getFirstName() + " "
                                  + employeePerson.getMiddleName()).trim();
            }
        } catch (final Exception ex) {
            LOG.warn("Error Obtaining Person for employee ID: {}\nException: {}",
                    reversingOriginEntry::getEmplid,
                    ex::getMessage
            );
        }
        reversingOriginEntry.setTransactionLedgerEntryDescription(newDescription);

        String newDocNum = StringUtils.left(reversingOriginEntry.getDocumentNumber(), 9);
        newDocNum += "00000";
        reversingOriginEntry.setDocumentNumber(newDocNum);

        if (encumbranceLine != null) {
            reversingOriginEntry.setFinancialSystemOriginationCode(parameterService.getParameterValueAsString(LaborCalculateEncumbranceAdjustmentsStep.class,
                    LaborParameterConstants.ORIGINATION_CODE
            ));
        }

        outputStream.println(reversingOriginEntry.getLine());
        transactionLedgerEntryAmountBalance =
                transactionLedgerEntryAmountBalance.add(reversingOriginEntry.getTransactionLedgerEntryAmount());
    }

    private void processBalances(
            final Date currentDate,
            final LaborOriginEntryBalanceRecordComparator lineComparator,
            final PrintStream outputStream,
            final String balanceLine,
            final LaborOriginEntry encumbranceEntryLine
    ) {
        LOG.debug("Key Match: Proceeding with amount comparison");

        final KualiDecimal newEncumbranceBalance = encumbranceEntryLine.getTransactionLedgerEntryAmount();
        final KualiDecimal existingEncumbranceBalance;
        final String amountString = balanceLine.substring(lineComparator.getBalanceAmountPOS(),
                lineComparator.getBalanceAmountPOS() + lineComparator.getBalanceAmountLength()
        ).trim();

        try {
            existingEncumbranceBalance = new KualiDecimal(amountString);
        } catch (final NumberFormatException ex) {
            LOG.fatal("Unable to parse amount from balance file.  Amount string was: " + amountString, ex);
            throw new RuntimeException(
                    "Unable to parse balance amount from a system generated " + "file.  There's a problem here!");
        }

        LOG.debug("Comparing Amounts: BalanceFile={} / InputFileAmount={}",
                existingEncumbranceBalance,
                newEncumbranceBalance
        );

        if (existingEncumbranceBalance.equals(newEncumbranceBalance)) {
            LOG.debug("Balances Match - no line created");
            matchingBalanceCount++;
        } else {
            final KualiDecimal difference;
            final String debitCreditCode;
            if (existingEncumbranceBalance.isGreaterThan(newEncumbranceBalance)) {
                // existing > new - reduce the balance
                LOG.debug("New Balance Less - Creating a credit");
                difference = existingEncumbranceBalance.subtract(newEncumbranceBalance);
                debitCreditCode = KFSConstants.GL_CREDIT_CODE;
                decreasedBalanceCount++;
            } else {
                // new > existing, increase the amount encumbered
                LOG.debug("New Balance More - Creating a debit to increase the amount");
                difference = newEncumbranceBalance.subtract(existingEncumbranceBalance);
                debitCreditCode = KFSConstants.GL_DEBIT_CODE;
                increasedBalanceCount++;
            }

            adjustEncumbranceOriginEntryLine(encumbranceEntryLine, difference, debitCreditCode, currentDate);

            final String financialObjectTypeCode = balanceLine.substring(lineComparator.getFinObjTypeCodePOS(),
                    lineComparator.getFinObjTypeCodePOS() + lineComparator.getFinObjTypeCodeLen()
            );
            encumbranceEntryLine.setFinancialObjectTypeCode(financialObjectTypeCode);

            encumbranceEntryLine.setFinancialSystemOriginationCode(parameterService.getParameterValueAsString(LaborCalculateEncumbranceAdjustmentsStep.class,
                    LaborParameterConstants.ORIGINATION_CODE
            ));

            outputStream.println(encumbranceEntryLine.getLine());
            transactionLedgerEntryAmountBalance =
                    transactionLedgerEntryAmountBalance.add(encumbranceEntryLine.getTransactionLedgerEntryAmount());

            LOG.debug("Writing line to output stream:\n {}", encumbranceEntryLine::getLine);
            outputCount++;
        }
    }

    private static int getCompareResult(
            final LaborOriginEntryBalanceRecordComparator lineComparator,
            final String balanceLine,
            final LaborOriginEntry encumbranceEntryLine
    ) {
        final int compareResult;
        if (encumbranceEntryLine == null) {
            compareResult = 1;
        } else if (balanceLine == null) {
            compareResult = -1;
        } else {
            compareResult = lineComparator.compare(encumbranceEntryLine, balanceLine);
        }
        return compareResult;
    }

    /**
     * Creates the .recon file.
     *
     * @param reconFile                           The name of the file to be created.
     * @param inputLinesProcessedCount            Amount of lines processed.
     * @param transactionLedgerEntryAmountBalance Total of all balances processed.
     */
    private void buildReconFile(
            final File reconFile,
            final int inputLinesProcessedCount,
            final KualiDecimal transactionLedgerEntryAmountBalance
    ) throws Exception {
        final PrintStream outputStream = new PrintStream(reconFile);
        outputStream.println("c ld_ldgr_entr_t " + inputLinesProcessedCount + ";");
        outputStream.println("s trn_ldgr_entr_amt " + transactionLedgerEntryAmountBalance + ";");
        outputStream.print("e 02;");

        if (outputStream != null) {
            outputStream.close();
        }
    }

    /**
     * Converts an origin entry line from the HCM system and sets or clears fields as needed.
     *
     * @param originEntry     The "template" origin entry to modify.
     * @param amount          Amount of the transaction.
     * @param debitCreditCode
     * @param currentDate     Transaction date of the entry.
     */
    private void adjustEncumbranceOriginEntryLine(
            final LaborOriginEntry originEntry, final KualiDecimal amount, final String debitCreditCode, final Date currentDate
    ) {
        originEntry.setTransactionLedgerEntryAmount(amount);
        originEntry.setTransactionDebitCreditCode(debitCreditCode);
        if (StringUtils.isBlank(originEntry.getFinancialDocumentTypeCode())) {
            originEntry.setFinancialDocumentTypeCode(LaborConstants.Encumbrance.LABOR_PERSONNEL_ENCUMBRANCE_DOC_TYPE);
        }
        originEntry.setTransactionPostingDate(currentDate);
        originEntry.setOrganizationDocumentNumber("");
        originEntry.setOrganizationReferenceId("");
        originEntry.setReferenceFinancialDocumentTypeCode("");
        originEntry.setReferenceFinancialDocumentNumber("");
        originEntry.setReferenceFinancialSystemOriginationCode("");
        originEntry.setReversalDate(null);
        originEntry.setPayPeriodEndDate(null);
        originEntry.setTransactionTotalHours(null);
        originEntry.setPayrollEndDateFiscalPeriodCode("");
        originEntry.setPayrollEndDateFiscalYear(null);
        originEntry.setEarnCode("");
        originEntry.setPayGroup("");
        originEntry.setSalaryAdministrationPlan("");
        originEntry.setGrade("");
        originEntry.setRunIdentifier("");
        originEntry.setLaborLedgerOriginalChartOfAccountsCode("");
        originEntry.setLaborLedgerOriginalAccountNumber("");
        originEntry.setLaborLedgerOriginalFinancialSubObjectCode("");
        originEntry.setLaborLedgerOriginalFinancialObjectCode("");
        originEntry.setLaborLedgerOriginalSubAccountNumber("");
    }

    /**
     * Takes a template origin entry (not modified) and merges with the key information in the given balance line.
     */
    private LaborOriginEntry createOriginEntryToReverseBalanceEntry(
            final String balanceLine, final LaborOriginEntry lastOriginEntry, final Date currentDate, final Map<String, Integer> fieldLengthMap
    ) {
        final LaborOriginEntry originEntry = new LaborOriginEntry(lastOriginEntry);

        LOG.debug("createOriginEntryToReverseBalanceEntry() Parsing balance line to make origin entry: {}",
                balanceLine);

        int startPos = 0;
        int fieldLen = fieldLengthMap.get(KFSPropertyConstants.EMPLID);
        originEntry.setEmplid(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.POSITION_NUMBER);
        originEntry.setPositionNumber(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        originEntry.setChartOfAccountsCode(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.ACCOUNT_NUMBER);
        originEntry.setAccountNumber(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        originEntry.setSubAccountNumber(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);
        originEntry.setFinancialObjectTypeCode(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        originEntry.setFinancialObjectCode(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        originEntry.setFinancialSubObjectCode(balanceLine.substring(startPos, startPos + fieldLen).trim());

        startPos += fieldLen;
        fieldLen = fieldLengthMap.get(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_AMOUNT) + 1;
        try {
            final KualiDecimal amount = new KualiDecimal(balanceLine.substring(startPos, startPos + fieldLen).trim());

            originEntry.setEmployeeRecord(null);
            originEntry.setProjectCode(KFSConstants.getDashProjectCode());
            originEntry.setTransactionLedgerEntrySequenceNumber(null);

            adjustEncumbranceOriginEntryLine(originEntry,
                    amount,
                    amount.isPositive() ? KFSConstants.GL_CREDIT_CODE : KFSConstants.GL_DEBIT_CODE,
                    currentDate
            );
        } catch (final NumberFormatException ex) {
            final String balanceLineSubString = balanceLine.substring(startPos, startPos + fieldLen);
            LOG.error(
                    "Unable to get amount from expense line.  Amount String: '{}'",
                    balanceLineSubString
            );
            throw new RuntimeException("Unable to get amount from expense line.  Amount String: '" + balanceLineSubString.trim() + "'",
                    ex
            );
        }
        return originEntry;
    }

    private void setupReportTrackingMetaData() {
        errorCount = 0;
        inputLinesProcessed = 0;
        balanceLinesProcessed = 0;
        newBalanceCount = 0;
        removedBalanceCount = 0;
        increasedBalanceCount = 0;
        decreasedBalanceCount = 0;
        matchingBalanceCount = 0;
        outputCount = 0;
        transactionLedgerEntryAmountBalance = new KualiDecimal(0.00);
        errorMessages = new LinkedList<>();
    }

    public void setLaborEncumbranceAdjustmentDao(final LaborEncumbranceAdjustmentDao laborEncumbranceAdjustmentDao) {
        this.laborEncumbranceAdjustmentDao = laborEncumbranceAdjustmentDao;
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setReportWriterService(final LaborEncumbranceReportWriterService reportWriterService) {
        this.reportWriterService = reportWriterService;
    }

    public void setPersonService(final PersonService personService) {
        this.personService = personService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

}
