/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.dataaccess.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.QueryByCriteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.kuali.kfs.gl.OJBUtility;
import org.kuali.kfs.gl.dataaccess.LedgerEntryBalancingDao;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.businessobject.LedgerEntry;
import org.kuali.kfs.module.ld.dataaccess.LaborLedgerEntryDao;
import org.kuali.kfs.module.ld.util.ConsolidationUtil;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.util.TransactionalServiceUtils;
import org.kuali.kfs.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This is the data access object for ledger entry.
 *
 * @see org.kuali.kfs.module.ld.businessobject.LedgerEntry
 */
public class LaborLedgerEntryDaoOjb extends PlatformAwareDaoBaseOjb implements LaborLedgerEntryDao,
        LedgerEntryBalancingDao {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public Integer getMaxSquenceNumber(final LedgerEntry ledgerEntry) {
        final Criteria criteria = new Criteria();

        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, ledgerEntry.getUniversityFiscalYear());
        criteria.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, ledgerEntry.getChartOfAccountsCode());
        criteria.addEqualTo(KFSPropertyConstants.ACCOUNT_NUMBER, ledgerEntry.getAccountNumber());
        criteria.addEqualTo(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, ledgerEntry.getSubAccountNumber());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, ledgerEntry.getFinancialObjectCode());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE, ledgerEntry.getFinancialSubObjectCode());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE,
                ledgerEntry.getFinancialBalanceTypeCode());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE,
                ledgerEntry.getFinancialObjectTypeCode());
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE,
                ledgerEntry.getUniversityFiscalPeriodCode());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_DOCUMENT_TYPE_CODE,
                ledgerEntry.getFinancialDocumentTypeCode());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_SYSTEM_ORIGINATION_CODE,
                ledgerEntry.getFinancialSystemOriginationCode());
        criteria.addEqualTo(KFSPropertyConstants.DOCUMENT_NUMBER, ledgerEntry.getDocumentNumber());

        final ReportQueryByCriteria query = QueryFactory.newReportQuery(getEntryClass(), criteria);
        query.setAttributes(new String[]{"max(" + KFSPropertyConstants.TRANSACTION_ENTRY_SEQUENCE_NUMBER + ")"});

        final Iterator iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
        int maxSequenceNumber = 0;

        if (iterator.hasNext()) {
            final Object[] data = (Object[]) TransactionalServiceUtils.retrieveFirstAndExhaustIterator(iterator);
            if (data[0] != null) {
                maxSequenceNumber = ((BigDecimal) data[0]).intValue();
            }
        }
        return maxSequenceNumber;
    }

    @Override
    public Iterator<LedgerEntry> find(final Map<String, String> fieldValues) {
        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LedgerEntry());
        final QueryByCriteria query = QueryFactory.newQuery(getEntryClass(), criteria);
        return getPersistenceBrokerTemplate().getIteratorByQuery(query);
    }

    @Override
    public List<String> findEmployeesWithPayType(
            final Map<Integer, Set<String>> payPeriods, final List<String> balanceTypes,
            final Map<String, Set<String>> earnCodePayGroupMap) {
        final Criteria criteria = buildPayTypeCriteria(payPeriods, balanceTypes, earnCodePayGroupMap);

        final ReportQueryByCriteria query = QueryFactory.newReportQuery(getEntryClass(), criteria);
        query.setAttributes(new String[]{KFSPropertyConstants.EMPLID});
        query.setDistinct(true);

        final Iterator<Object[]> employees = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
        final List<String> employeeList = new ArrayList<>();

        while (employees != null && employees.hasNext()) {
            final Object[] emplid = employees.next();
            employeeList.add(emplid == null ? "" : emplid[0].toString());
        }

        return employeeList;
    }

    @Override
    public Collection<LedgerEntry> getLedgerEntriesForEmployeeWithPayType(
            final String emplid, final Map<Integer,
            Set<String>> payPeriods, final List<String> balanceTypes, final Map<String, Set<String>> earnCodePayGroupMap) {
        final Criteria criteria = buildPayTypeCriteria(payPeriods, balanceTypes, earnCodePayGroupMap);
        criteria.addEqualTo(KFSPropertyConstants.EMPLID, emplid);

        final QueryByCriteria query = QueryFactory.newQuery(getEntryClass(), criteria);
        return getPersistenceBrokerTemplate().getCollectionByQuery(query);
    }

    @Override
    public boolean isEmployeeWithPayType(
            final String emplid, final Map<Integer, Set<String>> payPeriods,
            final List<String> balanceTypes, final Map<String, Set<String>> earnCodePayGroupMap) {
        final Criteria criteria = buildPayTypeCriteria(payPeriods, balanceTypes, earnCodePayGroupMap);
        criteria.addEqualTo(KFSPropertyConstants.EMPLID, emplid);

        final QueryByCriteria query = QueryFactory.newQuery(getEntryClass(), criteria);
        return getPersistenceBrokerTemplate().getCount(query) > 0;
    }

    @Override
    public void deleteLedgerEntriesPriorToYear(final Integer fiscalYear, final String chartOfAccountsCode) {
        LOG.debug("deleteLedgerEntriesPriorToYear() started");

        final Criteria criteria = new Criteria();
        criteria.addLessThan(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        criteria.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);

        final QueryByCriteria query = new QueryByCriteria(getEntryClass(), criteria);
        getPersistenceBrokerTemplate().deleteByQuery(query);

    }

    @Override
    public Object[] findEntryByGroup(
            final Integer universityFiscalYear, final String chartOfAccountsCode,
            final String financialObjectCode, final String financialBalanceTypeCode, final String universityFiscalPeriodCode,
            final String transactionDebitCreditCode) {
        final Criteria criteria = new Criteria();
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, universityFiscalYear);
        criteria.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, financialObjectCode);
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, financialBalanceTypeCode);
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE, universityFiscalPeriodCode);
        criteria.addEqualTo(KFSPropertyConstants.TRANSACTION_DEBIT_CREDIT_CODE, transactionDebitCreditCode);

        final ReportQueryByCriteria reportQuery = QueryFactory.newReportQuery(getEntryClass(), criteria);
        reportQuery.setAttributes(new String[]{"count(*)", ConsolidationUtil.sum(
                KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_AMOUNT)});
        reportQuery.addGroupBy(new String[]{
            KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR,
            KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE,
            KFSPropertyConstants.FINANCIAL_OBJECT_CODE,
            KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE,
            KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE,
            KFSPropertyConstants.TRANSACTION_DEBIT_CREDIT_CODE
        });

        final Iterator<Object[]> iterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(reportQuery);
        final Object[] returnResult = TransactionalServiceUtils.retrieveFirstAndExhaustIterator(iterator);

        if (ObjectUtils.isNotNull(returnResult)) {
            if (returnResult[0] instanceof BigDecimal) {
                returnResult[0] = ((BigDecimal) returnResult[0]).intValue();
            } else {
                returnResult[0] = ((Long) returnResult[0]).intValue();
            }
        }

        return returnResult;
    }

    // build the pay type criteria
    protected Criteria buildPayTypeCriteria(
            final Map<Integer, Set<String>> payPeriods, final List<String> balanceTypes,
            final Map<String, Set<String>> earnCodePayGroupMap) {
        final Criteria criteria = new Criteria();

        final Criteria criteriaForPayPeriods = new Criteria();
        for (final Integer fiscalYear : payPeriods.keySet()) {
            final Criteria criteriaForFiscalYear = new Criteria();

            criteriaForFiscalYear.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
            criteriaForFiscalYear.addIn(KFSPropertyConstants.UNIVERSITY_FISCAL_PERIOD_CODE, payPeriods.get(fiscalYear));

            criteriaForPayPeriods.addOrCriteria(criteriaForFiscalYear);
        }

        final Criteria criteriaForBalanceTypes = new Criteria();
        criteriaForBalanceTypes.addIn(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, balanceTypes);

        final Criteria criteriaForEarnCodePayGroup = new Criteria();
        for (final String payGroup : earnCodePayGroupMap.keySet()) {
            final Criteria criteriaForEarnPay = new Criteria();

            criteriaForEarnPay.addEqualTo(KFSPropertyConstants.PAY_GROUP, payGroup);
            criteriaForEarnPay.addIn(KFSPropertyConstants.EARN_CODE, earnCodePayGroupMap.get(payGroup));

            criteriaForEarnCodePayGroup.addOrCriteria(criteriaForEarnPay);
        }

        criteria.addAndCriteria(criteriaForPayPeriods);
        criteria.addAndCriteria(criteriaForBalanceTypes);
        criteria.addAndCriteria(criteriaForEarnCodePayGroup);

        return criteria;
    }

    @Override
    public Integer findCountGreaterOrEqualThan(final Integer year) {
        final Criteria criteria = new Criteria();
        criteria.addGreaterOrEqualThan(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, year);

        final ReportQueryByCriteria query = QueryFactory.newReportQuery(getEntryClass(), criteria);
        return getPersistenceBrokerTemplate().getCount(query);
    }

    /**
     * @return the Class type of the business object accessed and managed
     */
    protected Class getEntryClass() {
        return LedgerEntry.class;
    }
}
