/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.document.GeneralLedgerCorrectionProcessDocument;
import org.kuali.kfs.gl.service.OriginEntryGroupService;
import org.kuali.kfs.kns.bo.Step;
import org.kuali.kfs.module.ld.batch.LaborCorrectionProcessScrubberStep;
import org.kuali.kfs.module.ld.document.service.LaborCorrectionDocumentService;
import org.kuali.kfs.module.ld.service.LaborOriginEntryGroupService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.batch.BatchSpringContext;
import org.kuali.kfs.sys.context.ProxyUtils;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;

import java.util.Date;

/**
 * labor Document class for the Labor Ledger Correction Process.
 */
public class LedgerCorrectionDocument extends GeneralLedgerCorrectionProcessDocument {

    private static final Logger LOG = LogManager.getLogger();

    public LedgerCorrectionDocument() {
        super();
    }

    /**
     * @param statusChangeEvent
     */
    @Override
    public void doRouteStatusChange(final DocumentRouteStatusChange statusChangeEvent) {
        // NOTE: DO NOT call the super implementation.  It has the GLCP processing logic which should not be run here.
        // The code below is copied from the super.super implementation:
        if (getDocumentHeader().getWorkflowDocument().isCanceled()) {
            getDocumentHeader().setFinancialDocumentStatusCode(
                    KFSConstants.DocumentStatusCodes.CANCELLED);
        } else if (getDocumentHeader().getWorkflowDocument().isEnroute()) {
            getDocumentHeader().setFinancialDocumentStatusCode(
                    KFSConstants.DocumentStatusCodes.ENROUTE);
        }
        if (getDocumentHeader().getWorkflowDocument().isDisapproved()) {
            getDocumentHeader().setFinancialDocumentStatusCode(
                    KFSConstants.DocumentStatusCodes.DISAPPROVED);
        }
        if (getDocumentHeader().getWorkflowDocument().isProcessed()) {
            getDocumentHeader().setFinancialDocumentStatusCode(
                    KFSConstants.DocumentStatusCodes.APPROVED);
        }
        LOG.info(
                "Document: {} -- Status is: {}",
                statusChangeEvent::getDocumentId,
                () -> getDocumentHeader().getFinancialDocumentStatusCode()
        );
        // End of super.super code
        if (getDocumentHeader().getWorkflowDocument().isProcessed()) {
            final String docId = getDocumentNumber();
            LOG.info("Document {} moving to Processed Status - starting final processing", docId);
            // this code is performed asynchronously
            // First, save the origin entries to the origin entry table
            final LaborCorrectionDocumentService laborCorrectionDocumentService =
                    SpringContext.getBean(LaborCorrectionDocumentService.class);
            final OriginEntryGroupService originEntryGroupService = SpringContext.getBean(LaborOriginEntryGroupService.class);

            // QUESTION - since this *is* the labor correction document - why are we loading it again?
            final LedgerCorrectionDocument doc = laborCorrectionDocumentService.findByCorrectionDocumentHeaderId(docId);

            final String correctionType = doc.getCorrectionTypeCode();
            if (LaborCorrectionDocumentService.CORRECTION_TYPE_REMOVE_GROUP_FROM_PROCESSING.equals(correctionType)) {

                final String dataFileName = doc.getCorrectionInputFileName();
                final String doneFileName = dataFileName.replace(GeneralLedgerConstants.BatchFileSystem.EXTENSION,
                        GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION);
                originEntryGroupService.deleteFile(doneFileName);
                LOG.info("Document {} : deleted done file to remove from processing: {}", docId, doneFileName);

            } else if (LaborCorrectionDocumentService.CORRECTION_TYPE_MANUAL.equals(correctionType)
                    || LaborCorrectionDocumentService.CORRECTION_TYPE_CRITERIA.equals(correctionType)) {
                synchronized (LaborCorrectionDocumentService.class) {
                    if (!checkForExistingOutputDocument(docId)) {

                        final Date today = SpringContext.getBean(DateTimeService.class).getCurrentDate();

                        String outputFileName = "";
                        if (!doc.getCorrectionFileDelete()) {
                            outputFileName = laborCorrectionDocumentService.createOutputFileForProcessing(docId,
                                    today);
                        } else {
                            LOG.info("Document {} : set to delete output file - no file will be created", docId);
                        }
                        doc.setCorrectionOutputFileName(outputFileName);

                        LOG.info("Document {} : about to run scrubber -- output file: {}", docId, outputFileName);
                        final Step step = BatchSpringContext.getStep(LaborCorrectionProcessScrubberStep.STEP_NAME);
                        final LaborCorrectionProcessScrubberStep correctionStep =
                                (LaborCorrectionProcessScrubberStep) ProxyUtils.getTargetIfProxied(step);
                        correctionStep.setDocumentId(docId);
                        try {
                            step.execute(getClass().getName(), today);
                        } catch (final Exception e) {
                            LOG.error("LLCP scrubber encountered error:", e);
                            throw new RuntimeException("LLCP scrubber encountered error:", e);
                        }
                        correctionStep.setDocumentId(null);
                        LOG.info("Document {} : completed scrubber run -- generating reports", docId);

                        laborCorrectionDocumentService.generateCorrectionReport(this);
                        laborCorrectionDocumentService.aggregateCorrectionDocumentReports(this);
                    } else {
                        LOG.warn(
                                "Attempt to re-process final LLCP operations for document: {}  File with that "
                                + "document number already exists.",
                                docId
                        );
                    }
                }
            } else {
                LOG.error("LLCP doc {} has an unknown correction type code: {}", docId, correctionType);
            }
            LOG.info("Document {} moving to Processed Status - completed final processing", docId);
        }
    }

}
