/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.document.LaborExpenseTransferDocumentBase;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.document.AccountingDocument;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;

import java.util.HashSet;
import java.util.Set;

/**
 * benefit transfers cannot be made between two different fringe benefit labor object codes
 */
public class BenefitExpenseTransferSameFringeBenefitObjectCodeValidation extends GenericValidation {

    private Document documentForValidation;

    @Override
    public boolean validate(final AttributedDocumentEvent event) {
        boolean result = true;

        final Document documentForValidation = getDocumentForValidation();

        final AccountingDocument accountingDocument = (AccountingDocument) documentForValidation;

        // benefit transfers cannot be made between two different fringe benefit labor object codes.
        final boolean sameFringeBenefitObjectCodes = hasSameFringeBenefitObjectCodes(accountingDocument);
        if (!sameFringeBenefitObjectCodes) {
            GlobalVariables.getMessageMap().putError(KFSPropertyConstants.TARGET_ACCOUNTING_LINES,
                    LaborKeyConstants.DISTINCT_OBJECT_CODE_ERROR);
            result = false;
        }

        return result;
    }

    /**
     * Determines whether target accounting lines have the same fringe benefit object codes as source accounting lines
     *
     * @param accountingDocument the given accounting document
     * @return true if target accounting lines have the same fringe benefit object codes as source accounting lines;
     *         otherwise, false
     */
    protected boolean hasSameFringeBenefitObjectCodes(final AccountingDocument accountingDocument) {
        final LaborExpenseTransferDocumentBase expenseTransferDocument =
                (LaborExpenseTransferDocumentBase) accountingDocument;

        final Set<String> objectCodesFromSourceLine = new HashSet<>();
        for (final Object sourceAccountingLine : expenseTransferDocument.getSourceAccountingLines()) {
            final AccountingLine line = (AccountingLine) sourceAccountingLine;
            objectCodesFromSourceLine.add(line.getFinancialObjectCode());
        }

        final Set<String> objectCodesFromTargetLine = new HashSet<>();
        for (final Object targetAccountingLine : expenseTransferDocument.getTargetAccountingLines()) {
            final AccountingLine line = (AccountingLine) targetAccountingLine;
            objectCodesFromTargetLine.add(line.getFinancialObjectCode());
        }

        if (objectCodesFromSourceLine.size() != objectCodesFromTargetLine.size()) {
            return false;
        }

        return objectCodesFromSourceLine.containsAll(objectCodesFromTargetLine);
    }

    public Document getDocumentForValidation() {
        return documentForValidation;
    }

    public void setDocumentForValidation(final Document documentForValidation) {
        this.documentForValidation = documentForValidation;
    }
}
