/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferAccountingLine;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferSourceAccountingLine;
import org.kuali.kfs.module.ld.businessobject.LedgerBalance;
import org.kuali.kfs.module.ld.document.LaborExpenseTransferDocumentBase;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class LaborExpenseTransferNegtiveAmountBeTransferredValidation extends GenericValidation {

    private static final Logger LOG = LogManager.getLogger();

    private Document documentForValidation;
    private BusinessObjectService businessObjectService;
    private OptionsService optionsService;

    /**
     * Determines whether a negative amount can be transferred from one account to another
     *
     * @return true if a negative amount can be transferred from one account to another; false otherwise.
     */
    @Override
    public boolean validate(final AttributedDocumentEvent event) {
        boolean result = true;

        final Document documentForValidation = getDocumentForValidation();

        final LaborExpenseTransferDocumentBase expenseTransferDocument =
                (LaborExpenseTransferDocumentBase) documentForValidation;

        final List sourceLines = expenseTransferDocument.getSourceAccountingLines();

        // allow a negative amount to be moved from one account to another but do not allow a negative amount to be
        // created when the balance is positive
        final Map<String, ExpenseTransferAccountingLine> accountingLineGroupMap = getAccountingLineGroupMap(sourceLines,
                ExpenseTransferSourceAccountingLine.class);
        final boolean canNegtiveAmountBeTransferred = canNegtiveAmountBeTransferred(accountingLineGroupMap);
        if (!canNegtiveAmountBeTransferred) {
            GlobalVariables.getMessageMap().putError(KFSPropertyConstants.SOURCE_ACCOUNTING_LINES,
                    LaborKeyConstants.ERROR_CANNOT_TRANSFER_NEGATIVE_AMOUNT);
            result = false;
        }

        return result;
    }

    /**
     * Determines whether a negative amount can be transferred from one account to another
     *
     * @param accountingLineGroupMap the given accountingLineGroupMap
     * @return true if a negative amount can be transferred from one account to another; otherwise, false
     */
    protected boolean canNegtiveAmountBeTransferred(final Map<String, ExpenseTransferAccountingLine> accountingLineGroupMap) {
        for (final String key : accountingLineGroupMap.keySet()) {
            final ExpenseTransferAccountingLine accountingLine = accountingLineGroupMap.get(key);
            final Map<String, Object> fieldValues = buildFieldValueMap(accountingLine);

            final KualiDecimal balanceAmount = getBalanceAmount(fieldValues,
                    accountingLine.getPayrollEndDateFiscalPeriodCode());
            final KualiDecimal transferAmount = accountingLine.getAmount();

            // a negative amount cannot be transferred if the balance amount is positive
            if (transferAmount.isNegative() && balanceAmount.isPositive()) {
                return false;
            }
        }
        return true;
    }

    /**
     * build the field-value maps through the given accounting line
     *
     * @param accountingLine the given accounting line
     * @return the field-value maps built from the given accounting line
     */
    protected Map<String, Object> buildFieldValueMap(final ExpenseTransferAccountingLine accountingLine) {
        final Map<String, Object> fieldValues = new HashMap<>();

        fieldValues.put(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, accountingLine.getPostingYear());
        fieldValues.put(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, accountingLine.getChartOfAccountsCode());
        fieldValues.put(KFSPropertyConstants.ACCOUNT_NUMBER, accountingLine.getAccountNumber());

        String subAccountNumber = accountingLine.getSubAccountNumber();
        subAccountNumber = StringUtils.isBlank(subAccountNumber) ? KFSConstants.getDashSubAccountNumber() :
                subAccountNumber;
        fieldValues.put(KFSPropertyConstants.SUB_ACCOUNT_NUMBER, subAccountNumber);

        fieldValues.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, accountingLine.getBalanceTypeCode());
        fieldValues.put(KFSPropertyConstants.FINANCIAL_OBJECT_CODE, accountingLine.getFinancialObjectCode());

        final SystemOptions options = optionsService.getOptions(accountingLine.getPostingYear());
        fieldValues.put(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE, options.getFinObjTypeExpenditureexpCd());

        String subObjectCode = accountingLine.getFinancialSubObjectCode();
        subObjectCode = StringUtils.isBlank(subObjectCode) ? KFSConstants.getDashFinancialSubObjectCode() :
                subObjectCode;
        fieldValues.put(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE, subObjectCode);

        fieldValues.put(KFSPropertyConstants.EMPLID, accountingLine.getEmplid());
        fieldValues.put(KFSPropertyConstants.POSITION_NUMBER, accountingLine.getPositionNumber());

        return fieldValues;
    }

    /**
     * Groups the accounting lines by the specified key fields
     *
     * @param accountingLines the given accounting lines that are stored in a list
     * @param clazz           the class type of given accounting lines
     * @return the accounting line groups
     */
    protected Map<String, ExpenseTransferAccountingLine> getAccountingLineGroupMap(
            final List<ExpenseTransferAccountingLine> accountingLines, final Class clazz) {
        final Map<String, ExpenseTransferAccountingLine> accountingLineGroupMap = new HashMap<>();

        for (final ExpenseTransferAccountingLine accountingLine : accountingLines) {
            final String stringKey = ObjectUtil.buildPropertyMap(accountingLine,
                    defaultKeyOfExpenseTransferAccountingLine()).toString();
            final ExpenseTransferAccountingLine line;

            if (accountingLineGroupMap.containsKey(stringKey)) {
                line = accountingLineGroupMap.get(stringKey);
                final KualiDecimal amount = line.getAmount();
                line.setAmount(amount.add(accountingLine.getAmount()));
            } else {
                try {
                    line = (ExpenseTransferAccountingLine) clazz.newInstance();
                    ObjectUtil.buildObject(line, accountingLine);
                    accountingLineGroupMap.put(stringKey, line);
                } catch (final Exception e) {
                    LOG.error("Cannot create a new instance of ExpenseTransferAccountingLine{}", e);
                }
            }
        }
        return accountingLineGroupMap;
    }

    protected List<String> defaultKeyOfExpenseTransferAccountingLine() {
        final List<String> defaultKey = new ArrayList<>();

        defaultKey.add(KFSPropertyConstants.POSTING_YEAR);
        defaultKey.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        defaultKey.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        defaultKey.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);

        defaultKey.add(KFSPropertyConstants.BALANCE_TYPE_CODE);
        defaultKey.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        defaultKey.add(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);

        defaultKey.add(KFSPropertyConstants.EMPLID);
        defaultKey.add(KFSPropertyConstants.POSITION_NUMBER);

        defaultKey.add(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_YEAR);
        defaultKey.add(LaborPropertyConstants.PAYROLL_END_DATE_FISCAL_PERIOD_CODE);

        return defaultKey;
    }

    /**
     * get the amount for a given period from a ledger balance that has the given values for specified fields
     *
     * @param fieldValues the given fields and their values
     * @param periodCode  the given period
     * @return the amount for a given period from the qualified ledger balance
     */
    protected KualiDecimal getBalanceAmount(final Map<String, Object> fieldValues, final String periodCode) {
        if (periodCode == null) {
            return KualiDecimal.ZERO;
        }

        fieldValues.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, KFSConstants.BALANCE_TYPE_ACTUAL);
        final KualiDecimal actualBalanceAmount = getBalanceAmountOfGivenPeriod(fieldValues, periodCode);

        fieldValues.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, KFSConstants.BALANCE_TYPE_A21);
        final KualiDecimal effortBalanceAmount = getBalanceAmountOfGivenPeriod(fieldValues, periodCode);

        return actualBalanceAmount.add(effortBalanceAmount);
    }

    /**
     * Gets the balance amount of a given period
     *
     * @param fieldValues
     * @param periodCode
     * @return
     */
    protected KualiDecimal getBalanceAmountOfGivenPeriod(final Map<String, Object> fieldValues, final String periodCode) {
        KualiDecimal balanceAmount = KualiDecimal.ZERO;
        final List<LedgerBalance> ledgerBalances = (List<LedgerBalance>) businessObjectService
                .findMatching(LedgerBalance.class, fieldValues);
        if (!ledgerBalances.isEmpty()) {
            balanceAmount = ledgerBalances.get(0).getAmount(periodCode);
        }
        return balanceAmount;
    }

    public Document getDocumentForValidation() {
        return documentForValidation;
    }

    public void setDocumentForValidation(final Document documentForValidation) {
        this.documentForValidation = documentForValidation;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }
}
