/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.service.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.krad.service.LookupService;
import org.kuali.kfs.module.ld.businessobject.AccountStatusBaseFunds;
import org.kuali.kfs.module.ld.businessobject.EmployeeFunding;
import org.kuali.kfs.module.ld.businessobject.July1PositionFunding;
import org.kuali.kfs.module.ld.businessobject.LaborCalculatedSalaryFoundationTracker;
import org.kuali.kfs.module.ld.dataaccess.LaborCalculatedSalaryFoundationTrackerDao;
import org.kuali.kfs.module.ld.service.LaborCalculatedSalaryFoundationTrackerService;
import org.kuali.kfs.sys.ObjectUtil;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * This class provides its clients with access to CSF tracker entries in the backend data store.
 *
 * @see org.kuali.kfs.module.ld.businessobject.LaborCalculatedSalaryFoundationTracker
 */
@Transactional
public class LaborCalculatedSalaryFoundationTrackerServiceImpl implements
        LaborCalculatedSalaryFoundationTrackerService {

    private static final Logger LOG = LogManager.getLogger();

    private LaborCalculatedSalaryFoundationTrackerDao laborCalculatedSalaryFoundationTrackerDao;
    private LookupService lookupService;

    @Override
    public List<LaborCalculatedSalaryFoundationTracker> findCSFTracker(final Map fieldValues, final boolean isConsolidated) {
        LOG.debug("start findCSFTracker()");
        return laborCalculatedSalaryFoundationTrackerDao.findCSFTrackers(fieldValues, isConsolidated);
    }

    /**
     * Check if there is a CSF track in the given set that matches the given object
     *
     * @param CSFTrackerCollection the given set of CSF trackers
     * @param anotherObject        the object to be searched
     * @return the CSF tracker if there is a CSF track in the given set that matches the given object
     */
    protected LaborCalculatedSalaryFoundationTracker findCSFTracker(
            final List<LaborCalculatedSalaryFoundationTracker> CSFTrackerCollection, final Object anotherObject) {
        for (final LaborCalculatedSalaryFoundationTracker CSFTracker : CSFTrackerCollection) {
            final boolean found = ObjectUtil.equals(CSFTracker, anotherObject, CSFTracker.getKeyFieldList());
            if (found) {
                return CSFTracker;
            }
        }
        return null;
    }

    @Override
    public List<LaborCalculatedSalaryFoundationTracker> findCSFTrackerWithJuly1(
            final Map fieldValues,
            final boolean isConsolidated) {
        LOG.debug("start findCSFTrackerWithJuly1()");

        final List<LaborCalculatedSalaryFoundationTracker> CSFTrackerCollection = findCSFTracker(fieldValues,
                isConsolidated);
        final Collection<July1PositionFunding> july1PositionFundings = lookupService.findCollectionBySearch(
                July1PositionFunding.class, fieldValues);
        for (final July1PositionFunding july1PositionFunding : july1PositionFundings) {
            LaborCalculatedSalaryFoundationTracker CSFTracker = findCSFTracker(CSFTrackerCollection,
                    july1PositionFunding);

            if (CSFTracker != null) {
                CSFTracker.setJuly1BudgetAmount(CSFTracker.getJuly1BudgetAmount().add(
                        july1PositionFunding.getJuly1BudgetAmount()));
                CSFTracker.setJuly1BudgetFteQuantity(CSFTracker.getJuly1BudgetFteQuantity().add(
                        july1PositionFunding.getJuly1BudgetFteQuantity()));
                CSFTracker.setJuly1BudgetTimePercent(CSFTracker.getJuly1BudgetTimePercent().add(
                        july1PositionFunding.getJuly1BudgetTimePercent()));
            } else {
                CSFTracker = new LaborCalculatedSalaryFoundationTracker();
                ObjectUtil.buildObject(CSFTracker, july1PositionFunding);
                CSFTrackerCollection.add(CSFTracker);
            }
        }
        return CSFTrackerCollection;
    }

    @Override
    public List<AccountStatusBaseFunds> findCSFTrackersAsAccountStatusBaseFunds(
            final Map fieldValues,
            final boolean isConsolidated) {
        LOG.debug("start findCSFTrackersAsAccountStatusBaseFunds()");
        return laborCalculatedSalaryFoundationTrackerDao.findCSFTrackersAsAccountStatusBaseFunds(fieldValues,
                isConsolidated);
    }

    @Override
    public List<EmployeeFunding> findCSFTrackersAsEmployeeFunding(final Map fieldValues, final boolean isConsolidated) {
        LOG.debug("start findCSFTrackersAsEmployeeFunding()");
        return laborCalculatedSalaryFoundationTrackerDao.findCSFTrackersAsEmployeeFunding(fieldValues, isConsolidated);
    }

    public void setLaborCalculatedSalaryFoundationTrackerDao(
            final LaborCalculatedSalaryFoundationTrackerDao laborCalculatedSalaryFoundationTrackerDao) {
        this.laborCalculatedSalaryFoundationTrackerDao = laborCalculatedSalaryFoundationTrackerDao;
    }

    public void setLookupService(final LookupService lookupService) {
        this.lookupService = lookupService;
    }
}
