/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntryFieldUtil;
import org.kuali.kfs.sys.KFSPropertyConstants;

import java.util.Map;

/**
 * Special comparator for labor origin entry records and the encumbrance balance file. It compares the keys in the
 * balance file with the appropriate attributes of the origin entry.
 */
public class LaborOriginEntryBalanceRecordComparator {
    private static final Logger LOG = LogManager.getLogger();
    private static final char SINGLE_BLANK_SPACE = ' ';
    private final LaborOriginEntryFieldUtil laborOriginEntryFieldUtil = new LaborOriginEntryFieldUtil();
    private final Map<String, Integer> lengthMap = laborOriginEntryFieldUtil.getFieldLengthMap();

    // Decode all of these up front to avoid the decoding of the information on *every* comparison, which could
    // happen quite a number of times
    private final int EMPLID_LEN = lengthMap.get(KFSPropertyConstants.EMPLID);
    private final int POSITION_NUMBER_LEN = lengthMap.get(KFSPropertyConstants.POSITION_NUMBER);
    private final int CHART_OF_ACCOUNTS_CODE_LEN = lengthMap.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
    private final int ACCOUNT_NUMBER_LEN = lengthMap.get(KFSPropertyConstants.ACCOUNT_NUMBER);
    private final int SUB_ACCOUNT_NUMBER_LEN = lengthMap.get(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
    private final int FINANCIAL_OBJECT_TYPE_CODE_LEN = lengthMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);
    private final int FINANCIAL_OBJECT_CODE_LEN = lengthMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
    private final int FINANCIAL_SUB_OBJECT_CODE_LEN = lengthMap.get(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
    private final int TOTAL_KEY_LEN = EMPLID_LEN + POSITION_NUMBER_LEN + CHART_OF_ACCOUNTS_CODE_LEN + ACCOUNT_NUMBER_LEN
                               + SUB_ACCOUNT_NUMBER_LEN + FINANCIAL_OBJECT_TYPE_CODE_LEN + FINANCIAL_OBJECT_CODE_LEN
                               + FINANCIAL_SUB_OBJECT_CODE_LEN;
    private final int BALANCE_AMOUNT_POS = TOTAL_KEY_LEN;
    private final int BALANCE_AMOUNT_LEN = lengthMap.get(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_AMOUNT) + 1;
    private final int FINANCIAL_OBJECT_TYPE_CODE_POS =
            EMPLID_LEN + POSITION_NUMBER_LEN + CHART_OF_ACCOUNTS_CODE_LEN + ACCOUNT_NUMBER_LEN
            + SUB_ACCOUNT_NUMBER_LEN;
    private final int FINANCIAL_OBJECT_CODE_POS = FINANCIAL_OBJECT_TYPE_CODE_POS + FINANCIAL_OBJECT_TYPE_CODE_LEN;
    private final StringBuilder sb = new StringBuilder(TOTAL_KEY_LEN);

    public int compare(final LaborOriginEntry originEntry, final String balanceRecord) {
        sb.setLength(0);

        sb.append(StringUtils.rightPad(originEntry.getEmplid(), EMPLID_LEN, SINGLE_BLANK_SPACE));
        sb.append(StringUtils.rightPad(originEntry.getPositionNumber(), POSITION_NUMBER_LEN, SINGLE_BLANK_SPACE));
        sb.append(StringUtils.rightPad(originEntry.getChartOfAccountsCode(), CHART_OF_ACCOUNTS_CODE_LEN, SINGLE_BLANK_SPACE));
        sb.append(StringUtils.rightPad(originEntry.getAccountNumber(), ACCOUNT_NUMBER_LEN, SINGLE_BLANK_SPACE));
        sb.append(StringUtils.rightPad(originEntry.getSubAccountNumber(), SUB_ACCOUNT_NUMBER_LEN, SINGLE_BLANK_SPACE));
        sb.append(StringUtils.rightPad(originEntry.getFinancialObjectTypeCode(),
                FINANCIAL_OBJECT_TYPE_CODE_LEN,
                SINGLE_BLANK_SPACE
        ));
        sb.append(StringUtils.rightPad(originEntry.getFinancialObjectCode(), FINANCIAL_OBJECT_CODE_LEN, SINGLE_BLANK_SPACE));
        sb.append(StringUtils.rightPad(originEntry.getFinancialSubObjectCode(),
                FINANCIAL_SUB_OBJECT_CODE_LEN,
                SINGLE_BLANK_SPACE
        ));
        final String originEntryKeyString = sb.toString();

        final String balanceRecordKeyString = StringUtils.substring(balanceRecord, 0, TOTAL_KEY_LEN);
        LOG.debug("Comparing Strings: \nENC: {}\nBAL: {}", originEntryKeyString,
                    balanceRecordKeyString);

        String originEntrySubKey = originEntryKeyString.substring(0, FINANCIAL_OBJECT_TYPE_CODE_POS);
        String balanceRecordSubKey = balanceRecordKeyString.substring(0, FINANCIAL_OBJECT_TYPE_CODE_POS);

        final int compareResult = originEntrySubKey.compareTo(balanceRecordSubKey);
        if (compareResult != 0) {
            return compareResult;
        }

        originEntrySubKey = originEntryKeyString.substring(FINANCIAL_OBJECT_CODE_POS);
        balanceRecordSubKey = balanceRecordKeyString.substring(FINANCIAL_OBJECT_CODE_POS);

        return originEntrySubKey.compareTo(balanceRecordSubKey);
    }

    public int getBalanceAmountPOS() {
        return BALANCE_AMOUNT_POS;
    }

    public int getBalanceAmountLength() {
        return BALANCE_AMOUNT_LEN;
    }

    public int getFinObjTypeCodePOS() {
        return FINANCIAL_OBJECT_TYPE_CODE_POS;
    }

    public int getFinObjTypeCodeLen() {
        return FINANCIAL_OBJECT_TYPE_CODE_LEN;
    }
}
