/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.dataaccess.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.framework.persistence.jdbc.dao.PlatformAwareDaoBaseJdbc;
import org.kuali.kfs.module.ld.batch.dataaccess.LaborEncumbranceAdjustmentDao;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntryFieldUtil;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.service.OptionsService;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Map;

public class LaborEncumbranceAdjustmentDaoJdbc extends PlatformAwareDaoBaseJdbc implements
        LaborEncumbranceAdjustmentDao {

    private static final Logger LOG = LogManager.getLogger();

    private static final String LABOR_OBJECT_SALARY_CODE = "S";
    private static final char PAD_CHAR = ' ';
    private OptionsService optionsService;
    private final String additionalWhereClause = "";

    private static final String ENCUMBRANCE_BALANCE_SQL =
            "select l.EMPLID, l.position_nbr, l.fin_coa_cd, l.account_nbr, l.sub_acct_nbr, l.fin_obj_typ_cd,"
            + " l.FIN_OBJECT_CD, l.FIN_SUB_OBJ_CD, SUM(l.ACLN_ANNL_BAL_AMT) AS encumbrance_balance\n"
            + "    from LD_LDGR_BAL_T l, ld_labor_obj_t lo\n"
            + "    WHERE l.UNIV_FISCAL_YR = ?\n"
            + "      AND l.FIN_BALANCE_TYP_CD = ?\n"
            + "      AND lo.univ_fiscal_yr = l.UNIV_FISCAL_YR\n"
            + "      AND lo.FIN_COA_CD = l.FIN_COA_CD\n"
            + "      AND lo.FIN_OBJECT_CD = l.FIN_OBJECT_CD\n"
            + "      AND lo.FINOBJ_FRNGSLRY_CD = '" + LABOR_OBJECT_SALARY_CODE + "'";

    private static final String ENCUMBRANCE_BALANCE_SQL_2 =
            "    GROUP BY l.EMPLID, l.position_nbr, l.fin_coa_cd, l.account_nbr, l.sub_acct_nbr, l.fin_obj_typ_cd, "
            + "l.FIN_OBJECT_CD, l.FIN_SUB_OBJ_CD\n"
            + "    HAVING SUM(l.ACLN_ANNL_BAL_AMT) != 0\n"
            + "    ORDER BY EMPLID, position_nbr, fin_coa_cd, account_nbr, sub_acct_nbr,"
            + "             FIN_OBJECT_CD, FIN_SUB_OBJ_CD, fin_obj_typ_cd";

    @Override
    public int buildFileForEncumbranceBalances(final Integer fiscalYear, final File fileName) {
        LOG.debug("buildFileForEncumbranceBalances() started");

        int outputLines = 0;
        final Map<String, Integer> lMap = new LaborOriginEntryFieldUtil().getFieldLengthMap();

        // this is a temp file, so if one exists, remove it
        if (fileName.exists()) {
            fileName.delete();
        }

        final String internalEncumbranceBalanceTypeCode =
                optionsService.getOptions(fiscalYear).getIntrnlEncumFinBalanceTypCd();

        LOG.info("Opening {} for output of balance records.", fileName::getAbsolutePath);
        try (Connection con = getConnection();
             PrintStream outputStream = new PrintStream(fileName);
             PreparedStatement stmt = con.prepareStatement(
                ENCUMBRANCE_BALANCE_SQL
                + additionalWhereClause
                + ENCUMBRANCE_BALANCE_SQL_2)) {
            stmt.setInt(1, fiscalYear);
            stmt.setString(2, internalEncumbranceBalanceTypeCode);

            try (ResultSet rs = stmt.executeQuery()) {
                final StringBuffer lineBuffer = new StringBuffer(100);
                try {
                    while (rs.next()) {
                        lineBuffer.setLength(0);

                        lineBuffer.append(StringUtils.rightPad(rs.getString(1),
                                lMap.get(KFSPropertyConstants.EMPLID),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(2),
                                lMap.get(KFSPropertyConstants.POSITION_NUMBER),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(3),
                                lMap.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(4),
                                lMap.get(KFSPropertyConstants.ACCOUNT_NUMBER),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(5),
                                lMap.get(KFSPropertyConstants.SUB_ACCOUNT_NUMBER),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(6),
                                lMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(7),
                                lMap.get(KFSPropertyConstants.FINANCIAL_OBJECT_CODE),
                                PAD_CHAR
                        ));
                        lineBuffer.append(StringUtils.rightPad(rs.getString(8),
                                lMap.get(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE),
                                PAD_CHAR
                        ));
                        final BigDecimal bal = rs.getBigDecimal(9);
                        lineBuffer.append((bal.signum() == -1) ? '-' : '+');
                        lineBuffer.append(StringUtils.rightPad(bal.abs().toString(),
                                lMap.get(KFSPropertyConstants.TRANSACTION_LEDGER_ENTRY_AMOUNT),
                                PAD_CHAR
                        ));
                        outputStream.println(lineBuffer.toString());
                        outputLines++;
                    }
                } catch (final SQLException ex) {
                    LOG.error("Unable to iterate over encumbrance balance records", ex);
                    throw new RuntimeException("Unable to iterate over encumbrance balance records", ex);
                }

                LOG.info("{} records written to output file.", outputLines);
            } catch (final SQLException ex) {
                LOG.error("Unable to create and execute statement to retrieve labor encumbrance balances", ex);
                throw new RuntimeException("Unable to create and execute statement to retrieve labor encumbrance " +
                                           "balances",
                        ex
                );
            }
        } catch (final FileNotFoundException ex) {
            LOG.error("Unable to open output file for writing: {}", fileName::getAbsolutePath);
            throw new RuntimeException("Unable to open output file for writing: " + fileName.getAbsolutePath(), ex);
        } catch (final IOException ex) {
            LOG.error("Unable to create output file for balance records: {}", fileName::getAbsolutePath);
            throw new RuntimeException("Unable to create output file for balance records: " + fileName.getAbsolutePath(),
                    ex
            );
        } catch (final SQLException ex) {
            LOG.error("Unable to open SQL connection");
            throw new RuntimeException(ex);
        }
        return outputLines;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }
}
