/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.batch.service.PostTransaction;
import org.kuali.kfs.gl.batch.service.VerifyTransaction;
import org.kuali.kfs.gl.report.LedgerSummaryReport;
import org.kuali.kfs.gl.report.TransactionListingReport;
import org.kuali.kfs.gl.service.OriginEntryGroupService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.batch.LaborPosterStep;
import org.kuali.kfs.module.ld.batch.service.LaborPosterService;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.document.validation.impl.TransactionFieldValidator;
import org.kuali.kfs.module.ld.service.LaborOriginEntryService;
import org.kuali.kfs.module.ld.service.LaborTransactionDescriptionService;
import org.kuali.kfs.module.ld.util.LaborLedgerUnitOfWork;
import org.kuali.kfs.module.ld.util.LaborOriginEntryFileIterator;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.sys.service.ReportWriterService;
import org.springframework.transaction.annotation.Transactional;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.nio.charset.StandardCharsets;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * The Labor Ledger Poster accepts pending entries generated by Labor Ledger e-docs (such as Salary Expense Transfer
 * and Benefit Expense Transfer), and combines them with entries from external systems. It edits the entries for
 * validity. Invalid entries can be marked for Labor Ledger Error Correction process. The Poster writes valid entries
 * to the Labor Ledger Entry table, updates balances in the Labor Ledger Balance table, and summarizes the entries for
 * posting to the General Ledger.
 */
@Transactional
public class LaborPosterServiceImpl implements LaborPosterService {

    private static final Logger LOG = LogManager.getLogger();

    protected LaborOriginEntryService laborOriginEntryService;
    protected OriginEntryGroupService originEntryGroupService;
    protected LaborTransactionDescriptionService laborTransactionDescriptionService;

    protected ReportWriterService reportWriterService;
    protected ReportWriterService errorListingReportWriterService;
    protected ReportWriterService ledgerSummaryReportWriterService;
    protected ReportWriterService laborGlEntryStatisticsReportWriterService;

    protected DateTimeService dateTimeService;
    protected VerifyTransaction laborPosterTransactionValidator;
    protected ParameterService parameterService;

    protected PostTransaction laborLedgerEntryPoster;
    protected PostTransaction laborLedgerBalancePoster;
    protected PostTransaction laborGLLedgerEntryPoster;

    protected int numberOfErrorOriginEntry;

    protected String batchFileDirectoryName;
    protected PrintStream POSTER_OUTPUT_ERR_FILE_ps;

    @Override
    public void postMainEntries() {
        LOG.debug("postMainEntries() started");

        final Date runDate = dateTimeService.getCurrentSqlDate();
        postLaborLedgerEntries(runDate);
    }

    /**
     * post the qualified origin entries into Labor Ledger tables
     *
     * @param runDate      the data when the process is running
     */
    protected void postLaborLedgerEntries(final Date runDate) {
        LOG.debug("postLaborLedgerEntries() started..........................");
        numberOfErrorOriginEntry = 0;
        // change file name to FIS

        final String postInputFileName = batchFileDirectoryName + File.separator +
                                         LaborConstants.BatchFileSystem.POSTER_INPUT_FILE + GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final String postErrFileName = batchFileDirectoryName + File.separator +
                                       LaborConstants.BatchFileSystem.POSTER_ERROR_OUTPUT_FILE +
                                       GeneralLedgerConstants.BatchFileSystem.EXTENSION;

        final FileReader INPUT_GLE_FILE;
        try {
            INPUT_GLE_FILE = new FileReader(postInputFileName, StandardCharsets.UTF_8);
        } catch (final IOException e) {
            throw new RuntimeException(e);
        }

        try {
            POSTER_OUTPUT_ERR_FILE_ps = new PrintStream(postErrFileName);
        } catch (final IOException e) {
            LOG.error("postLaborLedgerEntries cannot open file: {}", e::getMessage, () -> e);
            throw new RuntimeException(e);
        }

        int lineNumber = 0;
        int loadedCount = 0;

        final LaborLedgerUnitOfWork laborLedgerUnitOfWork = new LaborLedgerUnitOfWork();

        final LedgerSummaryReport ledgerSummaryReport = new LedgerSummaryReport();

        final Map<String, Integer> reportSummary = constructPosterReportSummary();
        final Map<String, Integer> glEntryReportSummary = constructGlEntryReportSummary();

        try {
            final BufferedReader INPUT_GLE_FILE_br = new BufferedReader(INPUT_GLE_FILE);
            String currentLine = INPUT_GLE_FILE_br.readLine();

            while (currentLine != null) {
                LaborOriginEntry laborOriginEntry = null;

                try {
                    lineNumber++;
                    if (StringUtils.isNotEmpty(currentLine) && StringUtils.isNotBlank(currentLine.trim())) {
                        laborOriginEntry = new LaborOriginEntry();

                        // checking parsing process and stop poster when it has errors.
                        final List<Message> parsingError = laborOriginEntry.setFromTextFileForBatch(currentLine, lineNumber);
                        if (parsingError.size() > 0) {
                            throw new RuntimeException("Exception happened from parsing process");
                        }

                        loadedCount++;
                        if (loadedCount % 1000 == 0) {
                            LOG.info("{} {}", loadedCount, laborOriginEntry);
                        }

                        final boolean isPostable = postSingleEntryIntoLaborLedger(laborOriginEntry, reportSummary,
                                runDate, currentLine);
                        if (isPostable) {
                            updateReportSummary(glEntryReportSummary, LaborConstants.DestinationNames.ORIGIN_ENTRY,
                                    KFSConstants.OperationType.READ);
                            writeLaborGLEntry(laborOriginEntry, laborLedgerUnitOfWork, runDate, lineNumber,
                                    glEntryReportSummary);

                            ledgerSummaryReport.summarizeEntry(laborOriginEntry);

                            laborOriginEntry = null;
                        }
                    }

                    currentLine = INPUT_GLE_FILE_br.readLine();
                } catch (final RuntimeException re) {
                    // catch here again, it should be from postSingleEntryIntoLaborLedger
                    final int loggableLoadedCount = loadedCount;
                    LOG.error(
                            "postLaborLedgerEntries stopped due to: {} on line number : {}",
                            re::getMessage,
                            () -> loggableLoadedCount,
                            () -> re
                    );
                    LOG.error("laborOriginEntry failure occurred on: {}", laborOriginEntry);
                    throw new RuntimeException("Unable to execute: " + re.getMessage() + " on line number : " + loadedCount, re);
                }
            }

            writeLaborGLEntry(null, laborLedgerUnitOfWork, runDate, lineNumber, glEntryReportSummary);

            INPUT_GLE_FILE_br.close();
            INPUT_GLE_FILE.close();
            POSTER_OUTPUT_ERR_FILE_ps.close();

            fillPosterReportWriter(lineNumber, reportSummary, glEntryReportSummary);
            fillGlEntryReportWriter(glEntryReportSummary);

            // Generate Error Listing Report
            ledgerSummaryReport.writeReport(ledgerSummaryReportWriterService);
            new TransactionListingReport().generateReport(errorListingReportWriterService,
                    new LaborOriginEntryFileIterator(new File(postErrFileName)));
        } catch (final IOException ioe) {
            LOG.error("postLaborLedgerEntries stopped due to: {}", ioe::getMessage, () -> ioe);
            throw new RuntimeException("Unable to execute: " + ioe.getMessage() + " on line number : " + loadedCount,
                    ioe);
        }
    }

    /**
     * post the given entry into the labor ledger tables if the entry is qualified; otherwise report error
     *
     * @param originEntry   the given origin entry, a transaction
     * @param reportSummary the report summary object that need to be update when a transaction is posted
     * @param runDate       the data when the process is running
     * @return true if the given transaction is posted into ledger tables; otherwise, return false
     */
    protected boolean postSingleEntryIntoLaborLedger(
            final LaborOriginEntry originEntry, final Map<String, Integer> reportSummary,
            final Date runDate, final String line) {
        // reject the invalid entry so that it can be available for error correction
        List<Message> errors = new ArrayList<>();
        try {
            errors = validateEntry(originEntry);
        } catch (final Exception e) {
            errors.add(new Message(e.toString() + " occurred for this record.", Message.TYPE_FATAL));
        }

        if (errors != null && !errors.isEmpty()) {
            reportWriterService.writeError(originEntry, errors);
            numberOfErrorOriginEntry += errors.size();
            writeErrorEntry(line);
            return false;
        }

        final String operationOnLedgerEntry = postAsLedgerEntry(originEntry, runDate);
        updateReportSummary(reportSummary, laborLedgerEntryPoster.getDestinationName(), operationOnLedgerEntry);

        final String operationOnLedgerBalance = updateLedgerBalance(originEntry, runDate);
        updateReportSummary(reportSummary, laborLedgerBalancePoster.getDestinationName(), operationOnLedgerBalance);

        return true;
    }

    /**
     * validate the given entry, and generate an error list if the entry cannot meet the business rules
     *
     * @param originEntry the given origin entry, a transaction
     * @return error message list. If the given transaction is invalid, the list has message(s); otherwise, it is empty
     */
    protected List<Message> validateEntry(final LaborOriginEntry originEntry) {
        return laborPosterTransactionValidator.verifyTransaction(originEntry);
    }

    /**
     * post the given entry to the labor entry table
     *
     * @param originEntry the given origin entry, a transaction
     * @param postDate    the data when the transaction is processes return the operation type of the process
     */
    protected String postAsLedgerEntry(final LaborOriginEntry originEntry, final Date postDate) {
        return laborLedgerEntryPoster.post(originEntry, 0, postDate, null);
    }

    /**
     * update the labor ledger balance for the given entry
     *
     * @param originEntry the given origin entry, a transaction
     * @param postDate    the data when the transaction is processes return the operation type of the process
     */
    protected String updateLedgerBalance(final LaborOriginEntry originEntry, final Date postDate) {
        return laborLedgerBalancePoster.post(originEntry, 0, postDate, null);
    }

    /**
     * determine if the given origin entry can be posted back to Labor GL entry
     *
     * @param originEntry the given origin entry, a transaction
     * @return a message list. The list has message(s) if the given origin entry cannot be posted back to Labor GL
     *         entry; otherwise, it is empty
     */
    protected List<Message> isPostableForLaborGLEntry(final LaborOriginEntry originEntry) {
        final List<Message> errors = new ArrayList<>();
        CollectionUtils.addIgnoreNull(errors, TransactionFieldValidator.checkPostablePeridCode(originEntry,
                getPeriodCodesNotProcessed()));
        CollectionUtils.addIgnoreNull(errors, TransactionFieldValidator.checkPostableBalanceTypeCode(originEntry,
                getBalanceTypesNotProcessed()));
        CollectionUtils.addIgnoreNull(errors, TransactionFieldValidator.checkZeroTotalAmount(originEntry));
        return errors;
    }

    // construct a poster report summary object
    protected void fillPosterReportWriter(
            final int lineNumber, final Map<String, Integer> reportSummary,
            final Map<String, Integer> glEntryReportSummary) {
        reportWriterService.writeStatisticLine("SEQUENTIAL RECORDS READ                    %,9d",
                lineNumber);
        reportWriterService.writeStatisticLine("LLEN RECORDS INSERTED (LD_LDGR_ENTR_T)     %,9d",
                reportSummary.get(laborLedgerEntryPoster.getDestinationName() + "," +
                        KFSConstants.OperationType.INSERT));
        reportWriterService.writeStatisticLine("LLBL RECORDS INSERTED (LD_LDGR_BAL_T)      %,9d",
                reportSummary.get(laborLedgerBalancePoster.getDestinationName() + "," +
                        KFSConstants.OperationType.INSERT));
        reportWriterService.writeStatisticLine("LLBL RECORDS UPDATED  (LD_LDGR_BAL_T)      %,9d",
                reportSummary.get(laborLedgerBalancePoster.getDestinationName() + "," +
                        KFSConstants.OperationType.UPDATE));
        reportWriterService.writeStatisticLine("LLGL RECORDS INSERTED (LD_LBR_GL_ENTRY_T)  %,9d",
                glEntryReportSummary.get(laborGLLedgerEntryPoster.getDestinationName() + "," +
                        KFSConstants.OperationType.INSERT));
        reportWriterService.writeStatisticLine("WARNING RECORDS WRITTEN                    %,9d",
                numberOfErrorOriginEntry);
    }

    // fill the poster report writer with the collected data
    protected Map<String, Integer> constructPosterReportSummary() {
        final Map<String, Integer> reportSummary = new HashMap<>();
        reportSummary.put(laborLedgerEntryPoster.getDestinationName() + "," + KFSConstants.OperationType.INSERT, 0);
        reportSummary.put(laborLedgerBalancePoster.getDestinationName() + "," + KFSConstants.OperationType.INSERT, 0);
        reportSummary.put(laborLedgerBalancePoster.getDestinationName() + "," + KFSConstants.OperationType.UPDATE, 0);
        reportSummary.put(laborGLLedgerEntryPoster.getDestinationName() + "," + KFSConstants.OperationType.INSERT, 0);
        return reportSummary;
    }

    // construct a gl entry report summary object
    protected Map<String, Integer> constructGlEntryReportSummary() {
        final Map<String, Integer> glEntryReportSummary = new HashMap<>();
        glEntryReportSummary.put(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                KFSConstants.OperationType.READ, 0);
        glEntryReportSummary.put(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                KFSConstants.OperationType.BYPASS, 0);
        glEntryReportSummary.put(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                KFSConstants.OperationType.SELECT, 0);
        glEntryReportSummary.put(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                KFSConstants.OperationType.REPORT_ERROR, 0);
        glEntryReportSummary.put(laborGLLedgerEntryPoster.getDestinationName() + "," +
                KFSConstants.OperationType.INSERT, 0);

        return glEntryReportSummary;
    }

    // fill the gl entry report writer with the collected data
    protected void fillGlEntryReportWriter(final Map<String, Integer> glEntryReportSummary) {
        laborGlEntryStatisticsReportWriterService.writeStatisticLine("NUMBER OF RECORDS READ              %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                        KFSConstants.OperationType.READ));
        laborGlEntryStatisticsReportWriterService.writeStatisticLine("NUMBER OF RECORDS BYPASSED          %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                        KFSConstants.OperationType.BYPASS));
        laborGlEntryStatisticsReportWriterService.writeStatisticLine("NUMBER OF RECORDS SELECTED          %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                        KFSConstants.OperationType.SELECT));
        laborGlEntryStatisticsReportWriterService.writeStatisticLine("NUMBER OF RECORDS IN ERROR          %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                        KFSConstants.OperationType.REPORT_ERROR));
        laborGlEntryStatisticsReportWriterService.writeStatisticLine("NUMBER OF RECORDS INSERTED          %,9d",
                glEntryReportSummary.get(laborGLLedgerEntryPoster.getDestinationName() + "," +
                        KFSConstants.OperationType.INSERT));
    }

    /**
     * summary the valid origin entries for the General Ledger
     *
     * @param laborOriginEntry      the current entry to check for summarization
     * @param laborLedgerUnitOfWork the current (in process) summarized entry for the GL
     * @param runDate               the data when the process is running
     * @param lineNumber            the line in the input file (used for error message only)
     */
    protected LaborOriginEntry summarizeLaborGLEntries(
            final LaborOriginEntry laborOriginEntry,
            final LaborLedgerUnitOfWork laborLedgerUnitOfWork, final Date runDate, final int lineNumber,
            final Map<String, Integer> glEntryReportSummary) {
        // KFSMI-5308: Description update moved here due to requirement for this to happen before consolidation
        if (ObjectUtils.isNotNull(laborOriginEntry)) {
            final String description = laborTransactionDescriptionService.getTransactionDescription(laborOriginEntry);
            if (StringUtils.isNotEmpty(description)) {
                laborOriginEntry.setTransactionLedgerEntryDescription(description);
            }
        }

        LaborOriginEntry summarizedEntry = null;
        if (laborLedgerUnitOfWork.canContain(laborOriginEntry)) {
            laborLedgerUnitOfWork.addEntryIntoUnit(laborOriginEntry);
            updateReportSummary(glEntryReportSummary, LaborConstants.DestinationNames.ORIGIN_ENTRY,
                    KFSConstants.OperationType.SELECT);
        } else {
            summarizedEntry = laborLedgerUnitOfWork.getWorkingEntry();
            laborLedgerUnitOfWork.resetLaborLedgerUnitOfWork(laborOriginEntry);
        }

        return summarizedEntry;
    }

    protected void writeLaborGLEntry(
            final LaborOriginEntry laborOriginEntry, final LaborLedgerUnitOfWork laborLedgerUnitOfWork,
            final Date runDate, final int lineNumber, final Map<String, Integer> glEntryReportSummary) {
        final LaborOriginEntry summarizedEntry = summarizeLaborGLEntries(laborOriginEntry, laborLedgerUnitOfWork, runDate,
                lineNumber, glEntryReportSummary);
        if (summarizedEntry == null) {
            return;
        }

        try {
            final List<Message> errors = isPostableForLaborGLEntry(summarizedEntry);
            if (errors == null || errors.isEmpty()) {
                final String operationType = laborGLLedgerEntryPoster.post(summarizedEntry, 0, runDate, null);
                updateReportSummary(glEntryReportSummary, laborGLLedgerEntryPoster.getDestinationName(),
                        operationType);
            } else {
                updateReportSummary(glEntryReportSummary, LaborConstants.DestinationNames.ORIGIN_ENTRY,
                        KFSConstants.OperationType.BYPASS);
            }
        } catch (final RuntimeException ioe) {
            // catch here again, it should be from postSingleEntryIntoLaborLedger
            LOG.error("postLaborGLEntries stopped due to: {} on line number : {}",
                    ioe::getMessage,
                    () -> lineNumber,
                    () -> ioe
            );
            throw new RuntimeException("Unable to execute: " + ioe.getMessage() + " on line number : " + lineNumber,
                    ioe);
        }
    }

    protected void updateReportSummary(final Map<String, Integer> reportSummary, final String destination, final String operation) {
        final String key = destination + "," + operation;

        if (reportSummary.containsKey(key)) {
            final Integer count = reportSummary.get(key);
            reportSummary.put(key, count + 1);
        } else {
            reportSummary.put(key, 1);
        }
    }

    protected void writeErrorEntry(final String line) {
        try {
            POSTER_OUTPUT_ERR_FILE_ps.printf("%s\n", line);
        } catch (final Exception e) {
            LOG.error("postAsProcessedOriginEntry stopped due to: {}", e::getMessage, () -> e);
            throw new RuntimeException("Unable to execute: " + e.getMessage(), e);
        }
    }

    /**
     * @return a set of the balance type codes that are bypassed by Labor Poster
     */
    public Collection<String> getBalanceTypesNotProcessed() {
        return parameterService.getParameterValuesAsString(LaborPosterStep.class, LaborParameterConstants.BALANCE_TYPES);
    }

    /**
     * @return a set of the fiscal period codes that are bypassed by Labor Poster
     */
    public Collection<String> getPeriodCodesNotProcessed() {
        return parameterService.getParameterValuesAsString(LaborPosterStep.class, LaborParameterConstants.FISCAL_PERIODS);
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setLaborLedgerBalancePoster(final PostTransaction laborLedgerBalancePoster) {
        this.laborLedgerBalancePoster = laborLedgerBalancePoster;
    }

    public void setLaborGLLedgerEntryPoster(final PostTransaction laborGLLedgerEntryPoster) {
        this.laborGLLedgerEntryPoster = laborGLLedgerEntryPoster;
    }

    public void setLaborLedgerEntryPoster(final PostTransaction laborLedgerEntryPoster) {
        this.laborLedgerEntryPoster = laborLedgerEntryPoster;
    }

    public void setLaborOriginEntryService(final LaborOriginEntryService laborOriginEntryService) {
        this.laborOriginEntryService = laborOriginEntryService;
    }

    public void setOriginEntryGroupService(final OriginEntryGroupService originEntryGroupService) {
        this.originEntryGroupService = originEntryGroupService;
    }

    public void setLaborTransactionDescriptionService(
            final LaborTransactionDescriptionService laborTransactionDescriptionService) {
        this.laborTransactionDescriptionService = laborTransactionDescriptionService;
    }

    public void setReportWriterService(final ReportWriterService reportWriterService) {
        this.reportWriterService = reportWriterService;
    }

    public void setErrorListingReportWriterService(final ReportWriterService errorListingReportWriterService) {
        this.errorListingReportWriterService = errorListingReportWriterService;
    }

    public void setLedgerSummaryReportWriterService(final ReportWriterService ledgerSummaryReportWriterService) {
        this.ledgerSummaryReportWriterService = ledgerSummaryReportWriterService;
    }

    public void setLaborPosterTransactionValidator(final VerifyTransaction laborPosterTransactionValidator) {
        this.laborPosterTransactionValidator = laborPosterTransactionValidator;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setLaborGlEntryStatisticsReportWriterService(
            final ReportWriterService laborGlEntryStatisticsReportWriterService) {
        this.laborGlEntryStatisticsReportWriterService = laborGlEntryStatisticsReportWriterService;
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }
}
