/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.batch.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.gl.GLParameterConstants;
import org.kuali.kfs.gl.GeneralLedgerConstants;
import org.kuali.kfs.gl.businessobject.OriginEntryInformation;
import org.kuali.kfs.gl.report.PosterOutputSummaryReport;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.batch.LaborYearEndBalanceForwardStep;
import org.kuali.kfs.module.ld.batch.service.LaborYearEndBalanceForwardService;
import org.kuali.kfs.module.ld.businessobject.LaborOriginEntry;
import org.kuali.kfs.module.ld.businessobject.LedgerBalanceForYearEndBalanceForward;
import org.kuali.kfs.module.ld.service.LaborLedgerBalanceService;
import org.kuali.kfs.module.ld.util.DebitCreditUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.Message;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.businessobject.SystemOptions;
import org.kuali.kfs.sys.service.OptionsService;
import org.kuali.kfs.sys.service.ReportWriterService;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.springframework.transaction.annotation.Transactional;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.sql.Date;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Labor Ledger Year End Inception to Date Beginning Balance process moves the Year-to-Date Total plus the Contracts
 * and Grants Beginning Balances to the Contracts and Grants Beginning Balances of the new fiscal year for a
 * designated group of accounts (to be identified by fund group and sub fund group).
 */
@Transactional
public class LaborYearEndBalanceForwardServiceImpl implements LaborYearEndBalanceForwardService {

    private static final Logger LOG = LogManager.getLogger();

    private LaborLedgerBalanceService laborLedgerBalanceService;
    private OptionsService optionsService;

    private DateTimeService dateTimeService;
    private ParameterService parameterService;
    private String batchFileDirectoryName;

    private ReportWriterService laborBalanceForwardReportWriterService;
    private ConfigurationService configurationService;

    protected static String PROCESSED_BALANCE_TYPES_LABEL = "PROCESSED BALANCE TYPES";
    protected static String PROCESSED_OBJECT_TYPES_LABEL = "PROCESSED OBJECT TYPES";

    @Override
    public void forwardBalance() {
        final Integer fiscalYear = Integer.valueOf(parameterService.getParameterValueAsString(
                LaborYearEndBalanceForwardStep.class, LaborParameterConstants.FISCAL_YEAR));
        forwardBalance(fiscalYear);
    }

    @Override
    public void forwardBalance(final Integer fiscalYear) {
        forwardBalance(fiscalYear, fiscalYear + 1);
    }

    @Override
    public void forwardBalance(final Integer fiscalYear, final Integer newFiscalYear) {
        final SystemOptions options = optionsService.getOptions(fiscalYear);
        final Date runDate = dateTimeService.getCurrentSqlDate();

        final Map<String, Integer> reportSummary = constructReportSummary();
        final PosterOutputSummaryReport posterOutputSummaryReport = new PosterOutputSummaryReport();

        final Collection<String> processableBalanceTypeCodes = getProcessableBalanceTypeCode(options);
        final Collection<String> processableObjectTypeCodes = getProcessableObjectTypeCodes(options);
        final Collection<String> subFundGroupCodes = getSubFundGroupProcessed();
        final Collection<String> fundGroupCodes = getFundGroupProcessed();

        // create files
        final String balanceForwardsFileName = batchFileDirectoryName + File.separator +
                                               LaborConstants.BatchFileSystem.BALANCE_FORWARDS_FILE +
                                               GeneralLedgerConstants.BatchFileSystem.EXTENSION;
        final File balanceForwardsFile = new File(balanceForwardsFileName);
        final PrintStream balanceForwardsPs;

        try {
            balanceForwardsPs = new PrintStream(balanceForwardsFile, StandardCharsets.UTF_8);
        } catch (final IOException e) {
            throw new RuntimeException("balanceForwardsFile Files doesn't exist " + balanceForwardsFileName);
        }

        // process the selected balances by balance type and object type
        final Map<String, String> fieldValues = new HashMap<>();
        for (final String balanceTypeCode : processableBalanceTypeCodes) {
            fieldValues.put(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, balanceTypeCode);

            for (final String objectTypeCode : processableObjectTypeCodes) {
                fieldValues.put(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE, objectTypeCode);

                fieldValues.remove(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
                fieldValues.remove(KFSPropertyConstants.ACCOUNT_NUMBER);

                final Iterator<LedgerBalanceForYearEndBalanceForward> balanceIterator = laborLedgerBalanceService
                        .findBalancesForFiscalYear(fiscalYear, fieldValues, subFundGroupCodes, fundGroupCodes);
                postSelectedBalancesAsOriginEntries(balanceIterator, newFiscalYear, balanceForwardsPs, runDate,
                        posterOutputSummaryReport, reportSummary);
            }
        }

        fillStatisticsReportWriter(reportSummary);
        fillParametersReportWriter(runDate, fiscalYear, fundGroupCodes, subFundGroupCodes, getOriginationCode(),
                processableBalanceTypeCodes, processableObjectTypeCodes, getDocumentTypeCode());
        posterOutputSummaryReport.writeReport(laborBalanceForwardReportWriterService);

        balanceForwardsPs.close();
        createDoneFile(balanceForwardsFileName);
    }

    // create a done file after balance forward completes.
    protected void createDoneFile(final String originEntryFileName) {
        final String doneFileName = originEntryFileName.replace(GeneralLedgerConstants.BatchFileSystem.EXTENSION,
                GeneralLedgerConstants.BatchFileSystem.DONE_FILE_EXTENSION);
        final File doneFile = new File(doneFileName);

        if (!doneFile.exists()) {
            try {
                doneFile.createNewFile();
            } catch (final IOException e) {
                throw new RuntimeException();
            }
        }
    }

    /**
     * post the qualified balances into origin entry table for the further labor ledger processing
     *
     * @param balanceIterator the given ledger balances that will be carried forward
     * @param newFiscalYear   the new fiscal year
     * @param balanceForwardsPs
     * @param runDate         the date the transaction is posted
     * @param posterOutputSummaryReport
     * @param reportSummary
     * @return the number of qualified balances
     */
    protected int postSelectedBalancesAsOriginEntries(
            final Iterator<LedgerBalanceForYearEndBalanceForward> balanceIterator,
            final Integer newFiscalYear, final PrintStream balanceForwardsPs, final Date runDate,
            final PosterOutputSummaryReport posterOutputSummaryReport, final Map<String, Integer> reportSummary) {
        int numberOfSelectedBalance = 0;
        final String description = getDescription();
        final String originationCode = getOriginationCode();
        final String documentTypeCode = getDocumentTypeCode();

        while (balanceIterator != null && balanceIterator.hasNext()) {
            final LedgerBalanceForYearEndBalanceForward balance = balanceIterator.next();
            updateReportSummary(reportSummary, LaborConstants.DestinationNames.LEDGER_BALANCE,
                    KFSConstants.OperationType.READ);

            final List<Message> errors = null;

            final boolean isValidBalance = validateBalance(balance, errors);
            final LaborOriginEntry laborOriginEntry = new LaborOriginEntry();
            if (isValidBalance) {
                laborOriginEntry.setUniversityFiscalYear(newFiscalYear);
                laborOriginEntry.setFinancialDocumentTypeCode(documentTypeCode);
                laborOriginEntry.setFinancialSystemOriginationCode(originationCode);
                laborOriginEntry.setTransactionLedgerEntryDescription(description);

                postAsOriginEntry(balance, laborOriginEntry, balanceForwardsPs, runDate);
                numberOfSelectedBalance++;

                posterOutputSummaryReport.summarize((OriginEntryInformation) laborOriginEntry);
                updateReportSummary(reportSummary, LaborConstants.DestinationNames.LEDGER_BALANCE,
                        KFSConstants.OperationType.SELECT);
                updateReportSummary(reportSummary, LaborConstants.DestinationNames.LEDGER_BALANCE,
                        KFSConstants.OperationType.INSERT);
            } else if (errors != null && !errors.isEmpty()) {
                ObjectUtil.buildObject(laborOriginEntry, balance);

                laborBalanceForwardReportWriterService.writeError(laborOriginEntry, errors);
                updateReportSummary(reportSummary, LaborConstants.DestinationNames.LEDGER_BALANCE,
                        KFSConstants.OperationType.REPORT_ERROR);
            }
        }

        return numberOfSelectedBalance;
    }

    /**
     * determine if the given balance is qualified to be carried forward to new fiscal year
     *
     * @param balance the given ledger balance that could be carried forward
     * @param errors  the error list that is updated if the given balance is not qualified for carry forward
     * @return true if the balance is qualified; otherwise, false
     */
    protected boolean validateBalance(final LedgerBalanceForYearEndBalanceForward balance, final List<Message> errors) {
        // This is the placeholder for additional business rule validation. The former rules were moved down to data
        // access layer.
        return true;
    }

    /**
     * post the qualified balance into origin entry table for the further labor ledger processing
     *
     * @param balance            the given ledger balance that will be carried forward
     * @param originEntry
     * @param balanceForwardsPs
     * @param postingDate       the date the transaction is posted
     */
    protected void postAsOriginEntry(
            final LedgerBalanceForYearEndBalanceForward balance, final LaborOriginEntry originEntry,
            final PrintStream balanceForwardsPs, final Date postingDate) {
        try {
            originEntry.setAccountNumber(balance.getAccountNumber());
            originEntry.setChartOfAccountsCode(balance.getChartOfAccountsCode());
            originEntry.setSubAccountNumber(balance.getSubAccountNumber());
            originEntry.setFinancialObjectCode(balance.getFinancialObjectCode());
            originEntry.setFinancialSubObjectCode(balance.getFinancialSubObjectCode());
            originEntry.setFinancialBalanceTypeCode(balance.getFinancialBalanceTypeCode());
            originEntry.setFinancialObjectTypeCode(balance.getFinancialObjectTypeCode());

            originEntry.setPositionNumber(balance.getPositionNumber());
            originEntry.setEmplid(balance.getEmplid());
            originEntry.setDocumentNumber(balance.getFinancialBalanceTypeCode() + balance.getAccountNumber());

            originEntry.setProjectCode(KFSConstants.getDashProjectCode());
            originEntry.setUniversityFiscalPeriodCode(KFSConstants.PERIOD_CODE_CG_BEGINNING_BALANCE);

            KualiDecimal transactionAmount = balance.getAccountLineAnnualBalanceAmount();
            transactionAmount = transactionAmount.add(balance.getContractsGrantsBeginningBalanceAmount());

            originEntry.setTransactionLedgerEntryAmount(transactionAmount.abs());
            originEntry.setTransactionDebitCreditCode(DebitCreditUtil.getDebitCreditCode(transactionAmount, false));

            originEntry.setTransactionLedgerEntrySequenceNumber(null);
            originEntry.setTransactionTotalHours(BigDecimal.ZERO);
            originEntry.setTransactionDate(postingDate);

            try {
                balanceForwardsPs.printf("%s\n", originEntry.getLine());
            } catch (final Exception e) {
                throw new RuntimeException(e.toString());
            }
        } catch (final Exception e) {
            LOG.error(e);
        }
    }

    /**
     * @return the fund group codes that are acceptable by year-end process
     */
    protected Collection<String> getFundGroupProcessed() {
        return parameterService.getParameterValuesAsString(LaborYearEndBalanceForwardStep.class,
                LaborParameterConstants.INCEPTION_TO_DATE_FUNDS
        );
    }

    /**
     * @return the fund group codes that are acceptable by year-end process
     */
    protected Collection<String> getSubFundGroupProcessed() {
        return parameterService.getParameterValuesAsString(LaborYearEndBalanceForwardStep.class,
                GLParameterConstants.INCEPTION_TO_DATE_SUB_FUNDS
        );
    }

    /**
     * @return the balance type codes that are acceptable by year-end process
     */
    protected List<String> getProcessableBalanceTypeCode(final SystemOptions options) {
        final List<String> processableBalanceTypeCodes = new ArrayList<>();
        processableBalanceTypeCodes.add(options.getActualFinancialBalanceTypeCd());
        return processableBalanceTypeCodes;
    }

    /**
     * get the object type codes that are acceptable by year-end process
     *
     * @param options the given system options
     * @return the object type codes that are acceptable by year-end process
     */
    protected List<String> getProcessableObjectTypeCodes(final SystemOptions options) {
        final List<String> processableObjectTypeCodes = new ArrayList<>();

        processableObjectTypeCodes.add(options.getFinObjTypeExpenditureexpCd());
        processableObjectTypeCodes.add(options.getFinObjTypeExpNotExpendCode());

        return processableObjectTypeCodes;
    }

    // fill the report writer with the collected data
    protected Map<String, Integer> constructReportSummary() {
        final Map<String, Integer> reportSummary = new HashMap<>();
        reportSummary.put(LaborConstants.DestinationNames.LEDGER_BALANCE + "," + KFSConstants.OperationType.READ, 0);
        reportSummary.put(LaborConstants.DestinationNames.LEDGER_BALANCE + "," + KFSConstants.OperationType.SELECT, 0);
        reportSummary.put(LaborConstants.DestinationNames.LEDGER_BALANCE + "," +
                KFSConstants.OperationType.REPORT_ERROR, 0);
        reportSummary.put(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," + KFSConstants.OperationType.INSERT, 0);

        return reportSummary;
    }

    // fill the gl entry report writer with the collected data
    protected void fillStatisticsReportWriter(final Map<String, Integer> glEntryReportSummary) {
        laborBalanceForwardReportWriterService.writeStatisticLine("NUMBER OF RECORDS READ              %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.LEDGER_BALANCE + "," +
                        KFSConstants.OperationType.READ));
        laborBalanceForwardReportWriterService.writeStatisticLine("NUMBER OF RECORDS SELECTED          %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.LEDGER_BALANCE + "," +
                        KFSConstants.OperationType.SELECT));
        laborBalanceForwardReportWriterService.writeStatisticLine("NUMBER OF RECORDS IN ERROR          %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.LEDGER_BALANCE + "," +
                        KFSConstants.OperationType.REPORT_ERROR));
        laborBalanceForwardReportWriterService.writeStatisticLine("NUMBER OF RECORDS INSERTED          %,9d",
                glEntryReportSummary.get(LaborConstants.DestinationNames.ORIGIN_ENTRY + "," +
                        KFSConstants.OperationType.INSERT));
    }

    /**
     * Report out which significant parameters were used in this report.
     *
     * @param runDate                   the date when this job was run
     * @param closingYear               the fiscal year closed by this job
     * @param processedFundGroups       the fund groups processed by this job
     * @param processedSubFundGroups    the sub-fund groups processed by this job
     * @param originationCode           the origination code of the posted entries
     * @param processedBalanceTypeCodes the balance type codes processed by this job
     * @param processedObjectTypeCodes  the object type codes processed by this job
     * @param documentTypeCode          the document type code of posted entries
     */
    protected void fillParametersReportWriter(
            final Date runDate, final Integer closingYear,
            final Collection<String> processedFundGroups, final Collection<String> processedSubFundGroups, final String originationCode,
            final Collection<String> processedBalanceTypeCodes, final Collection<String> processedObjectTypeCodes,
            final String documentTypeCode) {
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                GLParameterConstants.ANNUAL_CLOSING_TRANSACTION_DATE, runDate.toString());
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                LaborParameterConstants.FISCAL_YEAR, closingYear);
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                LaborParameterConstants.INCEPTION_TO_DATE_FUNDS, StringUtils.join(processedFundGroups, ", "));
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                GLParameterConstants.INCEPTION_TO_DATE_SUB_FUNDS, StringUtils.join(processedSubFundGroups, ", "));
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                LaborParameterConstants.ORIGINATION_CODE, originationCode);
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                LaborYearEndBalanceForwardServiceImpl.PROCESSED_BALANCE_TYPES_LABEL,
                StringUtils.join(processedBalanceTypeCodes, ", "));
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                LaborYearEndBalanceForwardServiceImpl.PROCESSED_OBJECT_TYPES_LABEL,
                StringUtils.join(processedObjectTypeCodes, ", "));
        laborBalanceForwardReportWriterService.writeParameterLine("%32s %10s",
                GLParameterConstants.ANNUAL_CLOSING_DOCUMENT_TYPE, documentTypeCode);
        laborBalanceForwardReportWriterService.pageBreak();
    }

    // update the entry in the given report summary
    protected void updateReportSummary(final Map<String, Integer> reportSummary, final String destination, final String operation) {
        final String key = destination + "," + operation;

        if (reportSummary.containsKey(key)) {
            final Integer count = reportSummary.get(key);
            reportSummary.put(key, count + 1);
        } else {
            reportSummary.put(key, 1);
        }
    }

    /**
     * @return the document type code of the transaction posted by year-end process
     */
    protected String getDocumentTypeCode() {
        return parameterService.getParameterValueAsString(KfsParameterConstants.GENERAL_LEDGER_BATCH.class,
                GLParameterConstants.ANNUAL_CLOSING_DOCUMENT_TYPE);
    }

    /**
     * @return the origination code of the transaction posted by year-end process
     */
    protected String getOriginationCode() {
        return parameterService.getParameterValueAsString(LaborYearEndBalanceForwardStep.class,
                LaborParameterConstants.ORIGINATION_CODE);
    }

    /**
     * @return the description of the transaction posted by year-end process
     */
    protected String getDescription() {
        return configurationService.getPropertyValueAsString(LaborKeyConstants.MESSAGE_YEAR_END_TRANSACTION_DESCRIPTION);
    }

    public void setDateTimeService(final DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setLaborLedgerBalanceService(final LaborLedgerBalanceService laborLedgerBalanceService) {
        this.laborLedgerBalanceService = laborLedgerBalanceService;
    }

    public void setOptionsService(final OptionsService optionsService) {
        this.optionsService = optionsService;
    }

    public void setBatchFileDirectoryName(final String batchFileDirectoryName) {
        this.batchFileDirectoryName = batchFileDirectoryName;
    }

    public void setLaborBalanceForwardReportWriterService(final ReportWriterService laborBalanceForwardReportWriterService) {
        this.laborBalanceForwardReportWriterService = laborBalanceForwardReportWriterService;
    }

    public void setConfigurationService(final ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }
}
