/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject.lookup;

import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.integration.ld.SegmentedBusinessObject;
import org.kuali.kfs.integration.ld.businessobject.inquiry.AbstractPositionDataDetailsInquirableImpl;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.kns.document.authorization.BusinessObjectRestrictions;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.web.comparator.CellComparatorHelper;
import org.kuali.kfs.kns.web.struts.form.LookupForm;
import org.kuali.kfs.kns.web.ui.Column;
import org.kuali.kfs.kns.web.ui.ResultRow;
import org.kuali.kfs.krad.bo.PersistableBusinessObject;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.businessobject.LedgerBalance;
import org.kuali.kfs.module.ld.businessobject.inquiry.LedgerBalanceForExpenseTransferInquirableImpl;
import org.kuali.kfs.module.ld.businessobject.inquiry.PositionDataDetailsInquirableImpl;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.core.web.format.BooleanFormatter;
import org.kuali.kfs.core.web.format.Formatter;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.beans.PropertyDescriptor;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Service implementation of LedgerBalanceForExpenseTransferLookupableHelperService.
 */
public abstract class LedgerBalanceForExpenseTransferLookupableHelperServiceImpl extends
        LedgerBalanceLookupableHelperServiceImpl {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public HtmlData getInquiryUrl(final BusinessObject bo, final String propertyName) {
        if (KFSPropertyConstants.POSITION_NUMBER.equals(propertyName)) {
            final LedgerBalance balance = (LedgerBalance) bo;
            final AbstractPositionDataDetailsInquirableImpl positionDataDetailsInquirable =
                    new PositionDataDetailsInquirableImpl();

            final Map<String, String> fieldValues = new HashMap<>();
            fieldValues.put(propertyName, balance.getPositionNumber());

            final BusinessObject positionData = positionDataDetailsInquirable.getBusinessObject(fieldValues);

            return positionData == null ? new AnchorHtmlData() : positionDataDetailsInquirable.getInquiryUrl(
                    positionData, propertyName);
        }
        return new LedgerBalanceForExpenseTransferInquirableImpl().getInquiryUrl(bo, propertyName);
    }

    @Override
    public List<? extends BusinessObject> getSearchResults(final Map<String, String> fieldValues) {
        return null;
    }

    /**
     * This method performs the lookup and returns a collection of lookup items
     *
     * @param lookupForm
     * @param resultTable
     * @param bounded
     * @return a collection of lookup items
     */
    @Override
    public Collection performLookup(final LookupForm lookupForm, final Collection resultTable, final boolean bounded) {
        final Collection<BusinessObject> displayList;

        // call search method to get results
        if (bounded) {
            displayList = (Collection<BusinessObject>) getSearchResults(lookupForm.getFieldsForLookup());
        } else {
            displayList = (Collection<BusinessObject>) getSearchResultsUnbounded(lookupForm.getFieldsForLookup());
        }

        final List<String> pkNames = getBusinessObjectMetaDataService().listPrimaryKeyFieldNames(getBusinessObjectClass());
        final List<String> returnKeys = getReturnKeys();
        final Person user = GlobalVariables.getUserSession().getPerson();
        // iterate through result list and wrap rows with return url and action urls
        for (final BusinessObject element : displayList) {
            LOG.debug("Doing lookup for {}", element::getClass);
            final BusinessObjectRestrictions businessObjectRestrictions = getBusinessObjectAuthorizationService()
                    .getLookupResultRestrictions(element, user);
            final String returnUrl =
                getReturnUrl(element, lookupForm, returnKeys, businessObjectRestrictions).constructCompleteHtmlTag();

            if (element instanceof PersistableBusinessObject) {
                if (element instanceof SegmentedBusinessObject) {
                    LOG.debug(
                            "segmented property names {}",
                            () -> ((SegmentedBusinessObject) element).getSegmentedPropertyNames()
                    );
                    final Collection<Column> columns = getColumns(element, businessObjectRestrictions);
                    final ResultRow row = new ResultRow((List<Column>) columns, returnUrl, getActionUrls(element, pkNames,
                            businessObjectRestrictions));

                    for (final String propertyName : ((SegmentedBusinessObject) element).getSegmentedPropertyNames()) {
                        columns.add(setupResultsColumn(element, propertyName, businessObjectRestrictions));
                    }

                    row.setObjectId(((PersistableBusinessObject) element).getObjectId());
                    resultTable.add(row);
                } else {
                    final Collection<Column> columns = getColumns(element, businessObjectRestrictions);

                    final ResultRow row = new ResultRow((List<Column>) columns, returnUrl, getActionUrls(element, pkNames,
                            businessObjectRestrictions));
                    row.setObjectId(((PersistableBusinessObject) element).getObjectId());
                    resultTable.add(row);
                }
            }
        }

        return displayList;
    }

    /**
     * @param element
     * @param attributeName
     * @return Column
     */
    protected Column setupResultsColumn(
            final BusinessObject element, final String attributeName,
            final BusinessObjectRestrictions businessObjectRestrictions) {
        final Column col = new Column();

        col.setPropertyName(attributeName);

        String columnTitle = getDataDictionaryService().getAttributeLabel(getBusinessObjectClass(), attributeName);
        if (StringUtils.isBlank(columnTitle)) {
            columnTitle = getDataDictionaryService().getCollectionLabel(getBusinessObjectClass(), attributeName);
        }
        col.setColumnTitle(columnTitle);
        col.setMaxLength(getDataDictionaryService().getAttributeMaxLength(getBusinessObjectClass(), attributeName));

        final Class formatterClass = getDataDictionaryService().getAttributeFormatter(getBusinessObjectClass(), attributeName);
        Formatter formatter = null;
        if (formatterClass != null) {
            try {
                formatter = (Formatter) formatterClass.newInstance();
                col.setFormatter(formatter);
            } catch (InstantiationException | IllegalAccessException e) {
                LOG.error("Unable to get new instance of formatter class: {}", formatterClass::getName);
                throw new RuntimeException("Unable to get new instance of formatter class: " +
                        formatterClass.getName());
            }
        }

        // pick off result column from result list, do formatting
        String propValue = KFSConstants.EMPTY_STRING;
        final Object prop = ObjectUtils.getPropertyValue(element, attributeName);

        // set comparator and formatter based on property type
        Class propClass = null;
        try {
            final PropertyDescriptor propDescriptor = PropertyUtils.getPropertyDescriptor(element, col.getPropertyName());
            if (propDescriptor != null) {
                propClass = propDescriptor.getPropertyType();
            }
        } catch (final Exception e) {
            throw new RuntimeException("Cannot access PropertyType for property " + "'" + col.getPropertyName() +
                    "' " + " on an instance of '" + element.getClass().getName() + "'.", e);
        }

        // formatters
        if (prop != null) {
            // for Booleans, always use BooleanFormatter
            if (prop instanceof Boolean) {
                formatter = new BooleanFormatter();
            }

            if (formatter != null) {
                propValue = (String) formatter.format(prop);
            } else {
                propValue = prop.toString();
            }
        }

        col.setComparator(CellComparatorHelper.getAppropriateComparatorForPropertyClass(propClass));
        col.setValueComparator(CellComparatorHelper.getAppropriateValueComparatorForPropertyClass(propClass));

        propValue = super.maskValueIfNecessary(element, col.getPropertyName(), propValue, businessObjectRestrictions);
        col.setPropertyValue(propValue);

        if (StringUtils.isNotBlank(propValue)) {
            col.setColumnAnchor(getInquiryUrl(element, col.getPropertyName()));
        }
        return col;
    }

    /**
     * Constructs the list of columns for the search results. All properties for the column objects come from the
     * DataDictionary.
     *
     * @param bo
     * @return Collection<Column>
     */
    protected Collection<Column> getColumns(final BusinessObject bo, final BusinessObjectRestrictions businessObjectRestrictions) {
        final Collection<Column> columns = new ArrayList<>();

        for (final String attributeName :
                getBusinessObjectDictionaryService().getLookupResultFieldNames(getBusinessObjectClass())) {
            columns.add(setupResultsColumn(bo, attributeName, businessObjectRestrictions));
        }
        return columns;
    }
}
