/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.dataaccess.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.ojb.broker.query.Criteria;
import org.apache.ojb.broker.query.Query;
import org.apache.ojb.broker.query.QueryByCriteria;
import org.apache.ojb.broker.query.QueryFactory;
import org.apache.ojb.broker.query.ReportQueryByCriteria;
import org.kuali.kfs.gl.OJBUtility;
import org.kuali.kfs.gl.dataaccess.LedgerBalanceBalancingDao;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborPropertyConstants;
import org.kuali.kfs.module.ld.businessobject.EmployeeFunding;
import org.kuali.kfs.module.ld.businessobject.LaborBalanceSummary;
import org.kuali.kfs.module.ld.businessobject.LedgerBalance;
import org.kuali.kfs.module.ld.businessobject.LedgerBalanceForYearEndBalanceForward;
import org.kuali.kfs.module.ld.dataaccess.LaborLedgerBalanceDao;
import org.kuali.kfs.module.ld.util.ConsolidationUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.core.framework.persistence.ojb.dao.PlatformAwareDaoBaseOjb;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * This is the data access object for ledger balance.
 *
 * @see org.kuali.kfs.module.ld.businessobject.LedgerBalance
 */
public class LaborLedgerBalanceDaoOjb extends PlatformAwareDaoBaseOjb implements LaborLedgerBalanceDao,
        LedgerBalanceBalancingDao {

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public Iterator<LedgerBalance> findBalancesForFiscalYear(final Integer year) {
        LOG.debug("findBalancesForFiscalYear() started");

        final Criteria c = new Criteria();
        c.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, year);

        final QueryByCriteria query = QueryFactory.newQuery(LedgerBalance.class, c);
        query.addOrderByAscending(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        query.addOrderByAscending(KFSPropertyConstants.ACCOUNT_NUMBER);
        query.addOrderByAscending(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);

        return getPersistenceBrokerTemplate().getIteratorByQuery(query);
    }

    @Override
    public Iterator<LedgerBalance> findBalancesForFiscalYear(
            final Integer fiscalYear, final Map<String, String> fieldValues,
            final List<String> encumbranceBalanceTypes) {
        final Criteria criteria = buildCriteriaFromMap(fieldValues, new LedgerBalance(), encumbranceBalanceTypes, false);
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);

        final QueryByCriteria query = QueryFactory.newQuery(LedgerBalance.class, criteria);

        query.addOrderByAscending(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        query.addOrderByAscending(KFSPropertyConstants.ACCOUNT_NUMBER);
        query.addOrderByAscending(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE);
        query.addOrderByAscending(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);

        return getPersistenceBrokerTemplate().getIteratorByQuery(query);
    }

    @Override
    public Iterator<LedgerBalanceForYearEndBalanceForward> findBalancesForFiscalYear(
            final Integer fiscalYear,
            final Map<String, String> fieldValues, final List<String> subFundGroupCodes, final List<String> fundGroupCodes) {
        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LedgerBalanceForYearEndBalanceForward());
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);

        final String chartAccountsCode = fieldValues.get(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        final String accountNumber = fieldValues.get(KFSPropertyConstants.ACCOUNT_NUMBER);

        // add subfund criteria if the account is not provided
        if (StringUtils.isEmpty(chartAccountsCode) || StringUtils.isEmpty(accountNumber)) {
            if (subFundGroupCodes != null && !subFundGroupCodes.isEmpty()) {
                final Criteria criteriaForSubFundGroup = new Criteria();
                final String subFundGroupFieldName = KFSPropertyConstants.ACCOUNT + "." +
                                                     KFSPropertyConstants.SUB_FUND_GROUP_CODE;
                criteriaForSubFundGroup.addIn(subFundGroupFieldName, subFundGroupCodes);

                if (fundGroupCodes != null && !fundGroupCodes.isEmpty()) {

                    final Criteria criteriaForFundGroup = new Criteria();
                    final String fundGroupFieldName = KFSPropertyConstants.ACCOUNT + "." +
                                                      KFSPropertyConstants.SUB_FUND_GROUP + "." + KFSPropertyConstants.FUND_GROUP_CODE;
                    criteriaForFundGroup.addIn(fundGroupFieldName, fundGroupCodes);

                    criteriaForSubFundGroup.addOrCriteria(criteriaForFundGroup);
                }

                criteria.addAndCriteria(criteriaForSubFundGroup);
            }
        }

        final QueryByCriteria query = QueryFactory.newQuery(LedgerBalanceForYearEndBalanceForward.class, criteria);
        return getPersistenceBrokerTemplate().getIteratorByQuery(query);
    }

    @Override
    public Iterator<LedgerBalance> findBalance(
            final Map fieldValues, final boolean isConsolidated,
            final List<String> encumbranceBalanceTypes, final boolean noZeroAmounts) {
        LOG.debug("findBalance() started");

        final Query query = getBalanceQuery(fieldValues, isConsolidated, encumbranceBalanceTypes, noZeroAmounts);
        OJBUtility.limitResultSize(query);

        if (isConsolidated) {
            return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
        }
        return getPersistenceBrokerTemplate().getIteratorByQuery(query);
    }

    @Override
    @Deprecated
    public Iterator<LedgerBalance> findBalance(
            final Map fieldValues, final boolean isConsolidated,
            final List<String> encumbranceBalanceTypes) {
        LOG.debug("findBalance() started");

        final Query query = getBalanceQuery(fieldValues, isConsolidated, encumbranceBalanceTypes, false);
        OJBUtility.limitResultSize(query);

        if (isConsolidated) {
            return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
        }
        return getPersistenceBrokerTemplate().getIteratorByQuery(query);
    }

    @Override
    public Iterator getConsolidatedBalanceRecordCount(
            final Map fieldValues, final List<String> encumbranceBalanceTypes,
            final boolean noZeroAmounts) {
        LOG.debug("getBalanceRecordCount() started");

        final ReportQueryByCriteria query = getBalanceCountQuery(fieldValues, encumbranceBalanceTypes, noZeroAmounts);
        return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
    }

    @Override
    @Deprecated
    public Iterator getConsolidatedBalanceRecordCount(final Map fieldValues, final List<String> encumbranceBalanceTypes) {
        LOG.debug("getBalanceRecordCount() started");

        final ReportQueryByCriteria query = getBalanceCountQuery(fieldValues, encumbranceBalanceTypes, false);
        return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
    }

    // build the query for balance search
    protected Query getBalanceQuery(
            final Map fieldValues, final boolean isConsolidated, final List<String> encumbranceBalanceTypes,
            final boolean noZeroAmounts) {
        LOG.debug("Building criteria from map fields: {}", fieldValues::keySet);
        LOG.debug("getBalanceQuery(Map, boolean) started");

        final Criteria criteria = buildCriteriaFromMap(fieldValues, new LedgerBalance(), encumbranceBalanceTypes,
                noZeroAmounts);
        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LedgerBalance.class, criteria);

        // if consolidated, then ignore subaccount number and balance type code
        if (isConsolidated) {
            ConsolidationUtil.buildConsolidatedQuery(query,
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.JULY.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.AUGUST.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.SEPTEMBER.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.OCTOBER.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.NOVEMBER.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.DECEMBER.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.JANUARY.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.FEBRUARY.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.MARCH.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.APRIL.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.MAY.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.JUNE.propertyName),
                    ConsolidationUtil.sum(LaborPropertyConstants.AccountingPeriodProperties.YEAR_END.propertyName));
        }

        return query;
    }

    // build the query for balance search
    protected ReportQueryByCriteria getBalanceCountQuery(
            final Map fieldValues, final List<String> encumbranceBalanceTypes,
            final boolean noZeroAmounts) {
        final Criteria criteria = buildCriteriaFromMap(fieldValues, new LedgerBalance(), encumbranceBalanceTypes,
                noZeroAmounts);
        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LedgerBalance.class, criteria);

        // set the selection attributes
        query.setAttributes(new String[]{"count(*)"});

        final Collection<String> groupByList = ConsolidationUtil.buildGroupByCollection();
        groupByList.remove(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        groupByList.remove(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        groupByList.remove(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);

        // add the group criteria into the selection statement
        final String[] groupBy = groupByList.toArray(new String[groupByList.size()]);
        query.addGroupBy(groupBy);
        return query;
    }

    /**
     * This method builds the query criteria based on the input field map
     *
     * @param fieldValues
     * @param balance
     * @param encumbranceBalanceTypes
     * @param noZeroAmounts           makes sure at least one of the 13 monthly buckets has an amount not equals to
     *                                zero
     * @return a query criteria
     */
    protected Criteria buildCriteriaFromMap(
            final Map fieldValues, final LedgerBalance balance,
            final List<String> encumbranceBalanceTypes, final boolean noZeroAmounts) {
        final Map localFieldValues = new HashMap(fieldValues);

        final Criteria criteria = new Criteria();

        // handle encumbrance balance type
        final String propertyName = KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE;
        if (localFieldValues.containsKey(propertyName)) {
            final String propertyValue = (String) localFieldValues.get(propertyName);
            if (KFSConstants.AGGREGATE_ENCUMBRANCE_BALANCE_TYPE_CODE.equals(propertyValue)) {
                localFieldValues.remove(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE);

                criteria.addIn(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, encumbranceBalanceTypes);
            }
        }

        criteria.addAndCriteria(OJBUtility.buildCriteriaFromMap(localFieldValues, new LedgerBalance()));
        if (noZeroAmounts) {
            final Criteria noZeroAmountsCriteria = new Criteria();
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH1_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH2_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH3_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH4_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH5_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH6_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH7_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH8_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH9_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH10_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH11_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH12_AMOUNT, 0);
            noZeroAmountsCriteria.addEqualTo(KFSPropertyConstants.MONTH13_AMOUNT, 0);
            noZeroAmountsCriteria.setNegative(true);
            criteria.addAndCriteria(noZeroAmountsCriteria);
        }
        return criteria;
    }

    @Override
    public List<LedgerBalance> findCurrentFunds(final Map fieldValues) {
        LOG.debug("Start findCurrentFunds()");

        final Iterator<Object[]> queryResults = findCurrentFundsRawData(fieldValues);
        final List<LedgerBalance> currentFundsCollection = new ArrayList<>();
        while (queryResults != null && queryResults.hasNext()) {
            currentFundsCollection.add(marshalFundsAsLedgerBalance(queryResults.next()));
        }
        return currentFundsCollection;
    }

    @Override
    public List<LedgerBalance> findEncumbranceFunds(final Map fieldValues) {
        LOG.debug("Start findEncumbranceFunds()");

        final Iterator<Object[]> queryResults = findEncumbranceFundsRawData(fieldValues);
        final List<LedgerBalance> currentFundsCollection = new ArrayList<>();
        while (queryResults != null && queryResults.hasNext()) {
            currentFundsCollection.add(marshalFundsAsLedgerBalance(queryResults.next()));
        }
        return currentFundsCollection;
    }

    @Override
    public List<EmployeeFunding> findCurrentEmployeeFunds(final Map fieldValues) {
        LOG.debug("Start findCurrentEmployeeFunds()");

        final Iterator<Object[]> queryResults = findCurrentFundsRawData(fieldValues);
        final List<EmployeeFunding> currentFundsCollection = new ArrayList<>();
        while (queryResults != null && queryResults.hasNext()) {
            currentFundsCollection.add(marshalFundsAsEmployeeFunding(queryResults.next()));
        }
        return currentFundsCollection;
    }

    @Override
    public List<EmployeeFunding> findEncumbranceEmployeeFunds(final Map fieldValues) {
        LOG.debug("Start findCurrentEmployeeFunds()");

        final Iterator<Object[]> queryResults = findEncumbranceFundsRawData(fieldValues);
        final List<EmployeeFunding> currentFundsCollection = new ArrayList<>();
        while (queryResults != null && queryResults.hasNext()) {
            currentFundsCollection.add(marshalFundsAsEmployeeFunding(queryResults.next()));
        }
        return currentFundsCollection;
    }

    @Override
    public List<LaborBalanceSummary> findBalanceSummary(final Integer fiscalYear, final Collection<String> balanceTypes) {
        LOG.debug("Start findBalanceSummary()");

        final Criteria criteria = new Criteria();
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        criteria.addIn(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, balanceTypes);

        final Iterator<Object[]> queryResults = findBalanceSummaryRawData(criteria);
        final List<LaborBalanceSummary> balanceSummaryCollection = new ArrayList<>();
        while (queryResults != null && queryResults.hasNext()) {
            balanceSummaryCollection.add(marshalFundsAsLaborBalanceSummary(queryResults.next()));
        }
        return balanceSummaryCollection;
    }

    // get the current funds according to the given criteria
    protected Iterator<Object[]> findCurrentFundsRawData(final Map fieldValues) {
        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LedgerBalance());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, KFSConstants.BALANCE_TYPE_ACTUAL);

        final List<String> objectTypeCodes = new ArrayList<>();
        objectTypeCodes.add(LaborConstants.BalanceInquiries.EMPLOYEE_FUNDING_EXPENSE_OBJECT_TYPE_CODE);
        objectTypeCodes.add(LaborConstants.BalanceInquiries.EMPLOYEE_FUNDING_NORMAL_OP_EXPENSE_OBJECT_TYPE_CODE);
        criteria.addIn(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE, objectTypeCodes);

        return findFundsRawData(criteria);
    }

    // get the encumbrance funds according to the given criteria
    protected Iterator<Object[]> findEncumbranceFundsRawData(final Map fieldValues) {
        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LedgerBalance());
        criteria.addEqualTo(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE,
                KFSConstants.BALANCE_TYPE_INTERNAL_ENCUMBRANCE);

        return findFundsRawData(criteria);
    }

    // get the funds based on the given criteria
    protected Iterator<Object[]> findFundsRawData(final Criteria criteria) {
        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LedgerBalance.class, criteria);

        final List<String> groupByList = getGroupByListForFundingInquiry();
        final String[] groupBy = groupByList.toArray(new String[groupByList.size()]);
        query.addGroupBy(groupBy);

        final List<String> getAttributeList = getAttributeListForFundingInquiry(false);
        final String[] attributes = getAttributeList.toArray(new String[getAttributeList.size()]);
        query.setAttributes(attributes);

        OJBUtility.limitResultSize(query);
        return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
    }

    // get the balance summary based on the given criteria
    protected Iterator<Object[]> findBalanceSummaryRawData(final Criteria criteria) {
        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LedgerBalance.class, criteria);

        final List<String> groupByList = getGroupByListForBalanceSummary();
        final String[] groupBy = groupByList.toArray(new String[groupByList.size()]);
        query.addGroupBy(groupBy);

        final List<String> getAttributeList = getAttributeListForBalanceSummary(false);
        final String[] attributes = getAttributeList.toArray(new String[getAttributeList.size()]);
        query.setAttributes(attributes);

        query.addOrderByAscending(groupByList.get(0));
        return getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);
    }

    // marshal into AccountStatusBaseFunds from the query result
    protected LedgerBalance marshalFundsAsLedgerBalance(final Object[] queryResult) {
        final LedgerBalance ledgerBalance = new LedgerBalance();
        final List<String> keyFields = getAttributeListForFundingInquiry(true);

        ObjectUtil.buildObject(ledgerBalance, queryResult, keyFields);
        return ledgerBalance;
    }

    // marshal into AccountStatusBaseFunds from the query result
    protected EmployeeFunding marshalFundsAsEmployeeFunding(final Object[] queryResult) {
        final EmployeeFunding employeeFunding = new EmployeeFunding();
        final List<String> keyFields = getAttributeListForFundingInquiry(true);

        ObjectUtil.buildObject(employeeFunding, queryResult, keyFields);
        return employeeFunding;
    }

    // marshal into LaborBalanceSummary from the query result
    protected LaborBalanceSummary marshalFundsAsLaborBalanceSummary(final Object[] queryResult) {
        return new LaborBalanceSummary(queryResult);
    }

    // define the attribute list that can be used to group the search results
    protected List<String> getGroupByListForFundingInquiry() {
        final List<String> groupByList = new ArrayList<>();
        groupByList.add(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR);
        groupByList.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        groupByList.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        groupByList.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        groupByList.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        groupByList.add(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        groupByList.add(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);
        groupByList.add(KFSPropertyConstants.POSITION_NUMBER);
        groupByList.add(KFSPropertyConstants.EMPLID);
        return groupByList;
    }

    // define the return attribute list for funding query
    protected List<String> getAttributeListForFundingInquiry(final boolean isAttributeNameNeeded) {
        final List<String> attributeList = getGroupByListForFundingInquiry();
        attributeList.add(ConsolidationUtil.wrapAttributeName(
                KFSPropertyConstants.ACCOUNTING_LINE_ANNUAL_BALANCE_AMOUNT, isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(
                KFSPropertyConstants.FINANCIAL_BEGINNING_BALANCE_LINE_AMOUNT, isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(
                KFSPropertyConstants.CONTRACTS_GRANTS_BEGINNING_BALANCE_AMOUNT, isAttributeNameNeeded));
        return attributeList;
    }

    // define the attribute list that can be used to group the search results
    protected List<String> getGroupByListForBalanceSummary() {
        final List<String> groupByList = new ArrayList<>();
        groupByList.add("account.subFundGroup.fundGroupCode");
        return groupByList;
    }

    // define the return attribute list for balance summary
    protected List<String> getAttributeListForBalanceSummary(final boolean isAttributeNameNeeded) {
        final List<String> attributeList = getGroupByListForBalanceSummary();
        attributeList.add(ConsolidationUtil.wrapAttributeName(
                KFSPropertyConstants.ACCOUNTING_LINE_ANNUAL_BALANCE_AMOUNT, isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(
                KFSPropertyConstants.FINANCIAL_BEGINNING_BALANCE_LINE_AMOUNT, isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(
                KFSPropertyConstants.CONTRACTS_GRANTS_BEGINNING_BALANCE_AMOUNT, isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH1_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH2_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH3_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH4_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH5_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH6_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH7_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH8_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH9_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH10_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH11_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH12_AMOUNT,
                isAttributeNameNeeded));
        attributeList.add(ConsolidationUtil.wrapAttributeName(KFSPropertyConstants.MONTH13_AMOUNT,
                isAttributeNameNeeded));
        return attributeList;
    }

    @Override
    public List<List<String>> findAccountsInFundGroups(
            final Integer fiscalYear, final Map<String, String> fieldValues,
            final List<String> subFundGroupCodes, final List<String> fundGroupCodes) {
        final Criteria criteria = OJBUtility.buildCriteriaFromMap(fieldValues, new LedgerBalanceForYearEndBalanceForward());
        criteria.addEqualTo(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);

        if (subFundGroupCodes != null && !subFundGroupCodes.isEmpty()) {
            final Criteria criteriaForSubFundGroup = new Criteria();
            final String subFundGroupFieldName = KFSPropertyConstants.ACCOUNT + "." +
                                                 KFSPropertyConstants.SUB_FUND_GROUP_CODE;
            criteriaForSubFundGroup.addIn(subFundGroupFieldName, subFundGroupCodes);

            if (fundGroupCodes != null && !fundGroupCodes.isEmpty()) {

                final Criteria criteriaForFundGroup = new Criteria();
                final String fundGroupFieldName = KFSPropertyConstants.ACCOUNT + "." + KFSPropertyConstants.SUB_FUND_GROUP +
                                                  "." + KFSPropertyConstants.FUND_GROUP_CODE;
                criteriaForFundGroup.addIn(fundGroupFieldName, fundGroupCodes);

                criteriaForSubFundGroup.addOrCriteria(criteriaForFundGroup);
            }
            criteria.addAndCriteria(criteriaForSubFundGroup);
        }

        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LedgerBalanceForYearEndBalanceForward.class,
                criteria);

        query.setAttributes(LaborConstants.ACCOUNT_FIELDS);
        query.setDistinct(true);

        final Iterator<Object[]> accountIterator = getPersistenceBrokerTemplate().getReportQueryIteratorByQuery(query);

        final List<List<String>> accounts = new ArrayList<>();
        while (accountIterator != null && accountIterator.hasNext()) {
            final Object[] accountObject = accountIterator.next();

            final List<String> account = new ArrayList<>();
            account.add(accountObject[0].toString());
            account.add(accountObject[1].toString());

            accounts.add(account);
        }
        return accounts;
    }

    @Override
    public Collection<LedgerBalance> findLedgerBalances(
            final Map<String, List<String>> fieldValues,
            final Map<String, List<String>> excludedFieldValues, final Set<Integer> fiscalYears, final List<String> balanceTypeList,
            final List<String> positionObjectGroupCodes) {
        final Criteria criteria = new Criteria();

        for (final String fieldName : fieldValues.keySet()) {
            final Criteria criteriaForIncludedFields = new Criteria();
            criteria.addIn(fieldName, fieldValues.get(fieldName));
            criteria.addAndCriteria(criteriaForIncludedFields);
        }

        for (final String fieldName : excludedFieldValues.keySet()) {
            final Criteria criteriaForExcludedFields = new Criteria();
            criteria.addNotIn(fieldName, excludedFieldValues.get(fieldName));
            criteria.addAndCriteria(criteriaForExcludedFields);
        }

        if (fiscalYears != null && !fiscalYears.isEmpty()) {
            final Criteria criteriaForFiscalYear = new Criteria();
            criteriaForFiscalYear.addIn(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYears);
            criteria.addAndCriteria(criteriaForFiscalYear);
        }

        if (balanceTypeList != null && !balanceTypeList.isEmpty()) {
            final Criteria criteriaForBalanceTypes = new Criteria();
            criteriaForBalanceTypes.addIn(KFSPropertyConstants.FINANCIAL_BALANCE_TYPE_CODE, balanceTypeList);
            criteria.addAndCriteria(criteriaForBalanceTypes);
        }

        if (positionObjectGroupCodes != null && !positionObjectGroupCodes.isEmpty()) {
            final Criteria criteriaForLaborObjects = new Criteria();
            criteriaForLaborObjects.addIn(KFSPropertyConstants.LABOR_OBJECT + "." +
                    KFSPropertyConstants.POSITION_OBJECT_GROUP_CODE, positionObjectGroupCodes);
            criteria.addAndCriteria(criteriaForLaborObjects);
        }

        final QueryByCriteria query = QueryFactory.newQuery(LedgerBalance.class, criteria);
        return getPersistenceBrokerTemplate().getCollectionByQuery(query);
    }

    @Override
    public void deleteLedgerBalancesPriorToYear(final Integer fiscalYear, final String chartOfAccountsCode) {
        LOG.debug("deleteLedgerBalancesPriorToYear() started");

        final Criteria criteria = new Criteria();
        criteria.addLessThan(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, fiscalYear);
        criteria.addEqualTo(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE, chartOfAccountsCode);

        final QueryByCriteria query = new QueryByCriteria(LedgerBalance.class, criteria);
        getPersistenceBrokerTemplate().deleteByQuery(query);
    }

    @Override
    public Integer findCountGreaterOrEqualThan(final Integer year) {
        final Criteria criteria = new Criteria();
        criteria.addGreaterOrEqualThan(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR, year);

        final ReportQueryByCriteria query = QueryFactory.newReportQuery(LedgerBalance.class, criteria);
        return getPersistenceBrokerTemplate().getCount(query);
    }
}
