/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coa.businessobject.Account;
import org.kuali.kfs.coa.service.AccountService;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.businessobject.ExpenseTransferAccountingLine;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.util.LaborPendingEntryGenerator;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.context.SpringContext;

import java.util.List;

/**
 * Labor Document class for the Benefit Expense Transfer Document and a base class for the year end benefit expense
 * transfer document
 */
public class BenefitExpenseTransferDocument extends LaborExpenseTransferDocumentBase {

    private static final Logger LOG = LogManager.getLogger();
    protected transient String chartOfAccountsCode;
    protected transient String accountNumber;
    protected transient Account account;

    public BenefitExpenseTransferDocument() {
        super();
    }

    @Override
    public boolean generateLaborLedgerPendingEntries(
            final AccountingLine accountingLine,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        LOG.debug("started processGenerateLaborLedgerPendingEntries()");
        boolean isSuccessful = true;
        final ExpenseTransferAccountingLine expenseTransferAccountingLine = (ExpenseTransferAccountingLine) accountingLine;

        final List<LaborLedgerPendingEntry> expensePendingEntries = LaborPendingEntryGenerator
                .generateExpensePendingEntries(this, expenseTransferAccountingLine, sequenceHelper);
        if (expensePendingEntries != null && !expensePendingEntries.isEmpty()) {
            isSuccessful = getLaborLedgerPendingEntries().addAll(expensePendingEntries);
        }

        return isSuccessful;
    }

    @Override
    public boolean generateLaborLedgerBenefitClearingPendingEntries(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        return true;
    }

    @Override
    public List getLaborLedgerPendingEntriesForSearching() {
        return super.getLaborLedgerPendingEntries();
    }

    public String getChartOfAccountsCode() {
        final AccountService accountService = SpringContext.getBean(AccountService.class);
        if (!accountService.accountsCanCrossCharts()) {
            if (ObjectUtils.isNotNull(account)) {
                chartOfAccountsCode = account.getChartOfAccountsCode();
            }
        }
        return chartOfAccountsCode;
    }

    public void setChartOfAccountsCode(final String chartOfAccountsCode) {
        this.chartOfAccountsCode = chartOfAccountsCode;
    }

    public String getAccountNumber() {
        return accountNumber;
    }

    public void setAccountNumber(final String accountNumber) {
        this.accountNumber = accountNumber;
        if (StringUtils.isNotEmpty(accountNumber)) {
            final AccountService accountService = SpringContext.getBean(AccountService.class);
            if (!accountService.accountsCanCrossCharts()) {
                final Account acct = accountService.getUniqueAccountForAccountNumber(accountNumber);
                setChartOfAccountsCode(acct.getChartOfAccountsCode());
                setAccount(acct);
            }
        }
    }

    public Account getAccount() {
        return account;
    }

    public void setAccount(final Account account) {
        this.account = account;
    }

}
