/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2023 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.fp.document.JournalVoucherDocument;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.rules.rule.event.KualiDocumentEvent;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborConstants.JournalVoucherOffsetType;
import org.kuali.kfs.module.ld.businessobject.LaborJournalVoucherAccountingLineParser;
import org.kuali.kfs.module.ld.businessobject.LaborJournalVoucherDetail;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.service.LaborLedgerPendingEntryService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.businessobject.AccountingLineParser;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.document.AmountTotaling;
import org.kuali.kfs.sys.service.GeneralLedgerPendingEntryService;
import org.kuali.kfs.sys.service.HomeOriginationService;

import java.util.ArrayList;
import java.util.List;

/**
 * Labor Document class for the Labor Ledger Journal Voucher.
 */
public class LaborJournalVoucherDocument extends JournalVoucherDocument implements LaborLedgerPostingDocument,
        AmountTotaling {

    private static final Logger LOG = LogManager.getLogger();
    protected String offsetTypeCode = JournalVoucherOffsetType.NO_OFFSET.typeCode;
    protected List<LaborLedgerPendingEntry> laborLedgerPendingEntries;

    public LaborJournalVoucherDocument() {
        super();
        setLaborLedgerPendingEntries(new ArrayList<>());
    }

    @Override
    public Class getSourceAccountingLineClass() {
        return LaborJournalVoucherDetail.class;
    }

    @Override
    public LaborLedgerPendingEntry getLaborLedgerPendingEntry(final int index) {
        while (laborLedgerPendingEntries.size() <= index) {
            laborLedgerPendingEntries.add(new LaborLedgerPendingEntry());
        }
        return laborLedgerPendingEntries.get(index);
    }

    public String getOffsetTypeCode() {
        return offsetTypeCode;
    }

    public void setOffsetTypeCode(final String offsetTypeCode) {
        this.offsetTypeCode = offsetTypeCode;
    }

    @Override
    public List<LaborLedgerPendingEntry> getLaborLedgerPendingEntries() {
        return laborLedgerPendingEntries;
    }

    @Override
    public void setLaborLedgerPendingEntries(final List<LaborLedgerPendingEntry> laborLedgerPendingEntries) {
        this.laborLedgerPendingEntries = laborLedgerPendingEntries;
    }

    /**
     * Used to get the appropriate <code>{@link AccountingLineParser}</code> for the <code>Document</code>
     *
     * @return AccountingLineParser
     */
    @Override
    public AccountingLineParser getAccountingLineParser() {
        return new LaborJournalVoucherAccountingLineParser(getBalanceTypeCode());
    }

    /**
     * Override to call super and then iterate over all GLPEs and update the approved code appropriately.
     */
    @Override
    public void doRouteStatusChange(final DocumentRouteStatusChange statusChangeEvent) {
        super.doRouteStatusChange(statusChangeEvent);
        if (getDocumentHeader().getWorkflowDocument().isProcessed()) {
            changeLedgerPendingEntriesApprovedStatusCode();
        } else if (getDocumentHeader().getWorkflowDocument().isCanceled()
                || getDocumentHeader().getWorkflowDocument().isDisapproved()) {
            removeLedgerPendingEntries();
        }
    }

    /**
     * This method iterates over all of the pending entries for a document and sets their approved status code to
     * APPROVED "A".
     */
    protected void changeLedgerPendingEntriesApprovedStatusCode() {
        for (final LaborLedgerPendingEntry pendingEntry : laborLedgerPendingEntries) {
            pendingEntry.setFinancialDocumentApprovedCode(KFSConstants.DocumentStatusCodes.APPROVED);
        }
    }

    /**
     * This method calls the service to remove all of the pending entries associated with this document
     */
    protected void removeLedgerPendingEntries() {
        final LaborLedgerPendingEntryService laborLedgerPendingEntryService =
                SpringContext.getBean(LaborLedgerPendingEntryService.class);
        laborLedgerPendingEntryService.delete(getDocumentHeader().getDocumentNumber());
    }

    @Override
    public boolean generateLaborLedgerBenefitClearingPendingEntries(
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        return true;
    }

    @Override
    public boolean generateLaborLedgerPendingEntries(
            final AccountingLine accountingLine,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        LOG.debug("processGenerateLaborLedgerPendingEntries() started");

        try {
            final LaborLedgerPendingEntry pendingLedgerEntry = new LaborLedgerPendingEntry();

            // populate the explicit entry
            ObjectUtil.buildObject(pendingLedgerEntry, accountingLine);

            final GeneralLedgerPendingEntryService pendingEntryService =
                    SpringContext.getBean(GeneralLedgerPendingEntryService.class);
            pendingEntryService.populateExplicitGeneralLedgerPendingEntry(this, accountingLine, sequenceHelper,
                    pendingLedgerEntry);

            // apply the labor JV specific information
            customizeExplicitGeneralLedgerPendingEntry(accountingLine, pendingLedgerEntry);
            pendingLedgerEntry.setFinancialDocumentTypeCode(getOffsetTypeCode());

            if (StringUtils.isBlank(((LaborJournalVoucherDetail) accountingLine).getEmplid())) {
                pendingLedgerEntry.setEmplid(LaborConstants.getDashEmplId());
            }

            if (StringUtils.isBlank(((LaborJournalVoucherDetail) accountingLine).getPositionNumber())) {
                pendingLedgerEntry.setPositionNumber(LaborConstants.getDashPositionNumber());
            }

            final String originationCode = SpringContext.getBean(HomeOriginationService.class).getHomeOrigination()
                    .getFinSystemHomeOriginationCode();
            pendingLedgerEntry.setFinancialSystemOriginationCode(originationCode);

            pendingLedgerEntry.setTransactionLedgerEntrySequenceNumber(sequenceHelper.getSequenceCounter());

            pendingLedgerEntry.refreshReferenceObject("financialObject");

            getLaborLedgerPendingEntries().add(pendingLedgerEntry);

            sequenceHelper.increment();
        } catch (final Exception e) {
            LOG.error("Cannot add a Labor Ledger Pending Entry into the list");
            return false;
        }

        return true;
    }

    /**
     * If the document has a total amount, call method on document to get the total and set in doc header.
     */
    @Override
    public void prepareForSave() {
        if (!SpringContext.getBean(LaborLedgerPendingEntryService.class).generateLaborLedgerPendingEntries(this)) {
            logErrors();
            throw new ValidationException("labor ledger LLPE generation failed");
        }

        super.prepareForSave();
    }

    @Override
    public void prepareForSave(final KualiDocumentEvent event) {
        prepareForSave();
        super.prepareForSave(event);
    }

    /**
     * This is a "do nothing" version of the method - it just won't create GLPEs
     */
    @Override
    public boolean generateGeneralLedgerPendingEntries(
            final GeneralLedgerPendingEntrySourceDetail glpeSourceDetail,
            final GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        return true;
    }

    /**
     * Labor docs never generate general ledger pending entries, and therefore, this method is never called, but we
     * always return true, since we're required to have a concrete representation
     */
    @Override
    public boolean isDebit(final GeneralLedgerPendingEntrySourceDetail postable) {
        return true;
    }

    @Override
    public List getLaborLedgerPendingEntriesForSearching() {
        return getLaborLedgerPendingEntries();
    }
}
