/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.kuali.kfs.coa.COAKeyConstants;
import org.kuali.kfs.krad.document.Document;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.ld.document.LaborExpenseTransferDocumentBase;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.businessobject.AccountingLine;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;

public class LaborExpenseTransferValidAccountValidation extends GenericValidation {

    private Document documentForValidation;

    /**
     * determine whether the given accounting line has already been in the given document
     *
     * @return true if the given accounting line has already been in the given document; otherwise, false
     */
    @Override
    public boolean validate(final AttributedDocumentEvent event) {
        final Document documentForValidation = getDocumentForValidation();

        final LaborExpenseTransferDocumentBase expenseTransferDocument =
                (LaborExpenseTransferDocumentBase) documentForValidation;

        // check to ensure the accounts in source/target accounting lines are valid
        return isValidAccount(expenseTransferDocument);
    }

    /**
     * Determine whether the accounts in source/target accounting lines are valid
     *
     * @param expenseTransferDocument the given accounting document
     * @return true if the accounts in source/target accounting lines are valid; otherwise, false
     */
    public boolean isValidAccount(final LaborExpenseTransferDocumentBase expenseTransferDocument) {
        for (final Object sourceAccountingLine : expenseTransferDocument.getSourceAccountingLines()) {
            final AccountingLine line = (AccountingLine) sourceAccountingLine;
            if (ObjectUtils.isNull(line.getAccount())) {
                GlobalVariables.getMessageMap().putError(KFSPropertyConstants.SOURCE_ACCOUNTING_LINES,
                        COAKeyConstants.ERROR_DOCUMENT_GLOBAL_ACCOUNT_INVALID_ACCOUNT, line.getChartOfAccountsCode(),
                        line.getAccountNumber());
                return false;
            }
        }

        for (final Object targetAccountingLine : expenseTransferDocument.getTargetAccountingLines()) {
            final AccountingLine line = (AccountingLine) targetAccountingLine;
            if (ObjectUtils.isNull(line.getAccount())) {
                GlobalVariables.getMessageMap().putError(KFSPropertyConstants.TARGET_ACCOUNTING_LINES,
                        COAKeyConstants.ERROR_DOCUMENT_GLOBAL_ACCOUNT_INVALID_ACCOUNT, line.getChartOfAccountsCode(),
                        line.getAccountNumber());
                return false;
            }
        }
        return true;
    }

    public Document getDocumentForValidation() {
        return documentForValidation;
    }

    public void setDocumentForValidation(final Document documentForValidation) {
        this.documentForValidation = documentForValidation;
    }
}
