/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.document.validation.impl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.LaborKeyConstants;
import org.kuali.kfs.module.ld.LaborParameterConstants;
import org.kuali.kfs.module.ld.document.SalaryExpenseTransferDocument;
import org.kuali.kfs.module.ld.document.service.SalaryExpenseTransferTransactionAgeService;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;

/**
 * Validates Salary Expense Transfer document's accounting lines and late adjustment tab
 */
public class SalaryExpenseTransferLateAdjustmentValidation extends GenericValidation {

    protected static ParameterService parameterService;
    protected static SalaryExpenseTransferTransactionAgeService salaryTransferTransactionAgeService;

    private static final Logger LOG = LogManager.getLogger();

    @Override
    public boolean validate(final AttributedDocumentEvent event) {
        LOG.debug("validate accounting lines and late adjustment tab called");

        final SalaryExpenseTransferDocument currDocument = (SalaryExpenseTransferDocument) event.getDocument();

        final int defaultNumFiscalPeriods = Integer.parseInt(parameterService.getParameterValueAsString(
                KfsParameterConstants.LABOR_DOCUMENT.class,
                LaborParameterConstants.DEFAULT_NUMBER_OF_FISCAL_PERIODS_LATE_ADJUSTMENT_TAB_REQUIRED));
        LOG.debug("defaultPeriodsFromParameter: {}", defaultNumFiscalPeriods);

        // check fiscal periods and the system parameters involving age of transaction
        boolean olderThanFiscalPeriods = !salaryTransferTransactionAgeService.defaultNumberOfFiscalPeriodsCheck(
                currDocument.getSourceAccountingLines(), defaultNumFiscalPeriods);
        LOG.debug("olderThanFiscalPeriods for source accounting lines: {}", olderThanFiscalPeriods);

        if (!olderThanFiscalPeriods) {
            olderThanFiscalPeriods = !salaryTransferTransactionAgeService.defaultNumberOfFiscalPeriodsCheck(
                    currDocument.getTargetAccountingLines(), defaultNumFiscalPeriods);
            LOG.debug("olderThanFiscalPeriods for target accounting lines: {}", olderThanFiscalPeriods);
        }

        // check for blank fields in the Late Adjustment Tab
        final LateAdjustmentValidation ecValidation = new LateAdjustmentValidation();
        final int numBlankFields = ecValidation.lateAdjustmentBlankFieldsCheck(currDocument);
        LOG.debug("numBlankFields: {}", numBlankFields);

        // if there is a transaction older than DEFAULT_NUMBER_OF_FISCAL_PERIODS_LATE_ADJUSTMENT_TAB_REQUIRED and
        // Late Adjustment tab is partially filled
        if (olderThanFiscalPeriods && numBlankFields > 0
            && numBlankFields <= LaborConstants.LateAdjustment.NUM_LATE_ADJUST_FIELDS) {
            GlobalVariables.getMessageMap().putErrorForSectionId(LaborKeyConstants.LateAdjustment.ERROR_LATE_ADJUST_KEY,
                    LaborKeyConstants.LateAdjustment.ERROR_LATE_ADJUST_FISCAL_PER_PARAM_TRIGGERED);
        }

        return true;
    }

    public ParameterService getParameterService() {
        return parameterService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public SalaryExpenseTransferTransactionAgeService getSalaryExpenseTransferTransactionAgeService() {
        return salaryTransferTransactionAgeService;
    }

    public void setSalaryExpenseTransferTransactionAgeService(
            final SalaryExpenseTransferTransactionAgeService salaryExpenseTransferTransactionAgeService) {
        salaryTransferTransactionAgeService = salaryExpenseTransferTransactionAgeService;
    }
}
