/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.ld.businessobject.lookup;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.gl.Constant;
import org.kuali.kfs.integration.ld.businessobject.inquiry.AbstractPositionDataDetailsInquirableImpl;
import org.kuali.kfs.kns.lookup.AbstractLookupableHelperServiceImpl;
import org.kuali.kfs.kns.lookup.HtmlData;
import org.kuali.kfs.kns.lookup.HtmlData.AnchorHtmlData;
import org.kuali.kfs.kns.lookup.CollectionIncomplete;
import org.kuali.kfs.krad.util.BeanPropertyComparator;
import org.kuali.kfs.module.ld.LaborConstants;
import org.kuali.kfs.module.ld.businessobject.EmployeeFunding;
import org.kuali.kfs.module.ld.businessobject.LaborLedgerPendingEntry;
import org.kuali.kfs.module.ld.businessobject.inquiry.EmployeeFundingInquirableImpl;
import org.kuali.kfs.module.ld.businessobject.inquiry.PositionDataDetailsInquirableImpl;
import org.kuali.kfs.module.ld.service.LaborInquiryOptionsService;
import org.kuali.kfs.module.ld.service.LaborLedgerBalanceService;
import org.kuali.kfs.module.ld.service.LaborLedgerPendingEntryService;
import org.kuali.kfs.module.ld.util.DebitCreditUtil;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.KFSPropertyConstants;
import org.kuali.kfs.sys.ObjectUtil;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.krad.bo.BusinessObject;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * The EmployeeFundingLookupableHelperServiceImpl class is the front-end for all Employee Funding balance inquiry
 * processing.
 */
public class EmployeeFundingLookupableHelperServiceImpl extends AbstractLookupableHelperServiceImpl {

    private LaborLedgerBalanceService laborLedgerBalanceService;
    private LaborInquiryOptionsService laborInquiryOptionsService;
    private LaborLedgerPendingEntryService laborLedgerPendingEntryService;

    @Override
    public HtmlData getInquiryUrl(final BusinessObject bo, final String propertyName) {
        if (KFSPropertyConstants.POSITION_NUMBER.equals(propertyName)) {
            final EmployeeFunding employeeFunding = (EmployeeFunding) bo;
            final AbstractPositionDataDetailsInquirableImpl positionDataDetailsInquirable =
                    new PositionDataDetailsInquirableImpl();

            final Map<String, String> fieldValues = new HashMap<>();
            fieldValues.put(propertyName, employeeFunding.getPositionNumber());

            final BusinessObject positionData = positionDataDetailsInquirable.getBusinessObject(fieldValues);

            return positionData == null ? new AnchorHtmlData(KFSConstants.EMPTY_STRING, KFSConstants.EMPTY_STRING) :
                    positionDataDetailsInquirable.getInquiryUrl(positionData, propertyName);
        }

        return new EmployeeFundingInquirableImpl().getInquiryUrl(bo, propertyName);
    }

    @Override
    public List<? extends BusinessObject> getSearchResults(final Map<String, String> fieldValues) {
        setBackLocation(fieldValues.get(KFSConstants.BACK_LOCATION));
        setDocFormKey(fieldValues.get(KFSConstants.DOC_FORM_KEY));

        final boolean showBlankLine = showBlankLines(fieldValues);
        fieldValues.remove(Constant.BLANK_LINE_OPTION);

        // get the pending entry option. This method must be prior to the get search results
        final String pendingEntryOption = laborInquiryOptionsService.getSelectedPendingEntryOption(fieldValues);

        // test if the consolidation option is selected or not
        final boolean isConsolidated = false;

        Collection<EmployeeFunding> searchResultsCollection =
                laborLedgerBalanceService.findEmployeeFundingWithCSFTracker(fieldValues, isConsolidated);

        if (!showBlankLine) {
            final Collection<EmployeeFunding> tempSearchResultsCollection = new ArrayList<>();
            for (final EmployeeFunding employeeFunding : searchResultsCollection) {
                // KFSCNTRB-1534- Properly group CSF Items and show all CSF items that should be shown
                if (employeeFunding.getCurrentAmount().isNonZero() ||
                    employeeFunding.getOutstandingEncumbrance().isNonZero() ||
                    employeeFunding.getCsfAmount() != null && employeeFunding.getCsfAmount().isNonZero()) {
                    tempSearchResultsCollection.add(employeeFunding);
                }
            }
            searchResultsCollection = tempSearchResultsCollection;
        }

        // update search results according to the selected pending entry option
        updateByPendingLedgerEntry(searchResultsCollection, fieldValues, pendingEntryOption, isConsolidated);
        searchResultsCollection = consolidateObjectTypeCode(searchResultsCollection);
        // get the actual size of all qualified search results
        final Long actualSize = (long) searchResultsCollection.size();

        return buildSearchResultList(searchResultsCollection, actualSize);
    }

    /**
     * Filter search results to consolidate them if the only difference is object type code.
     *
     * @param searchResultsCollection
     * @return
     */
    private Collection<EmployeeFunding> consolidateObjectTypeCode(final Collection<EmployeeFunding> searchResultsCollection) {
        // KFSCNTRB-1534- Properly group CSF Items and show all CSF items that should be shown
        final Collection<EmployeeFunding> ret = new ArrayList<>(searchResultsCollection.size());
        for (final EmployeeFunding empFunding : searchResultsCollection) {
            final EmployeeFunding temp = findEmployeeFunding(ret, empFunding);
            if (temp == null) {
                ret.add(empFunding);
            } else {
                /*we need to act on `temp' because that's the one from the collection.*/
                temp.setCsfFullTimeEmploymentQuantity(add(temp.getCsfFullTimeEmploymentQuantity(),
                        empFunding.getCsfFullTimeEmploymentQuantity()));
                temp.setCsfAmount(add(temp.getCsfAmount(), empFunding.getCsfAmount()));
                temp.setCurrentAmount(add(temp.getCurrentAmount(), empFunding.getCurrentAmount()));
                temp.setOutstandingEncumbrance(add(temp.getOutstandingEncumbrance(), empFunding.getOutstandingEncumbrance()));
                temp.setTotalAmount(add(temp.getTotalAmount(), empFunding.getTotalAmount()));
            }
        }
        return ret;
    }

    /**
     * Searches the given collection for an element that is equal to the given exhibit for purposes of consolidation.
     *
     * @param coll    The collection to search for a like element.
     * @param exhibit The search criteria.
     * @return The element from the collection that matches the exhibit or null if 1) no items match or
     * the 2) exhibit is null.
     */
    private static EmployeeFunding findEmployeeFunding(final Collection<EmployeeFunding> coll, final EmployeeFunding exhibit) {
        if (exhibit == null) {
            return null;
        }

        for (final EmployeeFunding temp : coll) {
            if (temp == null) {
                continue;
            }

            if (ObjectUtils.notEqual(temp.getEmplid(), exhibit.getEmplid())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getUniversityFiscalYear(), exhibit.getUniversityFiscalYear())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getChartOfAccountsCode(), exhibit.getChartOfAccountsCode())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getAccountNumber(), exhibit.getAccountNumber())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getSubAccountNumber(), exhibit.getSubAccountNumber())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getFinancialObjectCode(), exhibit.getFinancialObjectCode())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getFinancialSubObjectCode(), exhibit.getFinancialSubObjectCode())) {
                continue;
            }
            if (ObjectUtils.notEqual(temp.getPositionNumber(), exhibit.getPositionNumber())) {
                continue;
            }
            return temp;
        }
        /*no items in the collection match the exhibit.*/
        return null;
    }

    /**
     * Adds two KualiDecimal objects in a null-safe way. If one of them is null, the other is returned, otherwise,
     * a new KualiDecimal containing their sum is returned.
     *
     * @param one The first KualiDecimal to add.
     * @param two The second KualiDecimal to add.
     * @return The sum of the two KualiDecimals.
     */
    private static KualiDecimal add(final KualiDecimal one, final KualiDecimal two) {
        if (one == null) {
            return two;
        }
        if (two == null) {
            return one;
        }
        return one.add(two);
    }

    /**
     * Adds two BigDecimal objects. If one of them is null, the other is returned, otherwise, a new BigDecimal
     * containing their sum is returned.
     *
     * @param one The first BigDecimal to add.
     * @param two The second BigDecimal to add.
     * @return The sum of the two BigDecimals.
     */
    private static BigDecimal add(final BigDecimal one, final BigDecimal two) {
        if (one == null) {
            return two;
        }
        if (two == null) {
            return one;
        }
        return one.add(two);
    }

    private boolean showBlankLines(final Map fieldValues) {
        final String pendingEntryOption = (String) fieldValues.get(Constant.BLANK_LINE_OPTION);
        return Constant.SHOW_BLANK_LINE.equals(pendingEntryOption);
    }

    /**
     * build the search result list from the given collection and the number of all qualified search results
     *
     * @param searchResultsCollection the given search results, which may be a subset of the qualified search results
     * @param actualSize              the number of all qualified search results
     * @return the search result list with the given results and actual size
     */
    protected List buildSearchResultList(final Collection searchResultsCollection, final Long actualSize) {
        final CollectionIncomplete results = new CollectionIncomplete(searchResultsCollection, actualSize);

        // sort list if default sort column given
        final List searchResults = results;
        final List<String> defaultSortColumns = getDefaultSortColumns();
        if (defaultSortColumns.size() > 0) {
            results.sort(new BeanPropertyComparator(defaultSortColumns, true));
        }
        return searchResults;
    }

    public void updateByPendingLedgerEntry(
            final Collection entryCollection, final Map<String, String> fieldValues,
            final String pendingEntryOption, final boolean isConsolidated) {
        // determine if search results need to be updated by pending ledger entries
        if (Constant.ALL_PENDING_ENTRY.equals(pendingEntryOption)) {
            updateEntryCollection(entryCollection, fieldValues, false, isConsolidated);
        } else if (Constant.APPROVED_PENDING_ENTRY.equals(pendingEntryOption)) {
            updateEntryCollection(entryCollection, fieldValues, true, isConsolidated);
        }
    }

    public void updateEntryCollection(
            final Collection entryCollection, final Map<String, String> fieldValues, final boolean isApproved,
            final boolean isConsolidated) {
        // go through the pending entries to update the balance collection
        final Iterator<LaborLedgerPendingEntry> pendingEntryIterator =
                laborLedgerPendingEntryService.findPendingLedgerEntriesForLedgerBalance(fieldValues, isApproved);

        while (pendingEntryIterator.hasNext()) {
            final LaborLedgerPendingEntry pendingEntry = pendingEntryIterator.next();

            if (!isEmployeeFunding(pendingEntry)) {
                continue;
            }

            // if consolidated, change the following fields into the default values for consolidation
            if (isConsolidated) {
                pendingEntry.setSubAccountNumber(Constant.CONSOLIDATED_SUB_ACCOUNT_NUMBER);
                pendingEntry.setFinancialSubObjectCode(Constant.CONSOLIDATED_SUB_OBJECT_CODE);
                pendingEntry.setFinancialObjectTypeCode(Constant.CONSOLIDATED_OBJECT_TYPE_CODE);
            }

            EmployeeFunding ledgerBalance = (EmployeeFunding) laborLedgerBalanceService.findLedgerBalance(
                    entryCollection, pendingEntry, getKeyList());
            if (ledgerBalance == null) {
                ledgerBalance = new EmployeeFunding();
                ObjectUtil.buildObject(ledgerBalance, pendingEntry);
                entryCollection.add(ledgerBalance);
            } else {
                laborLedgerBalanceService.updateLedgerBalance(ledgerBalance, pendingEntry);
            }
            updateAmount(ledgerBalance, pendingEntry);
        }
    }

    /**
     * update the amount of the given employee funding with the given pending entry
     *
     * @param employeeFunding the given employee funding
     * @param pendingEntry    the given pending entry
     */
    private void updateAmount(final EmployeeFunding employeeFunding, final LaborLedgerPendingEntry pendingEntry) {
        final String balanceTypeCode = pendingEntry.getFinancialBalanceTypeCode();
        final KualiDecimal amount = DebitCreditUtil.getNumericAmount(pendingEntry.getTransactionLedgerEntryAmount(),
                pendingEntry.getTransactionDebitCreditCode());

        if (StringUtils.equals(balanceTypeCode, KFSConstants.BALANCE_TYPE_ACTUAL)) {
            employeeFunding.setCurrentAmount(amount.add(employeeFunding.getCurrentAmount()));
        } else if (StringUtils.equals(balanceTypeCode, KFSConstants.BALANCE_TYPE_INTERNAL_ENCUMBRANCE)) {
            employeeFunding.setOutstandingEncumbrance(amount.add(employeeFunding.getOutstandingEncumbrance()));
        }
    }

    /**
     * determine whether the given pending entry is qualified to be processed as an employee funding
     *
     * @param pendingEntry the given pending entry
     * @return true if the given pending entry is qualified to be processed as an employee funding; otherwise, false
     */
    private boolean isEmployeeFunding(final LaborLedgerPendingEntry pendingEntry) {
        final String balanceTypeCode = pendingEntry.getFinancialBalanceTypeCode();

        if (StringUtils.equals(balanceTypeCode, KFSConstants.BALANCE_TYPE_ACTUAL)) {
            final String objectTypeCode = pendingEntry.getFinancialObjectTypeCode();
            final String[] objectTypeCodes = {LaborConstants.BalanceInquiries.EMPLOYEE_FUNDING_EXPENSE_OBJECT_TYPE_CODE,
                LaborConstants.BalanceInquiries.EMPLOYEE_FUNDING_NORMAL_OP_EXPENSE_OBJECT_TYPE_CODE};

            return ArrayUtils.contains(objectTypeCodes, objectTypeCode);
        }

        return StringUtils.equals(balanceTypeCode, KFSConstants.BALANCE_TYPE_INTERNAL_ENCUMBRANCE);
    }

    /**
     * @return the primary key list of the business object
     */
    private List<String> getKeyList() {
        final List<String> keyList = new ArrayList<>();
        keyList.add(KFSPropertyConstants.UNIVERSITY_FISCAL_YEAR);
        keyList.add(KFSPropertyConstants.CHART_OF_ACCOUNTS_CODE);
        keyList.add(KFSPropertyConstants.ACCOUNT_NUMBER);
        keyList.add(KFSPropertyConstants.SUB_ACCOUNT_NUMBER);
        keyList.add(KFSPropertyConstants.FINANCIAL_OBJECT_CODE);
        keyList.add(KFSPropertyConstants.FINANCIAL_SUB_OBJECT_CODE);
        keyList.add(KFSPropertyConstants.FINANCIAL_OBJECT_TYPE_CODE);
        keyList.add(KFSPropertyConstants.POSITION_NUMBER);
        keyList.add(KFSPropertyConstants.EMPLID);
        return keyList;
    }

    public void setLaborLedgerBalanceService(final LaborLedgerBalanceService laborLedgerBalanceService) {
        this.laborLedgerBalanceService = laborLedgerBalanceService;
    }

    public void setLaborInquiryOptionsService(final LaborInquiryOptionsService laborInquiryOptionsService) {
        this.laborInquiryOptionsService = laborInquiryOptionsService;
    }

    public void setLaborLedgerPendingEntryService(final LaborLedgerPendingEntryService laborLedgerPendingEntryService) {
        this.laborLedgerPendingEntryService = laborLedgerPendingEntryService;
    }
}
