/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * Created on Aug 25, 2004
 *
 */
package org.kuali.kfs.module.purap.businessobject;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.document.ElectronicInvoiceRejectDocument;
import org.kuali.kfs.module.purap.util.PurApDateFormatUtils;
import org.kuali.kfs.module.purap.util.cxml.CxmlExtrinsic;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

public class ElectronicInvoiceItem {

    private static final Logger LOG = LogManager.getLogger();

    // this class is equipped to hold InvoiceDetailItem values as well as a few rudimentary InvoiceDetailServiceItem
    // values

    private String catalogNumber;
    private String invoiceLineNumber;
    private String quantity;
    private String unitOfMeasure;
    // UnitPrice is deprecated for InvoiceDetailServiceItem tags
    // has money xml node
    private String unitPrice;
    private String unitPriceCurrency;
    // has money xml node
    private String subTotalAmount;
    private String subTotalAmountCurrency;
    // has money xml node
    private String invoiceLineSpecialHandlingAmount;
    private String invoiceLineSpecialHandlingAmountCurrency;
    // has money xml node
    private String invoiceLineShippingAmount;
    private String invoiceLineShippingAmountCurrency;
    // has money xml node  (not all tax fields are stored as tax should never occur)
    private String taxAmount;
    private String taxAmountCurrency;
    private String taxDescription;
    // invoiceLineGrossAmount should = subtotalAmount + taxAmount + invoiceLineSpecialHandlingAmount +
    // invoiceLineShippingAmount
    private String invoiceLineGrossAmount;
    private String invoiceLineGrossAmountCurrency;
    // has money xml node
    private String invoiceLineDiscountAmount;
    private String invoiceLineDiscountAmountCurrency;
    private String invoiceLineDiscountPercentageRate;
    // invoiceLineNetAmount should = invoiceLineGrossAmount - invoiceLineDiscountAmount
    // has money xml node
    private String invoiceLineNetAmount;
    private String invoiceLineNetAmountCurrency;
    private String shippingDateString;
    private Date shippingDate;

    private List invoiceShippingContacts = new ArrayList();
    private List comments = new ArrayList();
    private List extrinsic = new ArrayList();

    // following fields describe PO information
    // only match available for InvoiceDetailServiceItem values
    private String referenceLineNumber;
    // attribute of <InvoiceDetailItemReference> deprecated to be <SerialNumber> inside it
    private String referenceSerialNumber;
    // used only if above String field is null
    private List<String> referenceSerialNumbers = new ArrayList<>();
    private String referenceItemIDSupplierPartID;
    private String referenceItemIDSupplierPartAuxID;
    private String referenceDescription;
    private String referenceManufacturerPartID;
    private String referenceManufacturerName;
    private String referenceCountryCode;
    private String referenceCountryName;

    private ElectronicInvoiceRejectDocument electronicInvoiceRejectDocument;

    public ElectronicInvoiceItem() {
    }

    public Integer getReferenceLineNumberInteger() {
        if (this.referenceLineNumber != null) {
            return Integer.parseInt(referenceLineNumber);
        }
        return null;
    }

    public String getInvoiceLineShippingDescription() {
        return "";
    }

    public BigDecimal getInvoiceLineQuantityBigDecimal() {
        if (StringUtils.isNotEmpty(quantity)) {
            return new BigDecimal(this.quantity);
        } else {
            return null;
        }
    }

    public BigDecimal getInvoiceLineUnitCostBigDecimal() {
        BigDecimal unitprice = BigDecimal.ZERO;
        if (StringUtils.isNotEmpty(unitPrice)) {
            try {
                unitprice = new BigDecimal(this.unitPrice);
            } catch (NumberFormatException nfe) {
                LOG.info("invalid unit price" + this.unitPrice);
            }
        }
        return unitprice;
    }

    public BigDecimal getInvoiceLineSubTotalAmountBigDecimal() {
        BigDecimal subTotalAmount = BigDecimal.ZERO;
        if (StringUtils.isNotEmpty(this.subTotalAmount)) {
            try {
                subTotalAmount = new BigDecimal(this.subTotalAmount);
            } catch (NumberFormatException nfe) {
                LOG.info("invalid sub Total Amount " + this.subTotalAmount);
            }
        }
        return subTotalAmount;
    }

    public BigDecimal getInvoiceLineTaxAmountBigDecimal() {
        if (StringUtils.isNotEmpty(taxAmount)) {
            return new BigDecimal(this.taxAmount);
        } else {
            return BigDecimal.ZERO;
        }
    }

    public BigDecimal getInvoiceLineSpecialHandlingAmountBigDecimal() {
        if (StringUtils.isNotEmpty(invoiceLineSpecialHandlingAmount)) {
            return new BigDecimal(this.invoiceLineSpecialHandlingAmount);
        } else {
            return BigDecimal.ZERO;
        }
    }

    public BigDecimal getInvoiceLineShippingAmountBigDecimal() {
        if (StringUtils.isNotEmpty(invoiceLineShippingAmount)) {
            return new BigDecimal(this.invoiceLineShippingAmount);
        } else {
            return BigDecimal.ZERO;
        }
    }

    public BigDecimal getInvoiceLineGrossAmountBigDecimal() {
        if (StringUtils.isNotEmpty(invoiceLineGrossAmount)) {
            return new BigDecimal(this.invoiceLineGrossAmount);
        } else {
            return BigDecimal.ZERO;
        }
    }

    public BigDecimal getInvoiceLineDiscountAmountBigDecimal() {
        if (StringUtils.isNotEmpty(invoiceLineDiscountAmount)) {
            return new BigDecimal(this.invoiceLineDiscountAmount);
        } else {
            return BigDecimal.ZERO;
        }
    }

    public BigDecimal getInvoiceLineNetAmountBigDecimal() {
        if (StringUtils.isNotEmpty(invoiceLineNetAmount)) {
            return new BigDecimal(this.invoiceLineNetAmount);
        } else {
            return BigDecimal.ZERO;
        }
    }

    public String getShippingDateString() {
        return shippingDateString;
    }

    public void setShippingDateString(String shippingDateString) {
        this.shippingDateString = shippingDateString;
        if (shippingDateString != null && !"".equals(shippingDateString)) {
            SimpleDateFormat sdf = PurApDateFormatUtils.getSimpleDateFormat(
                    PurapConstants.NamedDateFormats.CXML_DATE_FORMAT);
            try {
                this.shippingDate = sdf.parse(shippingDateString);
            } catch (ParseException e) {
                // setting shipping date to null to identify problem
                LOG.error("setShippingDateString() SimpleDateFormat parser error attempting to set invalid " +
                        "date string " + shippingDateString + " in ShippingDate field... setting date to null");
                this.shippingDate = null;
            }
        } else {
            this.shippingDate = null;
        }
    }

    public String getCatalogNumber() {
        return catalogNumber;
    }

    public void setCatalogNumber(String catalogNumber) {
        this.catalogNumber = catalogNumber;
    }

    public List getComments() {
        return comments;
    }

    public void setComments(List comments) {
        this.comments = comments;
    }

    public List getExtrinsic() {
        return extrinsic;
    }

    public void setExtrinsic(List extrinsic) {
        this.extrinsic = extrinsic;
    }

    public String getInvoiceLineDiscountAmount() {
        return invoiceLineDiscountAmount;
    }

    public void setInvoiceLineDiscountAmount(String invoiceLineDiscountAmount) {
        this.invoiceLineDiscountAmount = invoiceLineDiscountAmount;
    }

    public String getInvoiceLineDiscountAmountCurrency() {
        return invoiceLineDiscountAmountCurrency;
    }

    public void setInvoiceLineDiscountAmountCurrency(String invoiceLineDiscountAmountCurrency) {
        this.invoiceLineDiscountAmountCurrency = invoiceLineDiscountAmountCurrency;
    }

    public String getInvoiceLineGrossAmount() {
        return invoiceLineGrossAmount;
    }

    public void setInvoiceLineGrossAmount(String invoiceLineGrossAmount) {
        this.invoiceLineGrossAmount = invoiceLineGrossAmount;
    }

    public String getInvoiceLineGrossAmountCurrency() {
        return invoiceLineGrossAmountCurrency;
    }

    public void setInvoiceLineGrossAmountCurrency(String invoiceLineGrossAmountCurrency) {
        this.invoiceLineGrossAmountCurrency = invoiceLineGrossAmountCurrency;
    }

    public String getInvoiceLineNetAmount() {
        return invoiceLineNetAmount;
    }

    public void setInvoiceLineNetAmount(String invoiceLineNetAmount) {
        this.invoiceLineNetAmount = invoiceLineNetAmount;
    }

    public String getInvoiceLineNetAmountCurrency() {
        return invoiceLineNetAmountCurrency;
    }

    public void setInvoiceLineNetAmountCurrency(String invoiceLineNetAmountCurrency) {
        this.invoiceLineNetAmountCurrency = invoiceLineNetAmountCurrency;
    }

    public String getInvoiceLineNumber() {
        return invoiceLineNumber;
    }

    public void setInvoiceLineNumber(String invoiceLineNumber) {
        this.invoiceLineNumber = invoiceLineNumber;
    }

    public String getInvoiceLineShippingAmount() {
        return invoiceLineShippingAmount;
    }

    public void setInvoiceLineShippingAmount(String invoiceLineShippingAmount) {
        this.invoiceLineShippingAmount = invoiceLineShippingAmount;
    }

    public String getInvoiceLineShippingAmountCurrency() {
        return invoiceLineShippingAmountCurrency;
    }

    public void setInvoiceLineShippingAmountCurrency(String invoiceLineShippingAmountCurrency) {
        this.invoiceLineShippingAmountCurrency = invoiceLineShippingAmountCurrency;
    }

    public String getInvoiceLineSpecialHandlingAmount() {
        return invoiceLineSpecialHandlingAmount;
    }

    public void setInvoiceLineSpecialHandlingAmount(String invoiceLineSpecialHandlingAmount) {
        this.invoiceLineSpecialHandlingAmount = invoiceLineSpecialHandlingAmount;
    }

    public String getInvoiceLineSpecialHandlingAmountCurrency() {
        return invoiceLineSpecialHandlingAmountCurrency;
    }

    public void setInvoiceLineSpecialHandlingAmountCurrency(
        String invoiceLineSpecialHandlingAmountCurrency) {
        this.invoiceLineSpecialHandlingAmountCurrency = invoiceLineSpecialHandlingAmountCurrency;
    }

    public List getInvoiceShippingContacts() {
        return invoiceShippingContacts;
    }

    public void setInvoiceShippingContacts(List invoiceShippingContacts) {
        this.invoiceShippingContacts = invoiceShippingContacts;
    }

    public void addInvoiceShippingContacts(ElectronicInvoiceContact contact) {
        invoiceShippingContacts.add(contact);
    }

    public String getQuantity() {
        return quantity;
    }

    public void setQuantity(String quantity) {
        this.quantity = quantity;
    }

    public String getReferenceCountryCode() {
        return referenceCountryCode;
    }

    public void setReferenceCountryCode(String referenceCountryCode) {
        this.referenceCountryCode = referenceCountryCode;
    }

    public String getReferenceCountryName() {
        return referenceCountryName;
    }

    public void setReferenceCountryName(String referenceCountryName) {
        this.referenceCountryName = referenceCountryName;
    }

    public String getReferenceDescription() {
        return referenceDescription;
    }

    public void setReferenceDescription(String referenceDescription) {
        this.referenceDescription = referenceDescription;
    }

    public String getReferenceItemIDSupplierPartAuxID() {
        return referenceItemIDSupplierPartAuxID;
    }

    public void setReferenceItemIDSupplierPartAuxID(String referenceItemIDSupplierPartAuxID) {
        this.referenceItemIDSupplierPartAuxID = referenceItemIDSupplierPartAuxID;
    }

    public String getReferenceItemIDSupplierPartID() {
        return referenceItemIDSupplierPartID;
    }

    public void setReferenceItemIDSupplierPartID(String referenceItemIDSupplierPartID) {
        this.referenceItemIDSupplierPartID = referenceItemIDSupplierPartID;
    }

    public String getReferenceLineNumber() {
        return referenceLineNumber;
    }

    public void setReferenceLineNumber(String referenceLineNumber) {
        this.referenceLineNumber = referenceLineNumber;
    }

    public String getReferenceManufacturerName() {
        return referenceManufacturerName;
    }

    public void setReferenceManufacturerName(String referenceManufacturerName) {
        this.referenceManufacturerName = referenceManufacturerName;
    }

    public String getReferenceManufacturerPartID() {
        return referenceManufacturerPartID;
    }

    public void setReferenceManufacturerPartID(String referenceManufacturerPartID) {
        this.referenceManufacturerPartID = referenceManufacturerPartID;
    }

    public String getReferenceSerialNumber() {
        return referenceSerialNumber;
    }

    public void setReferenceSerialNumber(String referenceSerialNumber) {
        this.referenceSerialNumber = referenceSerialNumber;
    }

    public List getReferenceSerialNumbers() {
        return referenceSerialNumbers;
    }

    public void setReferenceSerialNumbers(List referenceSerialNumbers) {
        this.referenceSerialNumbers = referenceSerialNumbers;
    }

    public Date getShippingDate() {
        return shippingDate;
    }

    public void setShippingDate(Date shippingDate) {
        this.shippingDate = shippingDate;
    }

    public String getSubTotalAmount() {
        return subTotalAmount;
    }

    public void setSubTotalAmount(String subTotalAmount) {
        this.subTotalAmount = subTotalAmount;
    }

    public String getSubTotalAmountCurrency() {
        return subTotalAmountCurrency;
    }

    public void setSubTotalAmountCurrency(String subTotalAmountCurrency) {
        this.subTotalAmountCurrency = subTotalAmountCurrency;
    }

    public String getTaxAmount() {
        return taxAmount;
    }

    public void setTaxAmount(String taxAmount) {
        this.taxAmount = taxAmount;
    }

    public String getTaxAmountCurrency() {
        return taxAmountCurrency;
    }

    public void setTaxAmountCurrency(String taxAmountCurrency) {
        this.taxAmountCurrency = taxAmountCurrency;
    }

    public String getTaxDescription() {
        return taxDescription;
    }

    public void setTaxDescription(String taxDescription) {
        this.taxDescription = taxDescription;
    }

    public String getUnitOfMeasure() {
        return unitOfMeasure;
    }

    public void setUnitOfMeasure(String unitOfMeasure) {
        this.unitOfMeasure = unitOfMeasure;
    }

    public String getUnitPrice() {
        return unitPrice;
    }

    public void setUnitPrice(String unitPrice) {
        this.unitPrice = unitPrice;
    }

    public String getUnitPriceCurrency() {
        return unitPriceCurrency;
    }

    public void setUnitPriceCurrency(String unitPriceCurrency) {
        this.unitPriceCurrency = unitPriceCurrency;
    }

    public ElectronicInvoiceRejectDocument getElectronicInvoiceRejectDocument() {
        return electronicInvoiceRejectDocument;
    }

    public void setElectronicInvoiceRejectDocument(ElectronicInvoiceRejectDocument electronicInvoiceRejectDocument) {
        this.electronicInvoiceRejectDocument = electronicInvoiceRejectDocument;
    }

    public void addReferenceSerialNumber(String referenceSerialNumber) {
        this.referenceSerialNumbers.add(referenceSerialNumber);
    }

    public String[] getReferenceSerialNumbersAsArray() {
        if (referenceSerialNumbers.size() > 0) {
            String[] serialNumbers = new String[referenceSerialNumbers.size()];
            referenceSerialNumbers.toArray(serialNumbers);
            return serialNumbers;
        }
        return null;
    }

    public void addExtrinsic(CxmlExtrinsic extrinsic) {
        this.extrinsic.add(extrinsic);
    }

    public CxmlExtrinsic[] getExtrinsicAsArray() {
        if (extrinsic.size() > 0) {
            CxmlExtrinsic[] extrinsics = new CxmlExtrinsic[extrinsic.size()];
            extrinsic.toArray(extrinsics);
            return extrinsics;
        }
        return null;
    }

    public void addComments(String comment) {
        this.comments.add(comment);
    }

    public String getInvoiceLineDiscountPercentageRate() {
        return invoiceLineDiscountPercentageRate;
    }

    public void setInvoiceLineDiscountPercentageRate(String invoiceLineDiscountPercentageRate) {
        this.invoiceLineDiscountPercentageRate = invoiceLineDiscountPercentageRate;
    }

    public String toString() {
        ToStringBuilder toString = new ToStringBuilder(this);

        toString.append("invoiceLineNumber", getInvoiceLineNumber());
        toString.append("quantity", getQuantity());
        toString.append("catalogNumber", getCatalogNumber());
        toString.append("unitOfMeasure", getUnitOfMeasure());
        toString.append("unitPrice", getUnitPrice());
        toString.append("unitPriceCurrency", getUnitPriceCurrency());
        toString.append("subTotalAmount", getSubTotalAmount());
        toString.append("subTotalAmountCurrency", getSubTotalAmountCurrency());
        toString.append("invoiceLineSpecialHandlingAmount", getInvoiceLineSpecialHandlingAmount());
        toString.append("invoiceLineSpecialHandlingAmountCurrency", getInvoiceLineSpecialHandlingAmountCurrency());
        toString.append("invoiceLineShippingAmount", getInvoiceLineShippingAmount());
        toString.append("invoiceLineShippingAmountCurrency", getInvoiceLineShippingAmountCurrency());
        toString.append("taxAmount", getTaxAmount());
        toString.append("taxAmountCurrency", getTaxAmountCurrency());
        toString.append("taxDescription", getTaxDescription());
        toString.append("invoiceLineGrossAmount", getInvoiceLineGrossAmount());
        toString.append("invoiceLineGrossAmountCurrency", getInvoiceLineGrossAmountCurrency());
        toString.append("invoiceLineDiscountAmount", getInvoiceLineDiscountAmount());
        toString.append("invoiceLineDiscountAmountCurrency", getInvoiceLineDiscountAmountCurrency());
        toString.append("invoiceLineNetAmount", getInvoiceLineNetAmount());
        toString.append("invoiceLineNetAmountCurrency", getInvoiceLineNetAmountCurrency());
        toString.append("shippingDateString", getShippingDateString());

        toString.append("referenceLineNumber", getReferenceLineNumber());
        toString.append("referenceSerialNumber", getReferenceSerialNumber());
        toString.append("referenceSerialNumbersList", getReferenceSerialNumbers());
        toString.append("referenceItemIDSupplierPartID", getReferenceItemIDSupplierPartID());
        toString.append("referenceItemIDSupplierPartAuxID", getReferenceItemIDSupplierPartAuxID());
        toString.append("referenceDescription", getReferenceDescription());
        toString.append("referenceManufacturerPartID", getReferenceManufacturerPartID());
        toString.append("referenceManufacturerName", getReferenceManufacturerName());
        toString.append("referenceCountryCode", getReferenceCountryCode());
        toString.append("referenceCountryName", getReferenceCountryName());

        toString.append("invoiceShippingContacts", getInvoiceShippingContacts());
        toString.append("comments", getComments());
        toString.append("extrinsic", getExtrinsic());

        return toString.toString();
    }

}
