/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2020 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.validation.impl;

import org.apache.commons.lang3.StringUtils;
import org.kuali.kfs.fp.businessobject.NonresidentTaxPercent;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.MessageMap;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PaymentRequestStatuses;
import org.kuali.kfs.module.purap.PurapKeyConstants;
import org.kuali.kfs.module.purap.PurapPropertyConstants;
import org.kuali.kfs.module.purap.document.PaymentRequestDocument;
import org.kuali.kfs.sys.document.validation.GenericValidation;
import org.kuali.kfs.sys.document.validation.event.AttributedDocumentEvent;
import org.kuali.rice.core.api.util.type.KualiDecimal;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class PaymentRequestTaxAreaValidation extends GenericValidation {

    private BusinessObjectService businessObjectService;

    /**
     * Process business rules applicable to tax area data before calculating the withholding tax on payment request.
     *
     * @param event event containing payment request document
     * @return true if all business rules applicable passes; false otherwise.
     */
    @Override
    public boolean validate(AttributedDocumentEvent event) {
        PaymentRequestDocument preq = (PaymentRequestDocument) event.getDocument();

        // do this validation only at route level of awaiting tax review
        if (!StringUtils.equals(preq.getApplicationDocumentStatus(),
                PaymentRequestStatuses.APPDOC_AWAITING_TAX_REVIEW)) {
            return true;
        }

        MessageMap errorMap = GlobalVariables.getMessageMap();
        errorMap.clearErrorPath();
        errorMap.addToErrorPath(PurapConstants.PAYMENT_REQUEST_TAX_TAB_ERRORS);

        boolean valid = validateTaxIncomeClass(preq);
        valid &= validateTaxRates(preq);
        valid &= validateTaxIndicators(preq);

        errorMap.clearErrorPath();
        return valid;
    }

    /**
     * Validates tax income class: when Non-Reportable income class is chosen, all other fields shall be left blank;
     * It assumed that the input tax income class code is valid (existing and active in the system) since it's chosen
     * from drop-down list. otherwise tax rates and country are required;
     *
     * @param preq payment request document
     * @return true if this validation passes; false otherwise.
     */
    protected boolean validateTaxIncomeClass(PaymentRequestDocument preq) {
        boolean valid = true;
        MessageMap errorMap = GlobalVariables.getMessageMap();

        // TaxClassificationCode is required field
        if (StringUtils.isEmpty(preq.getTaxClassificationCode())) {
            valid = false;
            errorMap.putError(PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                    PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_REQUIRED,
                    PurapPropertyConstants.TAX_CLASSIFICATION_CODE);
        } else if (StringUtils.equalsIgnoreCase(preq.getTaxClassificationCode(), "N")) {
            // If TaxClassificationCode is N (Non_Reportable, then other fields shall be blank.
            if (preq.getTaxFederalPercent() != null
                    && preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() != null && preq.getTaxStatePercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_STATE_PERCENT);
            }
            if (!StringUtils.isEmpty(preq.getTaxCountryCode())) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_COUNTRY_CODE,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_COUNTRY_CODE);
            }
            if (!StringUtils.isEmpty(preq.getTaxNQIId())) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_NQI_ID,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_NQI_ID);
            }
            if (preq.getTaxSpecialW4Amount() != null
                    && preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT);
            }
            if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR);
            }
            if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_GROSS_UP_INDICATOR);
            }
            if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR);
            }
            if (Objects.equals(preq.getTaxUSAIDPerDiemIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR);
            }
            if (Objects.equals(preq.getTaxOtherExemptIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR);
            }
        } else {
            // If TaxClassificationCode is not N (Non_Reportable, then the federal/state tax percent and country are
            // required.
            if (preq.getTaxFederalPercent() == null) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_REQUIRED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() == null) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_REQUIRED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_STATE_PERCENT);
            }
            if (StringUtils.isEmpty(preq.getTaxCountryCode())) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_COUNTRY_CODE,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_REQUIRED_IF,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_COUNTRY_CODE);
            }
        }

        return valid;
    }

    /**
     * Validates federal and state tax rates based on each other and the income class. Validation will be bypassed if
     * income class is empty or N, or tax rates are null.
     *
     * @param preq payment request document
     * @return true if this validation passes; false otherwise.
     */
    protected boolean validateTaxRates(PaymentRequestDocument preq) {
        boolean valid = true;
        String code = preq.getTaxClassificationCode();
        BigDecimal fedRate = preq.getTaxFederalPercent();
        BigDecimal stRate = preq.getTaxStatePercent();
        MessageMap errorMap = GlobalVariables.getMessageMap();

        // only test the cases when income class and tax rates aren't empty/N
        if (StringUtils.isEmpty(code) || StringUtils.equalsIgnoreCase(code, "N") || fedRate == null || stRate == null) {
            return true;
        }

        // validate that the federal and state tax rates are among the allowed set
        ArrayList<BigDecimal> fedRates = retrieveTaxRates(code, "F");
        if (!listContainsValue(fedRates, fedRate)) {
            valid = false;
            errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                    PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_VALUE_INVALID_IF,
                    PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_FEDERAL_PERCENT);
        }
        ArrayList<BigDecimal> stRates = retrieveTaxRates(code, "S");
        if (!listContainsValue(stRates, stRate)) {
            valid = false;
            errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                    PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_VALUE_INVALID_IF,
                    PurapPropertyConstants.TAX_CLASSIFICATION_CODE, PurapPropertyConstants.TAX_STATE_PERCENT);
        }

        // validate that the federal and state tax rate abide to certain relationship
        if (fedRate.compareTo(new BigDecimal(0)) == 0 && stRate.compareTo(new BigDecimal(0)) != 0) {
            valid = false;
            errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                    PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                    PurapPropertyConstants.TAX_FEDERAL_PERCENT, PurapPropertyConstants.TAX_STATE_PERCENT);
        }
        boolean hasStRate = "F".equalsIgnoreCase(code) || "A".equalsIgnoreCase(code) || "O".equalsIgnoreCase(code);
        if (fedRate.compareTo(new BigDecimal(0)) > 0 && stRate.compareTo(new BigDecimal(0)) <= 0 && hasStRate) {
            valid = false;
            errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                    PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_NOT_ZERO_IF,
                    PurapPropertyConstants.TAX_FEDERAL_PERCENT, PurapPropertyConstants.TAX_STATE_PERCENT);
        }

        return valid;
    }

    /**
     * Validates rules among tax treaty, gross up, foreign source, USAID, other exempt, and Special W-4.
     *
     * @param preq payment request document
     * @return true if this validation passes; false otherwise.
     */
    protected boolean validateTaxIndicators(PaymentRequestDocument preq) {
        boolean valid = true;
        MessageMap errorMap = GlobalVariables.getMessageMap();

        // if choose tax treaty, cannot choose any of the other above
        if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
            if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR);
            }
            if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR);
            }
            if (Objects.equals(preq.getTaxUSAIDPerDiemIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR);
            }
            if (Objects.equals(preq.getTaxOtherExemptIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR);
            }
            if (preq.getTaxSpecialW4Amount() != null
                    && preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT);
            }
            if (preq.getTaxFederalPercent() != null
                    && preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() != null && preq.getTaxStatePercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR, PurapPropertyConstants.TAX_STATE_PERCENT);
            }
        }

        // if choose gross up, cannot choose any other above, and fed tax rate cannot be zero
        if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
            if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR);
            }
            if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR);
            }
            if (Objects.equals(preq.getTaxUSAIDPerDiemIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR);
            }
            if (Objects.equals(preq.getTaxOtherExemptIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR);
            }
            if (preq.getTaxSpecialW4Amount() != null
                    && preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR, PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT);
            }
            if (preq.getTaxFederalPercent() == null
                    || preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) == 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_NOT_ZERO_IF,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR, PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
        }

        // if choose foreign source, cannot choose any other above, and tax rates shall be zero
        if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
            if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR);
            }
            if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR);
            }
            if (Objects.equals(preq.getTaxUSAIDPerDiemIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR);
            }
            if (Objects.equals(preq.getTaxOtherExemptIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR);
            }
            if (preq.getTaxSpecialW4Amount() != null
                    && preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT);
            }
            if (preq.getTaxFederalPercent() != null
                    && preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() != null
                    && preq.getTaxStatePercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapPropertyConstants.TAX_STATE_PERCENT);
            }
        }

        // if choose USAID per diem, cannot choose any other above except other exempt code, which must be checked;
        // income class shall be fellowship with tax rates 0
        if (Objects.equals(preq.getTaxUSAIDPerDiemIndicator(), true)) {
            if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR);
            }
            if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR);
            }
            if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR);
            }
            if (!Objects.equals(preq.getTaxOtherExemptIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_REQUIRED_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR);
            }
            if (preq.getTaxSpecialW4Amount() != null
                    && preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT);
            }
            if (StringUtils.isEmpty(preq.getTaxClassificationCode())
                    || !StringUtils.equalsIgnoreCase(preq.getTaxClassificationCode(), "F")) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_VALUE_INVALID_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_CLASSIFICATION_CODE);
            }
            if (preq.getTaxFederalPercent() != null
                    && preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() != null
                    && preq.getTaxStatePercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_USAID_PER_DIEM_INDICATOR,
                        PurapPropertyConstants.TAX_STATE_PERCENT);
            }
        }

        // if choose exempt under other code, cannot choose any other above except USAID, and tax rates shall be zero
        if (Objects.equals(preq.getTaxOtherExemptIndicator(), true)) {
            if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR);
            }
            if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapPropertyConstants.TAX_GROSS_UP_INDICATOR);
            }
            if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR);
            }
            if (preq.getTaxFederalPercent() != null
                    && preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() != null
                    && preq.getTaxStatePercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR, PurapPropertyConstants.TAX_STATE_PERCENT);
            }
        }

        // if choose Special W-4, cannot choose tax treaty, gross up, and foreign source; income class shall be
        // fellowship with tax rates 0
        if (preq.getTaxSpecialW4Amount() != null
                && preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) != 0) {
            if (Objects.equals(preq.getTaxExemptTreatyIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapPropertyConstants.TAX_EXEMPT_TREATY_INDICATOR);
            }
            if (Objects.equals(preq.getTaxGrossUpIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_GROSS_UP_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT, PurapPropertyConstants.TAX_GROSS_UP_INDICATOR);
            }
            if (Objects.equals(preq.getTaxForeignSourceIndicator(), true)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_DISALLOWED_IF,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapPropertyConstants.TAX_FOREIGN_SOURCE_INDICATOR);
            }
            //if Exempt Under Other Code box is not checked then validation error...
            if (Objects.equals(preq.getTaxOtherExemptIndicator(), false)) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_EXEMPT_UNDER_OTHER_CODE_MUST_EXIST,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapPropertyConstants.TAX_OTHER_EXEMPT_INDICATOR);
            }
            if (preq.getTaxSpecialW4Amount().compareTo(new KualiDecimal(0)) < 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_VALUE_MUST_NOT_NEGATIVE,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT);
            }
            if (StringUtils.isEmpty(preq.getTaxClassificationCode())
                    || !StringUtils.equalsIgnoreCase(preq.getTaxClassificationCode(), "F")) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_CLASSIFICATION_CODE,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_FIELD_VALUE_INVALID_IF,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT, PurapPropertyConstants.TAX_CLASSIFICATION_CODE);
            }
            if (preq.getTaxFederalPercent() != null
                    && preq.getTaxFederalPercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_FEDERAL_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT, PurapPropertyConstants.TAX_FEDERAL_PERCENT);
            }
            if (preq.getTaxStatePercent() != null
                    && preq.getTaxStatePercent().compareTo(new BigDecimal(0)) != 0) {
                valid = false;
                errorMap.putError(PurapPropertyConstants.TAX_STATE_PERCENT,
                        PurapKeyConstants.ERROR_PAYMENT_REQUEST_TAX_RATE_MUST_ZERO_IF,
                        PurapPropertyConstants.TAX_SPECIAL_W4_AMOUNT, PurapPropertyConstants.TAX_STATE_PERCENT);
            }
        }

        return valid;
    }

    /**
     * Retrieve active nonresident tax rate percent from database based on the specified income class and
     * federal/state tax type.
     *
     * @param incomeClassCode   The specified income class type code.
     * @param incomeTaxTypeCode The specified income tax type code.
     * @return The array list containing the tax rates retrieved.
     */
    public ArrayList<BigDecimal> retrieveTaxRates(String incomeClassCode, String incomeTaxTypeCode) {
        ArrayList<BigDecimal> rates = new ArrayList<>();
        Map<String, String> criterion = new HashMap<>();
        criterion.put("incomeClassCode", incomeClassCode);
        criterion.put("incomeTaxTypeCode", incomeTaxTypeCode);
        // only retrieve active tax percents
        criterion.put("active", "Y");
        List<NonresidentTaxPercent> percents = (List<NonresidentTaxPercent>) businessObjectService
                .findMatching(NonresidentTaxPercent.class, criterion);

        for (NonresidentTaxPercent percent : percents) {
            rates.add(percent.getIncomeTaxPercent());
        }
        return rates;
    }

    /**
     * @param list  the specified ArrayList
     * @param value the specified BigDecimal
     * @return true if the specified ArrayList contains the specified BigDecimal value.
     */
    protected boolean listContainsValue(ArrayList<BigDecimal> list, BigDecimal value) {
        if (list == null || value == null) {
            return false;
        }
        for (BigDecimal val : list) {
            if (val.compareTo(value) == 0) {
                return true;
            }
        }
        return false;
    }

    public BusinessObjectService getBusinessObjectService() {
        return businessObjectService;
    }

    public void setBusinessObjectService(BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
}
