/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/*
 * Created on Feb 13, 2006
 *
 */
package org.kuali.kfs.module.purap.businessobject;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.util.ElectronicInvoiceUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

public class ElectronicInvoiceOrder {

    public static boolean INVOICE_ORDER_REJECTED = true;
    public static boolean INVOICE_ORDER_NOT_REJECTED = false;

    // the following fields come from the <InvoiceDetailOrderInfo> tag
    private String orderReferenceOrderID;
    private String orderReferenceDocumentRefPayloadID;
    private String orderReferenceDocumentRef;
    private String masterAgreementReferenceID;
    private Date masterAgreementReferenceDate;
    private String masterAgreementReferenceDateString;
    private String masterAgreementIDInfoID;
    private Date masterAgreementIDInfoDate;
    private String masterAgreementIDInfoDateString;
    private String orderIDInfoID;
    private Date orderIDInfoDate;
    private String orderIDInfoDateString;
    private String supplierOrderInfoID;

    private String invoicePurchaseOrderID;
    private String orderReferenceOrderDateString;
    private Integer purchaseOrderID = null;
    private String purchaseOrderCampusCode;

    private boolean rejected = INVOICE_ORDER_NOT_REJECTED;
    private List<ElectronicInvoiceRejectReason> orderRejectReasons = new ArrayList<>();

    private List<ElectronicInvoiceItem> invoiceItems = new ArrayList<>();

    public ElectronicInvoiceOrder() {
        super();
    }

    public ElectronicInvoiceItem getElectronicInvoiceItemByPOLineNumber(Integer poLineNumber) {
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            if ((poLineNumber.compareTo(invoiceItem.getReferenceLineNumberInteger())) == 0) {
                return invoiceItem;
            }
        }
        return null;
    }

    /**
     * This method takes in a roleID string and an addressName (constants from mapping file) and returns a valid
     * ElectronicInvoicePostalAddress or null if not found.  If the addressName string is null then the roleID is
     * used to find the first available
     *
     * @param roleID      Cxml role id attribute value
     * @param addressName Cxml name attribute of postaladdress tag
     * @return CxmlPostal Address relating to given parameters
     */
    public ElectronicInvoicePostalAddress getCxmlPostalAddressByRoleID(String roleID, String addressName) {
        if (roleID != null) {
            ElectronicInvoiceContact contact = this.getCxmlContactByRoleID(roleID);
            if (contact != null) {
                for (ElectronicInvoicePostalAddress cpa : contact.getPostalAddresses()) {
                    if (addressName == null) {
                        return cpa;
                    } else {
                        if (addressName.equalsIgnoreCase(cpa.getName())) {
                            return cpa;
                        }
                    }
                }
            }
        }
        return null;
    }

    public ElectronicInvoiceContact getCxmlContactByRoleID(String roleID) {
        if (roleID != null) {
            for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
                for (Object shippingContact : invoiceItem.getInvoiceShippingContacts()) {
                    ElectronicInvoiceContact contact = (ElectronicInvoiceContact) shippingContact;
                    if (roleID.equalsIgnoreCase(contact.getRole())) {
                        return contact;
                    }
                }
            }
        }
        return null;
    }

    /**
     * This method returns the first shipping date found in the list of items. This is called if shipping information
     * is in line. Since system only allows for one shipping date per invoice-order we take the first date we find.
     *
     * @return Date defining first shipping date found or null if none are found
     */
    public Date getInvoiceShippingDate() {
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            Date testDate = invoiceItem.getShippingDate();
            if (testDate != null) {
                return testDate;
            }
        }
        return null;
    }

    /**
     * This method returns the first shipping date string found in the list of items. This is called if shipping
     * information is in line. Since system only allows for one shipping date per invoice-order we take the first
     * date string we find.
     *
     * @return Date defining first shipping date found or null if none are found
     */
    public String getInvoiceShippingDateString() {
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            String testDateString = invoiceItem.getShippingDateString();
            if (testDateString != null && !"".equals(testDateString)) {
                return testDateString;
            }
        }
        return null;
    }

    public String getInvoiceTaxDescription() {
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            BigDecimal taxAmount = invoiceItem.getInvoiceLineTaxAmountBigDecimal();
            if (taxAmount != null && BigDecimal.ZERO.compareTo(taxAmount) != 0) {
                return invoiceItem.getTaxDescription();
            }
        }
        return null;
    }

    public String getInvoiceShippingDescription() {
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            BigDecimal shippingAmount = invoiceItem.getInvoiceLineShippingAmountBigDecimal();
            if (shippingAmount != null && BigDecimal.ZERO.compareTo(shippingAmount) != 0) {
                return PurapConstants.ElectronicInvoice.DEFAULT_SHIPPING_DESCRIPTION;
            }
        }
        return null;
    }

    public String getInvoiceSpecialHandlingDescription() {
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            BigDecimal specialHandlingAmount = invoiceItem.getInvoiceLineSpecialHandlingAmountBigDecimal();
            if (specialHandlingAmount != null && BigDecimal.ZERO.compareTo(specialHandlingAmount) != 0) {
                return PurapConstants.ElectronicInvoice.DEFAULT_SPECIAL_HANDLING_DESCRIPTION;
            }
        }
        return null;
    }

    public BigDecimal getInvoiceSubTotalAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineSubTotalAmountBigDecimal());
        }
        return total;
    }

    public BigDecimal getInvoiceTaxAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineTaxAmountBigDecimal());
        }
        return total;
    }

    public BigDecimal getInvoiceSpecialHandlingAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineSpecialHandlingAmountBigDecimal());
        }
        return total;
    }

    public BigDecimal getInvoiceShippingAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineShippingAmountBigDecimal());
        }
        return total;
    }

    public BigDecimal getInvoiceGrossAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineGrossAmountBigDecimal());
        }
        return total;
    }

    public BigDecimal getInvoiceDiscountAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineDiscountAmountBigDecimal());
        }
        return total;
    }

    public BigDecimal getInvoiceNetAmount() {
        BigDecimal total = BigDecimal.ZERO;
        for (ElectronicInvoiceItem invoiceItem : this.invoiceItems) {
            total = total.add(invoiceItem.getInvoiceLineNetAmountBigDecimal());
        }
        return total;
    }

    public void addRejectReasonToList(ElectronicInvoiceRejectReason reason) {
        this.orderRejectReasons.add(reason);
    }

    /**
     * Altered for special circumstances
     *
     * @param masterAgreementIDInfoDateString The masterAgreementIDInfoDateString to set.
     */
    public void setMasterAgreementIDInfoDateString(String masterAgreementIDInfoDateString) {
        this.masterAgreementIDInfoDateString = masterAgreementIDInfoDateString;
        setMasterAgreementIDInfoDate(ElectronicInvoiceUtils.getDate(masterAgreementIDInfoDateString));
    }

    /**
     * Altered for special circumstances
     *
     * @param masterAgreementReferenceDateString The masterAgreementReferenceDateString to set.
     */
    public void setMasterAgreementReferenceDateString(String masterAgreementReferenceDateString) {
        this.masterAgreementReferenceDateString = masterAgreementReferenceDateString;
        setMasterAgreementIDInfoDate(ElectronicInvoiceUtils.getDate(masterAgreementReferenceDateString));
    }

    /**
     * Altered for special circumstances
     *
     * @param orderIDInfoDateString The orderIDInfoDateString to set.
     */
    public void setOrderIDInfoDateString(String orderIDInfoDateString) {
        this.orderIDInfoDateString = orderIDInfoDateString;
        setOrderIDInfoDate(ElectronicInvoiceUtils.getDate(orderIDInfoDateString));
    }

    public List<ElectronicInvoiceItem> getInvoiceItems() {
        return invoiceItems;
    }

    public void setInvoiceItems(List<ElectronicInvoiceItem> invoiceItems) {
        this.invoiceItems = invoiceItems;
    }

    public String getInvoicePurchaseOrderID() {
        return invoicePurchaseOrderID;
    }

    public void setInvoicePurchaseOrderID(String invoicePurchaseOrderID) {
        this.invoicePurchaseOrderID = invoicePurchaseOrderID;
    }

    public Date getMasterAgreementIDInfoDate() {
        return masterAgreementIDInfoDate;
    }

    public void setMasterAgreementIDInfoDate(Date masterAgreementIDInfoDate) {
        this.masterAgreementIDInfoDate = masterAgreementIDInfoDate;
    }

    public String getMasterAgreementIDInfoID() {
        return masterAgreementIDInfoID;
    }

    public void setMasterAgreementIDInfoID(String masterAgreementIDInfoID) {
        this.masterAgreementIDInfoID = masterAgreementIDInfoID;
    }

    public Date getMasterAgreementReferenceDate() {
        return masterAgreementReferenceDate;
    }

    public void setMasterAgreementReferenceDate(Date masterAgreementReferenceDate) {
        this.masterAgreementReferenceDate = masterAgreementReferenceDate;
    }

    public String getMasterAgreementReferenceID() {
        return masterAgreementReferenceID;
    }

    public void setMasterAgreementReferenceID(String masterAgreementReferenceID) {
        this.masterAgreementReferenceID = masterAgreementReferenceID;
    }

    public Date getOrderIDInfoDate() {
        return orderIDInfoDate;
    }

    public void setOrderIDInfoDate(Date orderIDInfoDate) {
        this.orderIDInfoDate = orderIDInfoDate;
    }

    public String getOrderIDInfoID() {
        return orderIDInfoID;
    }

    public void setOrderIDInfoID(String orderIDInfoID) {
        this.orderIDInfoID = orderIDInfoID;
    }

    public String getOrderReferenceDocumentRef() {
        return orderReferenceDocumentRef;
    }

    public void setOrderReferenceDocumentRef(String orderReferenceDocumentRef) {
        this.orderReferenceDocumentRef = orderReferenceDocumentRef;
    }

    public String getOrderReferenceDocumentRefPayloadID() {
        return orderReferenceDocumentRefPayloadID;
    }

    public void setOrderReferenceDocumentRefPayloadID(String orderReferenceDocumentRefPayloadID) {
        this.orderReferenceDocumentRefPayloadID = orderReferenceDocumentRefPayloadID;
    }

    public String getOrderReferenceOrderID() {
        return orderReferenceOrderID;
    }

    public void setOrderReferenceOrderID(String orderReferenceOrderID) {
        this.orderReferenceOrderID = orderReferenceOrderID;
    }

    public List<ElectronicInvoiceRejectReason> getOrderRejectReasons() {
        return orderRejectReasons;
    }

    public void setOrderRejectReasons(List<ElectronicInvoiceRejectReason> orderRejectReasons) {
        this.orderRejectReasons = orderRejectReasons;
    }

    public String getPurchaseOrderCampusCode() {
        return purchaseOrderCampusCode;
    }

    public void setPurchaseOrderCampusCode(String purchaseOrderCampusCode) {
        this.purchaseOrderCampusCode = purchaseOrderCampusCode;
    }

    public Integer getPurchaseOrderID() {
        return purchaseOrderID;
    }

    public void setPurchaseOrderID(Integer purchaseOrderID) {
        this.purchaseOrderID = purchaseOrderID;
    }

    public boolean isRejected() {
        return rejected;
    }

    public void setRejected(boolean rejected) {
        this.rejected = rejected;
    }

    public String getSupplierOrderInfoID() {
        return supplierOrderInfoID;
    }

    public void setSupplierOrderInfoID(String supplierOrderInfoID) {
        this.supplierOrderInfoID = supplierOrderInfoID;
    }

    public String getMasterAgreementIDInfoDateString() {
        return masterAgreementIDInfoDateString;
    }

    public String getMasterAgreementReferenceDateString() {
        return masterAgreementReferenceDateString;
    }

    public String getOrderIDInfoDateString() {
        return orderIDInfoDateString;
    }

    public void addInvoiceItem(ElectronicInvoiceItem electronicInvoiceItem) {
        invoiceItems.add(electronicInvoiceItem);
        // TODO: This is not the right place for sorting... Have to move this to getter method with some flag to avoid
        // this sorting whenever the getter is called
        invoiceItems.sort(Comparator.comparing(ElectronicInvoiceItem::getReferenceLineNumberInteger));
    }

    public ElectronicInvoiceItem[] getInvoiceItemsAsArray() {
        if (invoiceItems.size() > 0) {
            ElectronicInvoiceItem[] tempItems = new ElectronicInvoiceItem[invoiceItems.size()];
            invoiceItems.toArray(tempItems);
            return tempItems;
        }
        return null;
    }

    public String getOrderReferenceOrderDateString() {
        return orderReferenceOrderDateString;
    }

    public void setOrderReferenceOrderDateString(String orderReferenceOrderDateString) {
        this.orderReferenceOrderDateString = orderReferenceOrderDateString;
    }

    public String toString() {
        ToStringBuilder toString = new ToStringBuilder(this);
        toString.append("orderReferenceOrderID", getOrderReferenceOrderID());
        toString.append("orderReferenceOrderDate", getOrderReferenceOrderDateString());
        toString.append("orderReferenceDocumentRefPayloadID", getOrderReferenceDocumentRefPayloadID());
        toString.append("orderReferenceDocumentRef", getOrderReferenceDocumentRef());
        toString.append("masterAgreementReferenceID", getMasterAgreementReferenceID());
        toString.append("masterAgreementReferenceDateString", getMasterAgreementReferenceDateString());
        toString.append("masterAgreementIDInfoID", getMasterAgreementIDInfoID());
        toString.append("masterAgreementIDInfoDateString", getMasterAgreementIDInfoDateString());
        toString.append("orderIDInfoID", getOrderIDInfoID());
        toString.append("orderIDInfoDateString", getOrderIDInfoDateString());
        toString.append("supplierOrderInfoID", getSupplierOrderInfoID());
        toString.append("invoiceItems", getInvoiceItems());
        return toString.toString();
    }

}
