/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.impl.config.property.Config;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderItem;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderVendorQuote;
import org.kuali.kfs.module.purap.document.BulkReceivingDocument;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.module.purap.document.service.PrintService;
import org.kuali.kfs.module.purap.exception.PurError;
import org.kuali.kfs.module.purap.exception.PurapConfigurationException;
import org.kuali.kfs.module.purap.pdf.BulkReceivingPdf;
import org.kuali.kfs.module.purap.pdf.PurchaseOrderParameters;
import org.kuali.kfs.module.purap.pdf.PurchaseOrderPdf;
import org.kuali.kfs.module.purap.pdf.PurchaseOrderQuotePdf;
import org.kuali.kfs.module.purap.pdf.PurchaseOrderQuoteRequestsPdf;
import org.kuali.kfs.module.purap.pdf.PurchaseOrderTransmitParameters;
import org.kuali.kfs.module.purap.service.ImageService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.service.impl.KfsParameterConstants;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.springframework.transaction.annotation.Transactional;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Locale;

@Transactional
public class PrintServiceImpl implements PrintService {

    private static final Logger LOG = LogManager.getLogger();

    protected static final boolean TRANSMISSION_IS_RETRANSMIT = true;
    protected static final boolean TRANSMISSION_IS_NOT_RETRANSMIT = !TRANSMISSION_IS_RETRANSMIT;

    private ImageService imageService;
    private ParameterService parameterService;
    private ConfigurationService kualiConfigurationService;
    private PurchaseOrderParameters purchaseOrderParameters;

    @Override
    public Collection generatePurchaseOrderQuoteRequestsListPdf(PurchaseOrderDocument po,
            ByteArrayOutputStream byteArrayOutputStream) {
        LOG.debug("generatePurchaseOrderQuoteRequestsListPdf() started");
        Collection<String> errors = new ArrayList<>();

        PurchaseOrderQuoteRequestsPdf poQuoteRequestsPdf = new PurchaseOrderQuoteRequestsPdf();

        try {
            PurchaseOrderTransmitParameters pdfParameters = getPurchaseOrderQuoteRequestsListPdfParameters(po);
            poQuoteRequestsPdf.generatePOQuoteRequestsListPdf(po, byteArrayOutputStream,
                    pdfParameters.getCampusParameter().getPurchasingInstitutionName());

        } catch (PurError | Exception pe) {
            LOG.error("Caught exception ", pe);
            errors.add(pe.getMessage());
        }

        LOG.debug("generatePurchaseOrderQuoteRequestsListPdf() ended");
        return errors;
    }

    @Override
    public Collection savePurchaseOrderQuoteRequestsListPdf(PurchaseOrderDocument po) {
        return null;
    }

    /**
     * @param po The PurchaseOrderDocument object to be used to obtain the PurchaseOrderPdfParameters.
     * @return The PurchaseOrderPdfParameters given the PurchaseOrderDocument.
     */
    protected PurchaseOrderTransmitParameters getPurchaseOrderQuoteRequestsListPdfParameters(
            PurchaseOrderDocument po) {
        purchaseOrderParameters.setPurchaseOrderPdfParameters(po);
        return (PurchaseOrderTransmitParameters) purchaseOrderParameters;
    }

    @Override
    public Collection generatePurchaseOrderQuotePdf(PurchaseOrderDocument po, PurchaseOrderVendorQuote povq,
            ByteArrayOutputStream byteArrayOutputStream, String environment) {
        LOG.debug("generatePurchaseOrderQuotePdf() started");

        PurchaseOrderQuotePdf poQuotePdf = new PurchaseOrderQuotePdf();
        Collection<String> errors = new ArrayList<>();

        try {
            purchaseOrderParameters.setPurchaseOrderPdfParameters(po, povq);
            PurchaseOrderTransmitParameters pdfParameters = (PurchaseOrderTransmitParameters) purchaseOrderParameters;
            String deliveryCampusName = pdfParameters.getCampusParameter().getCampus().getName();
            poQuotePdf.generatePOQuotePDF(po, povq, deliveryCampusName, pdfParameters.getContractManagerCampusCode(),
                    pdfParameters.getLogoImage(), byteArrayOutputStream, environment);
        } catch (PurError | PurapConfigurationException pe) {
            LOG.error("Caught exception ", pe);
            errors.add(pe.getMessage());
        }
        LOG.debug("generatePurchaseOrderQuotePdf() ended");
        return errors;
    }

    @Override
    public Collection savePurchaseOrderQuotePdf(PurchaseOrderDocument po, PurchaseOrderVendorQuote povq,
            String environment) {
        LOG.debug("savePurchaseOrderQuotePdf() started");

        String pdfQuoteFilename = "PURAP_PO_" + po.getPurapDocumentIdentifier().toString() + "_Quote_" +
                povq.getPurchaseOrderVendorQuoteIdentifier().toString() + "_" + System.currentTimeMillis() + ".pdf";
        PurchaseOrderQuotePdf poQuotePdf = new PurchaseOrderQuotePdf();
        Collection<String> errors = new ArrayList<>();

        PurchaseOrderTransmitParameters pdfParameters = null;
        try {
            purchaseOrderParameters.setPurchaseOrderPdfParameters(po, povq);
            pdfParameters = (PurchaseOrderTransmitParameters) purchaseOrderParameters;
            poQuotePdf.savePOQuotePDF(po, povq, pdfParameters, environment);
        } catch (PurError | PurapConfigurationException e) {
            LOG.error("Caught exception ", e);
            errors.add(e.getMessage());
        } finally {
            try {
                poQuotePdf.deletePdf(pdfParameters.getPdfFileLocation(), pdfQuoteFilename);
            } catch (Throwable e) {
                LOG.error("savePurchaseOrderQuotePdf() Error deleting Quote PDF" + pdfQuoteFilename +
                        " - Exception was " + e.getMessage(), e);
                errors.add(e.getMessage());
            }
        }

        LOG.debug("savePurchaseOrderQuotePdf() ended");
        return errors;
    }

    /**
     * Creates purchase order pdf document given the input parameters.
     *
     * @param po                    The PurchaseOrderDocument.
     * @param byteArrayOutputStream ByteArrayOutputStream that the action is using, where the pdf will be printed to.
     * @param isRetransmit          boolean true if this is a retransmit purchase order document.
     * @param environment           The current environment used (e.g. DEV if it is a development environment).
     * @param retransmitItems       The items selected by the user to be retransmitted.
     * @return Collection of error strings.
     */
    protected Collection generatePurchaseOrderPdf(PurchaseOrderDocument po,
            ByteArrayOutputStream byteArrayOutputStream, boolean isRetransmit, String environment,
            List<PurchaseOrderItem> retransmitItems) {
        LOG.debug("generatePurchaseOrderPdf() started");

        PurchaseOrderPdf poPdf = new PurchaseOrderPdf();
        Collection<String> errors = new ArrayList<>();
        try {
            purchaseOrderParameters.setPurchaseOrderPdfParameters(po);
            PurchaseOrderTransmitParameters pdfParameters = (PurchaseOrderTransmitParameters) purchaseOrderParameters;
            poPdf.generatePdf(po, pdfParameters, byteArrayOutputStream, isRetransmit, environment, retransmitItems);
        } catch (PurError | PurapConfigurationException e) {
            LOG.error("Caught exception ", e);
            errors.add(e.getMessage());
        }

        LOG.debug("generatePurchaseOrderPdf() ended");
        return errors;
    }

    @Override
    public Collection generatePurchaseOrderPdf(PurchaseOrderDocument po, ByteArrayOutputStream byteArrayOutputStream,
            String environment, List<PurchaseOrderItem> retransmitItems) {
        return generatePurchaseOrderPdf(po, byteArrayOutputStream, TRANSMISSION_IS_NOT_RETRANSMIT, environment,
                retransmitItems);
    }

    @Override
    public Collection generatePurchaseOrderPdfForRetransmission(PurchaseOrderDocument po,
            ByteArrayOutputStream byteArrayOutputStream, String environment, List<PurchaseOrderItem> retransmitItems) {
        return generatePurchaseOrderPdf(po, byteArrayOutputStream, TRANSMISSION_IS_RETRANSMIT, environment,
                retransmitItems);
    }

    /**
     * Saves the purchase order pdf document.
     *
     * @param po           The PurchaseOrderDocument.
     * @param isRetransmit boolean true if this is a retransmit purchase order document.
     * @param environment  The current environment used (e.g. DEV if it is a development environment).
     * @return Collection of error strings.
     */
    protected Collection savePurchaseOrderPdf(PurchaseOrderDocument po, boolean isRetransmit, String environment) {
        LOG.debug("savePurchaseOrderPdf() started");

        PurchaseOrderPdf poPdf = new PurchaseOrderPdf();
        Collection<String> errors = new ArrayList<>();

        PurchaseOrderTransmitParameters pdfParameters = null;

        try {
            purchaseOrderParameters.setPurchaseOrderPdfParameters(po);
            pdfParameters = (PurchaseOrderTransmitParameters) purchaseOrderParameters;
            poPdf.savePdf(po, pdfParameters, isRetransmit, environment);
        } catch (PurError | PurapConfigurationException e) {
            LOG.error("Caught exception ", e);
            errors.add(e.getMessage());
        } finally {
            try {
                poPdf.deletePdf(pdfParameters.getPdfFileLocation(), pdfParameters.getPdfFileName());
            } catch (Throwable e) {
                LOG.error("savePurchaseOrderPdf() Error deleting PDF" + pdfParameters.getPdfFileName() +
                        " - Exception was " + e.getMessage(), e);
                errors.add("Error while deleting the pdf after savePurchaseOrderPdf" + e.getMessage());
            }
        }

        LOG.debug("savePurchaseOrderPdf() ended");
        return errors;
    }

    @Override
    public Collection savePurchaseOrderPdf(PurchaseOrderDocument po, String environment) {
        return savePurchaseOrderPdf(po, TRANSMISSION_IS_NOT_RETRANSMIT, environment);
    }

    @Override
    public Collection savePurchaseOrderPdfForRetransmission(PurchaseOrderDocument po, String environment) {
        return savePurchaseOrderPdf(po, TRANSMISSION_IS_RETRANSMIT, environment);
    }

    @Override
    public Collection generateBulkReceivingPDF(BulkReceivingDocument blkRecDoc, ByteArrayOutputStream baosPDF) {
        LOG.debug("generateBulkReceivingPDF() started");

        BulkReceivingPdf recBlkTicketPDF = new BulkReceivingPdf();
        Collection<String> errors = new ArrayList<>();

        String imageTempLocation;
        String logoImage = StringUtils.EMPTY;

        // key can be any string;
        String key = blkRecDoc.getDocumentNumber();
        String campusCode = blkRecDoc.getDeliveryCampusCode().toLowerCase(Locale.US);

        String environment = kualiConfigurationService.getPropertyValueAsString(Config.ENVIRONMENT);

        boolean useImage = true;
        if (parameterService.parameterExists(KfsParameterConstants.PURCHASING_DOCUMENT.class,
                PurapConstants.PDF_IMAGES_AVAILABLE_INDICATOR)) {
            useImage = parameterService.getParameterValueAsBoolean(KfsParameterConstants.PURCHASING_DOCUMENT.class,
                    PurapConstants.PDF_IMAGES_AVAILABLE_INDICATOR);
        }

        if (useImage) {
            imageTempLocation = kualiConfigurationService.getPropertyValueAsString(KFSConstants.TEMP_DIRECTORY_KEY) +
                    "/";

            if (imageTempLocation == null) {
                throw new PurapConfigurationException("IMAGE_TEMP_PATH is missing");
            }

            // Get images
            logoImage = imageService.getLogo(key, campusCode, imageTempLocation);

            if (StringUtils.isEmpty(logoImage)) {
                throw new PurapConfigurationException("logoImage is null.");
            }
        }

        try {
            recBlkTicketPDF.generatePdf(blkRecDoc, baosPDF, logoImage, environment);
        } catch (PurapConfigurationException pce) {
            LOG.error("Caught exception ", pce);
            errors.add(pce.getMessage());
        }

        LOG.debug("generateBulkReceivingPDF() ended");
        return errors;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setImageService(ImageService imageService) {
        this.imageService = imageService;
    }

    public void setConfigurationService(ConfigurationService kualiConfigurationService) {
        this.kualiConfigurationService = kualiConfigurationService;
    }

    public void setPurchaseOrderParameters(PurchaseOrderParameters purchaseOrderParameters) {
        this.purchaseOrderParameters = purchaseOrderParameters;
    }

}

