/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.kew.actiontaken.ActionTaken;
import org.kuali.kfs.kew.api.KewApiConstants;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteLevelChange;
import org.kuali.kfs.kew.framework.postprocessor.DocumentRouteStatusChange;
import org.kuali.kfs.kim.api.identity.PersonService;
import org.kuali.kfs.kim.api.services.KimApiServiceLocator;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.document.Copyable;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.PersistenceService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapKeyConstants;
import org.kuali.kfs.module.purap.PurapParameterConstants;
import org.kuali.kfs.module.purap.PurapWorkflowConstants;
import org.kuali.kfs.module.purap.RequisitionStatuses;
import org.kuali.kfs.module.purap.businessobject.BillingAddress;
import org.kuali.kfs.module.purap.businessobject.DefaultPrincipalAddress;
import org.kuali.kfs.module.purap.businessobject.PurApAccountingLine;
import org.kuali.kfs.module.purap.businessobject.PurchaseRequisitionItemUseTax;
import org.kuali.kfs.module.purap.businessobject.RequisitionAccount;
import org.kuali.kfs.module.purap.businessobject.RequisitionCapitalAssetItem;
import org.kuali.kfs.module.purap.businessobject.RequisitionCapitalAssetSystem;
import org.kuali.kfs.module.purap.businessobject.RequisitionItem;
import org.kuali.kfs.module.purap.businessobject.options.RequisitionStatusValuesFinder;
import org.kuali.kfs.module.purap.document.service.PurapService;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.module.purap.document.service.PurchasingDocumentSpecificService;
import org.kuali.kfs.module.purap.document.service.PurchasingService;
import org.kuali.kfs.module.purap.document.service.RequisitionService;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.Building;
import org.kuali.kfs.sys.businessobject.ChartOrgHolder;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySequenceHelper;
import org.kuali.kfs.sys.businessobject.GeneralLedgerPendingEntrySourceDetail;
import org.kuali.kfs.sys.context.SpringContext;
import org.kuali.kfs.sys.service.FinancialSystemUserService;
import org.kuali.kfs.sys.service.UniversityDateService;
import org.kuali.kfs.vnd.businessobject.VendorContract;
import org.kuali.kfs.vnd.businessobject.VendorDetail;
import org.kuali.kfs.vnd.document.service.VendorService;
import org.kuali.kfs.vnd.service.PhoneNumberService;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Document class for the Requisition.
 */
public class RequisitionDocument extends PurchasingDocumentBase implements Copyable {

    private static final Logger LOG = LogManager.getLogger();

    protected String requisitionOrganizationReference1Text;
    protected String requisitionOrganizationReference2Text;
    protected String requisitionOrganizationReference3Text;
    protected String alternate1VendorName;
    protected String alternate2VendorName;
    protected String alternate3VendorName;
    protected String alternate4VendorName;
    protected String alternate5VendorName;
    protected KualiDecimal organizationAutomaticPurchaseOrderLimit;
    protected List reqStatusList;

    // non-persistent property used for controlling validation for accounting lines when doc is request for blanket
    // approve.
    protected boolean isBlanketApproveRequest = false;
    private static final int ALLOW_REQS_UNLIMITED_COPY_DAYS = 9999;

    public RequisitionDocument() {
        super();
    }

    @Override
    public PurchasingDocumentSpecificService getDocumentSpecificService() {
        return SpringContext.getBean(RequisitionService.class);
    }

    /**
     * Provides answers to the following splits:
     * AmountRequiresSeparationOfDutiesReview
     */
    @Override
    public boolean answerSplitNodeQuestion(String nodeName) throws UnsupportedOperationException {
        if (nodeName.equals(PurapWorkflowConstants.HAS_ACCOUNTING_LINES)) {
            return !isMissingAccountingLines();
        }
        if (nodeName.equals(PurapWorkflowConstants.AMOUNT_REQUIRES_SEPARATION_OF_DUTIES_REVIEW_SPLIT)) {
            return isSeparationOfDutiesReviewRequired();
        }
        return super.answerSplitNodeQuestion(nodeName);
    }

    public boolean isMissingAccountingLines() {
        return CollectionUtils.isNotEmpty(getListOfItemsMissingAccountingLines());
    }

    public List<RequisitionItem> getListOfItemsMissingAccountingLines() {
        List<RequisitionItem> itemsWithMissingAccountingLines = new ArrayList<>();
        for (Object reqItem : getItems()) {
            RequisitionItem item = (RequisitionItem) reqItem;
            if (item.isConsideredEntered() && item.isAccountListEmpty()) {
                itemsWithMissingAccountingLines.add(item);
            }
        }

        return itemsWithMissingAccountingLines;
    }

    protected boolean isSeparationOfDutiesReviewRequired() {
        Set<Person> priorApprovers = this.getAllPriorApprovers();

        // If there are more than 1 prior approvers there is no need for SOD.
        // If 1 approver exists, check that approver is not the initiator.
        if (priorApprovers.size() > 0) {
            if (priorApprovers.size() > 1) {
                return false;
            } else {
                for (Person priorApprover : priorApprovers) {
                    String initiatorPrincipalId = this.getDocumentHeader().getWorkflowDocument()
                            .getInitiatorPrincipalId();
                    if (!initiatorPrincipalId.equals(priorApprover.getPrincipalId())) {
                        return false;
                    }
                }
            }
        }

        // If there was no prior approver or if the initiator and the approver are the same person,
        // then we have to check the amounts to determine whether to route to separation of duties.
        ParameterService parameterService = SpringContext.getBean(ParameterService.class);
        KualiDecimal maxAllowedAmount = new KualiDecimal(parameterService.getParameterValueAsString(
                RequisitionDocument.class, PurapParameterConstants.SEPARATION_OF_DUTIES_DOLLAR_AMOUNT));
        // if app param amount is greater than or equal to documentTotalAmount... no need for separation of duties
        KualiDecimal totalAmount = getDocumentHeader().getFinancialDocumentTotalAmount();
        return ObjectUtils.isNull(maxAllowedAmount) || ObjectUtils.isNull(totalAmount)
                || maxAllowedAmount.compareTo(totalAmount) < 0;
    }

    public Set<Person> getAllPriorApprovers() {
        PersonService personService = KimApiServiceLocator.getPersonService();
        List<ActionTaken> actionsTaken = this.getDocumentHeader().getWorkflowDocument()
                .getActionsTaken();
        Set<String> principalIds = new HashSet<>();
        Set<Person> persons = new HashSet<>();

        for (ActionTaken actionTaken : actionsTaken) {
            if (KewApiConstants.ACTION_TAKEN_APPROVED_CD.equals(actionTaken.getActionTaken())) {
                String principalId = actionTaken.getPrincipalId();
                if (!principalIds.contains(principalId)) {
                    principalIds.add(principalId);
                    persons.add(personService.getPerson(principalId));
                }
            }
        }
        return persons;
    }

    /**
     * Overrides the method in PurchasingAccountsPayableDocumentBase to add the criteria specific to Requisition
     * Document.
     */
    @Override
    public boolean isInquiryRendered() {
        return !isPostingYearPrior()
               || !getApplicationDocumentStatus().equals(RequisitionStatuses.APPDOC_CLOSED)
                 && !getApplicationDocumentStatus().equals(RequisitionStatuses.APPDOC_CANCELLED);
    }

    /**
     * Performs logic needed to initiate Requisition Document.
     */
    public void initiateDocument() {
        this.setupAccountDistributionMethod();
        this.setRequisitionSourceCode(PurapConstants.RequisitionSources.STANDARD_ORDER);
        updateAndSaveAppDocStatus(RequisitionStatuses.APPDOC_IN_PROCESS);
        this.setPurchaseOrderCostSourceCode(PurapConstants.POCostSources.ESTIMATE);
        this.setPurchaseOrderTransmissionMethodCode(determinePurchaseOrderTransmissionMethod());
        this.setUseTaxIndicator(SpringContext.getBean(PurchasingService.class).getDefaultUseTaxIndicatorValue(this));

        Person currentUser = GlobalVariables.getUserSession().getPerson();
        ChartOrgHolder purapChartOrg = SpringContext.getBean(FinancialSystemUserService.class)
                .getPrimaryOrganization(currentUser, PurapConstants.PURAP_NAMESPACE);
        if (ObjectUtils.isNotNull(purapChartOrg)) {
            this.setChartOfAccountsCode(purapChartOrg.getChartOfAccountsCode());
            this.setOrganizationCode(purapChartOrg.getOrganizationCode());
        }
        this.setDeliveryCampusCode(currentUser.getCampusCode());
        this.setDeliveryToName(currentUser.getName());
        this.setDeliveryToEmailAddress(currentUser.getEmailAddressUnmasked());
        this.setDeliveryToPhoneNumber(SpringContext.getBean(PhoneNumberService.class)
                .formatNumberIfPossible(currentUser.getPhoneNumber()));
        this.setRequestorPersonName(currentUser.getName());
        this.setRequestorPersonEmailAddress(currentUser.getEmailAddressUnmasked());
        this.setRequestorPersonPhoneNumber(SpringContext.getBean(PhoneNumberService.class)
                .formatNumberIfPossible(currentUser.getPhoneNumber()));

        DefaultPrincipalAddress defaultPrincipalAddress = new DefaultPrincipalAddress(currentUser.getPrincipalId());
        Map addressKeys = SpringContext.getBean(PersistenceService.class).getPrimaryKeyFieldValues(
                defaultPrincipalAddress);
        defaultPrincipalAddress = SpringContext.getBean(BusinessObjectService.class).findByPrimaryKey(
                DefaultPrincipalAddress.class, addressKeys);
        if (ObjectUtils.isNotNull(defaultPrincipalAddress)
                && ObjectUtils.isNotNull(defaultPrincipalAddress.getBuilding())) {
            if (defaultPrincipalAddress.getBuilding().isActive()) {
                this.setDeliveryCampusCode(defaultPrincipalAddress.getCampusCode());
                this.templateBuildingToDeliveryAddress(defaultPrincipalAddress.getBuilding());
                this.setDeliveryBuildingRoomNumber(defaultPrincipalAddress.getBuildingRoomNumber());
            } else {
                //since building is now inactive, delete default building record
                SpringContext.getBean(BusinessObjectService.class).delete(defaultPrincipalAddress);
            }
        }

        // set the APO limit
        this.setOrganizationAutomaticPurchaseOrderLimit(SpringContext.getBean(PurapService.class).getApoLimit(
                this.getVendorContractGeneratedIdentifier(), this.getChartOfAccountsCode(),
                this.getOrganizationCode()));

        // populate billing address
        BillingAddress billingAddress = SpringContext.getBean(BusinessObjectService.class).findBySinglePrimaryKey(
                BillingAddress.class, getDeliveryCampusCode());
        this.templateBillingAddress(billingAddress);

        // populate receiving address with the default one for the chart/org
        loadReceivingAddress();

        // Load Requisition Statuses
        RequisitionStatusValuesFinder requisitionStatusValuesFinder = new RequisitionStatusValuesFinder();
        reqStatusList = requisitionStatusValuesFinder.getKeyValues();

        SpringContext.getBean(PurapService.class).addBelowLineItems(this);
        this.refreshNonUpdateableReferences();
    }

    public void templateBuildingToDeliveryAddress(Building building) {
        if (ObjectUtils.isNotNull(building)) {
            setDeliveryBuildingCode(building.getBuildingCode());
            setDeliveryBuildingName(building.getBuildingName());
            setDeliveryBuildingLine1Address(building.getBuildingStreetAddress());
            setDeliveryCityName(building.getBuildingAddressCityName());
            setDeliveryStateCode(building.getBuildingAddressStateCode());
            setDeliveryPostalCode(building.getBuildingAddressZipCode());
            setDeliveryCountryCode(building.getBuildingAddressCountryCode());
        }
    }

    /**
     * @return the PO PO transmission method to use.
     */
    protected String determinePurchaseOrderTransmissionMethod() {
        return SpringContext.getBean(ParameterService.class).getParameterValueAsString(RequisitionDocument.class,
                PurapParameterConstants.PURAP_DEFAULT_PO_TRANSMISSION_CODE);
    }

    /**
     * Checks whether copying of this document should be allowed. Copying is not allowed if this is a B2B requisition,
     * and more than a set number of days have passed since the document's creation.
     *
     * @return True if copying of this requisition is allowed.
     */
    @Override
    public boolean getAllowsCopy() {
        boolean allowsCopy = super.getAllowsCopy();
        Integer allowedCopyDays = getRequisitionSource().getAllowCopyDays();

        if (allowedCopyDays == 0) {
            return false;
        } else if (allowedCopyDays >= ALLOW_REQS_UNLIMITED_COPY_DAYS) {
            return true;
        } else if (allowedCopyDays > 0) {
            DateTimeService dateTimeService = SpringContext.getBean(DateTimeService.class);
            Calendar c = Calendar.getInstance();
            // The allowed copy date is the document creation date plus a set number of days.
            DateTime createDate = this.getDocumentHeader().getWorkflowDocument().getDateCreated();
            c.setTime(createDate.toDate());
            c.add(Calendar.DATE, allowedCopyDays);
            Date allowedCopyDate = c.getTime();
            Date currentDate = dateTimeService.getCurrentDate();
            // Return true if the current time is before the allowed copy date.
            allowsCopy = currentDate.before(allowedCopyDate);
        }

        return allowsCopy;
    }

    /**
     * Performs logic needed to copy Requisition Document.
     */
    @Override
    public void toCopy() throws ValidationException {
        super.toCopy();

        // Clear related views
        this.setAccountsPayablePurchasingDocumentLinkIdentifier(null);
        this.setRelatedViews(null);

        Person currentUser = GlobalVariables.getUserSession().getPerson();
        ChartOrgHolder purapChartOrg = SpringContext.getBean(FinancialSystemUserService.class)
                .getPrimaryOrganization(currentUser, PurapConstants.PURAP_NAMESPACE);
        this.setPurapDocumentIdentifier(null);

        // Set req status to INPR. for app doc status
        updateAndSaveAppDocStatus(RequisitionStatuses.APPDOC_IN_PROCESS);

        // Set fields from the user.
        if (ObjectUtils.isNotNull(purapChartOrg)) {
            this.setChartOfAccountsCode(purapChartOrg.getChartOfAccountsCode());
            this.setOrganizationCode(purapChartOrg.getOrganizationCode());
        }
        this.setPostingYear(SpringContext.getBean(UniversityDateService.class).getCurrentFiscalYear());

        boolean activeVendor = true;
        boolean activeContract = true;
        Date today = SpringContext.getBean(DateTimeService.class).getCurrentDate();
        VendorContract vendorContract = new VendorContract();
        vendorContract.setVendorContractGeneratedIdentifier(this.getVendorContractGeneratedIdentifier());
        Map keys = SpringContext.getBean(PersistenceService.class).getPrimaryKeyFieldValues(vendorContract);
        vendorContract = SpringContext.getBean(BusinessObjectService.class).findByPrimaryKey(VendorContract.class,
                keys);
        if (!(vendorContract != null && today.after(vendorContract.getVendorContractBeginningDate())
                && today.before(vendorContract.getVendorContractEndDate()))) {
            activeContract = false;
        }

        VendorDetail vendorDetail = SpringContext.getBean(VendorService.class).getVendorDetail(
                this.getVendorHeaderGeneratedIdentifier(), this.getVendorDetailAssignedIdentifier());
        if (!(vendorDetail != null && vendorDetail.isActiveIndicator())) {
            activeVendor = false;
        }

        // B2B - only copy if contract and vendor are both active (throw separate errors to print to screen)
        if (this.getRequisitionSourceCode().equals(PurapConstants.RequisitionSources.B2B)) {
            if (!activeContract) {
                throw new ValidationException(PurapKeyConstants.ERROR_REQ_COPY_EXPIRED_CONTRACT);
            }
            if (!activeVendor) {
                throw new ValidationException(PurapKeyConstants.ERROR_REQ_COPY_INACTIVE_VENDOR);
            }
        }

        if (!activeVendor) {
            this.setVendorContractGeneratedIdentifier(null);
        }
        if (!activeContract) {
            this.setVendorContractGeneratedIdentifier(null);
        }

        // These fields should not be set in this method; force to be null
        this.setOrganizationAutomaticPurchaseOrderLimit(null);
        this.setPurchaseOrderAutomaticIndicator(false);

        for (Object anItem : this.getItems()) {
            RequisitionItem item = (RequisitionItem) anItem;
            item.setPurapDocumentIdentifier(null);
            item.setItemIdentifier(null);

            for (PurApAccountingLine purApAccountingLine : item.getSourceAccountingLines()) {
                RequisitionAccount account = (RequisitionAccount) purApAccountingLine;
                account.setAccountIdentifier(null);
                account.setItemIdentifier(null);
                account.setObjectId(null);
                account.setVersionNumber(null);
            }
        }

        if (!PurapConstants.RequisitionSources.B2B.equals(this.getRequisitionSourceCode())) {
            SpringContext.getBean(PurapService.class).addBelowLineItems(this);
        }
        this.setOrganizationAutomaticPurchaseOrderLimit(SpringContext.getBean(PurapService.class).getApoLimit(
                this.getVendorContractGeneratedIdentifier(), this.getChartOfAccountsCode(),
                this.getOrganizationCode()));
        clearCapitalAssetFields();
        SpringContext.getBean(PurapService.class).clearTax(this, this.isUseTaxIndicator());

        this.refreshNonUpdateableReferences();
    }

    @Override
    public List<String> getWorkflowEngineDocumentIdsToLock() {
        List<String> docIdStrings = new ArrayList<>();
        docIdStrings.add(getDocumentNumber());
        return docIdStrings;
    }

    @Override
    public void doRouteStatusChange(DocumentRouteStatusChange statusChangeEvent) {
        LOG.debug("doRouteStatusChange() started");
        super.doRouteStatusChange(statusChangeEvent);

        // DOCUMENT PROCESSED
        if (this.getDocumentHeader().getWorkflowDocument().isProcessed()) {
            String newRequisitionStatus = RequisitionStatuses.APPDOC_AWAIT_CONTRACT_MANAGER_ASSGN;
            if (SpringContext.getBean(RequisitionService.class).isAutomaticPurchaseOrderAllowed(this)) {
                newRequisitionStatus = RequisitionStatuses.APPDOC_CLOSED;
                SpringContext.getBean(PurchaseOrderService.class).createAutomaticPurchaseOrderDocument(this);
            }
            // for app doc status
            String reqStatus = getRequisitionAppDocStatus(newRequisitionStatus);
            updateAndSaveAppDocStatus(reqStatus);
        } else if (this.getDocumentHeader().getWorkflowDocument().isDisapproved()) {
            // DOCUMENT DISAPPROVED
            String nodeName = SpringContext.getBean(WorkflowDocumentService.class).getCurrentRouteLevelName(
                    this.getDocumentHeader().getWorkflowDocument());
            String disapprovalStatus = getRequisitionAppDocStatus(nodeName);

            if (StringUtils.isNotBlank(disapprovalStatus)) {
                updateAndSaveAppDocStatus(disapprovalStatus);
            } else {
                logAndThrowRuntimeException("No status found to set for document being disapproved in node '" +
                        nodeName + "'");
            }
        } else if (this.getDocumentHeader().getWorkflowDocument().isCanceled()) {
            // DOCUMENT CANCELED
            String reqStatus = getRequisitionAppDocStatus(RequisitionStatuses.APPDOC_CANCELLED);
            updateAndSaveAppDocStatus(reqStatus);
        }
        LOG.debug("doRouteStatusChange() ending");
    }

    protected String getRequisitionAppDocStatus(String nodeName) {
        return RequisitionStatuses.getRequistionAppDocStatuses().get(nodeName);
    }

    @Override
    public void doRouteLevelChange(DocumentRouteLevelChange change) {
        LOG.debug("handleRouteLevelChange() started");
        super.doRouteLevelChange(change);
    }

    @Override
    public Class getSourceAccountingLineClass() {
        //NOTE: do not do anything with this method as it is used by routing etc!
        return super.getSourceAccountingLineClass();
    }

    public String getRequisitionOrganizationReference1Text() {
        return requisitionOrganizationReference1Text;
    }

    public void setRequisitionOrganizationReference1Text(String requisitionOrganizationReference1Text) {
        this.requisitionOrganizationReference1Text = requisitionOrganizationReference1Text;
    }

    public String getRequisitionOrganizationReference2Text() {
        return requisitionOrganizationReference2Text;
    }

    public void setRequisitionOrganizationReference2Text(String requisitionOrganizationReference2Text) {
        this.requisitionOrganizationReference2Text = requisitionOrganizationReference2Text;
    }

    public String getRequisitionOrganizationReference3Text() {
        return requisitionOrganizationReference3Text;
    }

    public void setRequisitionOrganizationReference3Text(String requisitionOrganizationReference3Text) {
        this.requisitionOrganizationReference3Text = requisitionOrganizationReference3Text;
    }

    public String getAlternate1VendorName() {
        return alternate1VendorName;
    }

    public void setAlternate1VendorName(String alternate1VendorName) {
        this.alternate1VendorName = alternate1VendorName;
    }

    public String getAlternate2VendorName() {
        return alternate2VendorName;
    }

    public void setAlternate2VendorName(String alternate2VendorName) {
        this.alternate2VendorName = alternate2VendorName;
    }

    public String getAlternate3VendorName() {
        return alternate3VendorName;
    }

    public void setAlternate3VendorName(String alternate3VendorName) {
        this.alternate3VendorName = alternate3VendorName;
    }

    public String getAlternate4VendorName() {
        return alternate4VendorName;
    }

    public void setAlternate4VendorName(String alternate4VendorName) {
        this.alternate4VendorName = alternate4VendorName;
    }

    public String getAlternate5VendorName() {
        return alternate5VendorName;
    }

    public void setAlternate5VendorName(String alternate5VendorName) {
        this.alternate5VendorName = alternate5VendorName;
    }

    public KualiDecimal getOrganizationAutomaticPurchaseOrderLimit() {
        return organizationAutomaticPurchaseOrderLimit;
    }

    public void setOrganizationAutomaticPurchaseOrderLimit(KualiDecimal organizationAutomaticPurchaseOrderLimit) {
        this.organizationAutomaticPurchaseOrderLimit = organizationAutomaticPurchaseOrderLimit;
    }

    @Override
    public Class getItemClass() {
        return RequisitionItem.class;
    }

    @Override
    public Class getItemUseTaxClass() {
        return PurchaseRequisitionItemUseTax.class;
    }

    /**
     * @return null as requisition has no source document.
     */
    @Override
    public PurchasingAccountsPayableDocument getPurApSourceDocumentIfPossible() {
        return null;
    }

    /**
     * @return null as requisition has no source document.
     */
    @Override
    public String getPurApSourceDocumentLabelIfPossible() {
        return null;
    }

    @Override
    public String getDocumentTitle() {
        String title;
        if (SpringContext.getBean(ParameterService.class).getParameterValueAsBoolean(RequisitionDocument.class,
                PurapParameterConstants.PURAP_OVERRIDE_REQ_DOC_TITLE)) {
            String docIdStr = "";
            if (this.getPurapDocumentIdentifier() != null
                    && StringUtils.isNotBlank(this.getPurapDocumentIdentifier().toString())) {
                docIdStr = "Requisition: " + this.getPurapDocumentIdentifier().toString();
            }
            String chartAcct = this.getFirstChartAccount();
            String chartAcctStr = chartAcct == null ? "" : " - Account Number:  " + chartAcct;
            title = docIdStr + chartAcctStr;
        } else {
            title = super.getDocumentTitle();
        }
        return title;
    }

    /**
     * Gets this requisition's Chart/Account of the first accounting line from the first item.
     *
     * @return The first Chart and Account, or an empty string if there is none.
     */
    protected String getFirstChartAccount() {
        String chartAcct = null;
        RequisitionItem item = (RequisitionItem) this.getItem(0);
        if (ObjectUtils.isNotNull(item)) {
            if (item.getSourceAccountingLines().size() > 0) {
                PurApAccountingLine accountLine = item.getSourceAccountingLine(0);
                if (ObjectUtils.isNotNull(accountLine) && ObjectUtils.isNotNull(accountLine.getChartOfAccountsCode())
                        && ObjectUtils.isNotNull(accountLine.getAccountNumber())) {
                    chartAcct = accountLine.getChartOfAccountsCode() + "-" + accountLine.getAccountNumber();
                }
            }
        }
        return chartAcct;
    }

    public Date getCreateDate() {
        return this.getDocumentHeader().getWorkflowDocument().getDateCreated().toDate();
    }

    public String getUrl() {
        return SpringContext.getBean(ConfigurationService.class).getPropertyValueAsString(
                KFSConstants.APPLICATION_URL_KEY) + "/DocHandler.do?docId=" + getDocumentNumber() +
                "&command=displayDocSearchView";
    }

    /**
     * This is a "do nothing" version of the method - it just won't create GLPEs
     */
    @Override
    public boolean generateGeneralLedgerPendingEntries(GeneralLedgerPendingEntrySourceDetail glpeSourceDetail,
            GeneralLedgerPendingEntrySequenceHelper sequenceHelper) {
        return true;
    }

    @Override
    public Class getPurchasingCapitalAssetItemClass() {
        return RequisitionCapitalAssetItem.class;
    }

    @Override
    public Class getPurchasingCapitalAssetSystemClass() {
        return RequisitionCapitalAssetSystem.class;
    }

    @Override
    public boolean shouldGiveErrorForEmptyAccountsProration() {
        return !isDocumentStoppedInRouteNode(RequisitionStatuses.NODE_CONTENT_REVIEW)
                && !getApplicationDocumentStatus().equals(RequisitionStatuses.APPDOC_IN_PROCESS);
    }

    public Date getCreateDateForResult() {
        return getCreateDate();
    }

    public boolean isBlanketApproveRequest() {
        return isBlanketApproveRequest;
    }

    public void setBlanketApproveRequest(boolean isBlanketApproveRequest) {
        this.isBlanketApproveRequest = isBlanketApproveRequest;
    }

    /**
     * retrieves the system parameter value for account distribution method and determines if the drop-down box on the
     * form should be read only or not. Sets the default value for account distribution method property on the
     * document.
     */
    public void setupAccountDistributionMethod() {
        setAccountDistributionMethod(PurapConstants.AccountDistributionMethodCodes.PROPORTIONAL_CODE);
    }
}
