/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2022 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.document.service.impl;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.config.property.ConfigurationService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.datadictionary.legacy.DataDictionaryService;
import org.kuali.kfs.kew.api.document.WorkflowDocumentService;
import org.kuali.kfs.kim.impl.identity.Person;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.exception.ValidationException;
import org.kuali.kfs.krad.service.BusinessObjectService;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.krad.util.GlobalVariables;
import org.kuali.kfs.krad.util.KRADPropertyConstants;
import org.kuali.kfs.krad.util.ObjectUtils;
import org.kuali.kfs.module.purap.CreditMemoStatuses;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapKeyConstants;
import org.kuali.kfs.module.purap.PurapParameterConstants;
import org.kuali.kfs.module.purap.PurapPropertyConstants;
import org.kuali.kfs.module.purap.PurchaseOrderStatuses;
import org.kuali.kfs.module.purap.businessobject.CreditMemoAccount;
import org.kuali.kfs.module.purap.businessobject.CreditMemoItem;
import org.kuali.kfs.module.purap.businessobject.PaymentRequestItem;
import org.kuali.kfs.module.purap.businessobject.PurApAccountingLine;
import org.kuali.kfs.module.purap.businessobject.PurchaseOrderItem;
import org.kuali.kfs.module.purap.businessobject.PurchasingCapitalAssetItem;
import org.kuali.kfs.module.purap.document.AccountsPayableDocument;
import org.kuali.kfs.module.purap.document.PaymentRequestDocument;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.module.purap.document.PurchasingAccountsPayableDocument;
import org.kuali.kfs.module.purap.document.VendorCreditMemoDocument;
import org.kuali.kfs.module.purap.document.dataaccess.CreditMemoDao;
import org.kuali.kfs.module.purap.document.service.AccountsPayableService;
import org.kuali.kfs.module.purap.document.service.CreditMemoService;
import org.kuali.kfs.module.purap.document.service.PaymentRequestService;
import org.kuali.kfs.module.purap.document.service.PurapService;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.module.purap.document.validation.event.AttributedContinuePurapEvent;
import org.kuali.kfs.module.purap.service.PurapAccountingService;
import org.kuali.kfs.module.purap.service.PurapGeneralLedgerService;
import org.kuali.kfs.module.purap.util.ExpiredOrClosedAccountEntry;
import org.kuali.kfs.module.purap.util.VendorGroupingHelper;
import org.kuali.kfs.sys.KFSConstants;
import org.kuali.kfs.sys.businessobject.Bank;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.businessobject.PaymentMethod;
import org.kuali.kfs.sys.businessobject.SourceAccountingLine;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.kuali.kfs.sys.service.BankService;
import org.kuali.kfs.vnd.VendorConstants;
import org.kuali.kfs.vnd.VendorUtils;
import org.kuali.kfs.vnd.businessobject.VendorAddress;
import org.kuali.kfs.vnd.businessobject.VendorDetail;
import org.kuali.kfs.vnd.document.service.VendorService;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Provides services to support the creation of a Credit Memo Document.
 */
@Transactional
public class CreditMemoServiceImpl implements CreditMemoService {

    private static final Logger LOG = LogManager.getLogger();

    protected AccountsPayableService accountsPayableService;
    protected CreditMemoDao creditMemoDao;
    protected DataDictionaryService dataDictionaryService;
    protected DocumentService documentService;
    protected ConfigurationService kualiConfigurationService;
    protected NoteService noteService;
    protected PaymentRequestService paymentRequestService;
    protected PurapAccountingService purapAccountingService;
    protected PurapGeneralLedgerService purapGeneralLedgerService;
    protected PurapService purapService;
    protected PurchaseOrderService purchaseOrderService;
    protected VendorService vendorService;
    protected WorkflowDocumentService workflowDocumentService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    private BankService bankService;
    private ParameterService parameterService;
    private BusinessObjectService businessObjectService;

    @Override
    public List<VendorCreditMemoDocument> getCreditMemosToExtract(final String campusCode) {
        LOG.debug("getCreditMemosToExtract() started");

        List<VendorCreditMemoDocument> docs = creditMemoDao.getCreditMemosToExtract(campusCode);
        docs = (List<VendorCreditMemoDocument>) filterCreditMemoByAppDocStatus(docs,
                CreditMemoStatuses.STATUSES_ALLOWED_FOR_EXTRACTION);

        return docs;

    }

    @Override
    public Collection<VendorCreditMemoDocument> getCreditMemosToExtractByVendor(
            final String campusCode,
            final VendorGroupingHelper vendor) {
        LOG.debug("getCreditMemosToExtractByVendor() started");

        Collection<VendorCreditMemoDocument> docs = creditMemoDao.getCreditMemosToExtractByVendor(campusCode, vendor);
        docs = filterCreditMemoByAppDocStatus(docs, CreditMemoStatuses.STATUSES_ALLOWED_FOR_EXTRACTION);
        return docs;

    }

    @Override
    public Set<VendorGroupingHelper> getVendorsOnCreditMemosToExtract(final String campusCode) {
        LOG.debug("getVendorsOnCreditMemosToExtract() started");
        final HashSet<VendorGroupingHelper> vendors = new HashSet<>();

        final List<VendorCreditMemoDocument> docs = this.getCreditMemosToExtract(campusCode);
        for (final VendorCreditMemoDocument doc : docs) {
            vendors.add(new VendorGroupingHelper(doc));
        }

        return vendors;
    }

    /**
     * This method queries documentHeader and filter credit memos against the provided status.
     *
     * @param creditMemoDocuments
     * @param appDocStatus
     * @return
     */
    protected Collection<VendorCreditMemoDocument> filterCreditMemoByAppDocStatus(
            final Collection<VendorCreditMemoDocument> creditMemoDocuments,
            final String... appDocStatus) {
        final List<String> appDocStatusList = Arrays.asList(appDocStatus);
        final Collection<VendorCreditMemoDocument> filteredCreditMemoDocuments = new ArrayList<>();
        //add to filtered collection if the app doc list contains credit memo's app doc status
        for (final VendorCreditMemoDocument creditMemo : creditMemoDocuments) {
            if (appDocStatusList.contains(creditMemo.getApplicationDocumentStatus())) {
                filteredCreditMemoDocuments.add(creditMemo);
            }
        }
        return filteredCreditMemoDocuments;
    }

    @Override
    public String creditMemoDuplicateMessages(final VendorCreditMemoDocument cmDocument) {
        String duplicateMessage = null;

        String vendorNumber = cmDocument.getVendorNumber();
        if (StringUtils.isEmpty(vendorNumber)) {
            final PurchasingAccountsPayableDocument sourceDocument = cmDocument.getPurApSourceDocumentIfPossible();
            if (ObjectUtils.isNotNull(sourceDocument)) {
                vendorNumber = sourceDocument.getVendorNumber();
            }
        }

        if (StringUtils.isNotEmpty(vendorNumber)) {
            // check for existence of another credit memo with the same vendor and vendor credit memo number
            if (creditMemoDao.duplicateExists(VendorUtils.getVendorHeaderId(vendorNumber),
                    VendorUtils.getVendorDetailId(vendorNumber), cmDocument.getCreditMemoNumber())) {
                duplicateMessage = kualiConfigurationService.getPropertyValueAsString(
                        PurapKeyConstants.MESSAGE_DUPLICATE_CREDIT_MEMO_VENDOR_NUMBER);
            }

            // check for existence of another credit memo with the same vendor and credit memo date
            if (creditMemoDao.duplicateExists(VendorUtils.getVendorHeaderId(vendorNumber),
                    VendorUtils.getVendorDetailId(vendorNumber), cmDocument.getCreditMemoDate(),
                    cmDocument.getCreditMemoAmount())) {
                duplicateMessage = kualiConfigurationService.getPropertyValueAsString(
                        PurapKeyConstants.MESSAGE_DUPLICATE_CREDIT_MEMO_VENDOR_NUMBER_DATE_AMOUNT);
            }
        }

        return duplicateMessage;
    }

    @Override
    public List<PurchaseOrderItem> getPOInvoicedItems(final PurchaseOrderDocument poDocument) {
        final List<PurchaseOrderItem> invoicedItems = new ArrayList<>();

        for (final Object item : poDocument.getItems()) {
            final PurchaseOrderItem poItem = (PurchaseOrderItem) item;

            poItem.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (poItem.getItemType().isQuantityBasedGeneralLedgerIndicator()
                    && poItem.getItemInvoicedTotalQuantity().isGreaterThan(KualiDecimal.ZERO)) {
                invoicedItems.add(poItem);
            } else {
                final BigDecimal unitPrice = poItem.getItemUnitPrice() == null ? new BigDecimal(0) :
                        poItem.getItemUnitPrice();
                if (unitPrice.doubleValue() > poItem.getItemOutstandingEncumberedAmount().doubleValue()) {
                    invoicedItems.add(poItem);
                }
            }
        }

        return invoicedItems;
    }

    @Override
    public void calculateCreditMemo(final VendorCreditMemoDocument cmDocument) {
        cmDocument.updateExtendedPriceOnItems();

        for (final CreditMemoItem item : (List<CreditMemoItem>) cmDocument.getItems()) {
            // make sure restocking fee is negative
            if (StringUtils.equals(PurapConstants.ItemTypeCodes.ITEM_TYPE_RESTCK_FEE_CODE, item.getItemTypeCode())) {
                if (item.getItemUnitPrice() != null) {
                    item.setExtendedPrice(item.getExtendedPrice().abs().negated());
                    item.setItemUnitPrice(item.getItemUnitPrice().abs().negate());
                }
            }
        }

        //calculate tax if cm not based on vendor
        if (!cmDocument.isSourceVendor()) {
            purapService.calculateTax(cmDocument);
        }

        // proration
        if (cmDocument.isSourceVendor()) {
            // no proration on vendor
            return;
        }

        for (final CreditMemoItem item : (List<CreditMemoItem>) cmDocument.getItems()) {
            // skip above the line
            item.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (item.getItemType().isLineItemIndicator()) {
                continue;
            }

            if (item.getSourceAccountingLines().isEmpty() && ObjectUtils.isNotNull(item.getExtendedPrice())
                    && KualiDecimal.ZERO.compareTo(item.getExtendedPrice()) != 0) {

                final KualiDecimal totalAmount = cmDocument.getPurApSourceDocumentIfPossible().getTotalDollarAmount();

                // this should do nothing on preq which is fine
                purapAccountingService.updateAccountAmounts(cmDocument.getPurApSourceDocumentIfPossible());
                final List<SourceAccountingLine> summaryAccounts = purapAccountingService.generateSummary(
                        cmDocument.getPurApSourceDocumentIfPossible().getItems());
                final List<PurApAccountingLine> distributedAccounts =
                        purapAccountingService.generateAccountDistributionForProration(summaryAccounts, totalAmount,
                                PurapConstants.PRORATION_SCALE, CreditMemoAccount.class);

                if (CollectionUtils.isNotEmpty(distributedAccounts)
                        && CollectionUtils.isEmpty(item.getSourceAccountingLines())) {
                    item.setSourceAccountingLines(distributedAccounts);
                }
            }
        }
    }

    @Override
    public VendorCreditMemoDocument getCreditMemoByDocumentNumber(final String documentNumber) {
        LOG.debug("getCreditMemoByDocumentNumber() started");

        if (ObjectUtils.isNotNull(documentNumber)) {
            return (VendorCreditMemoDocument) documentService.getByDocumentHeaderId(documentNumber);
        }
        return null;
    }

    @Override
    public VendorCreditMemoDocument getCreditMemoDocumentById(final Integer purchasingDocumentIdentifier) {
        return getCreditMemoByDocumentNumber(creditMemoDao.getDocumentNumberByCreditMemoId(
                purchasingDocumentIdentifier));
    }

    @Override
    public void populateAndSaveCreditMemo(final VendorCreditMemoDocument document) {
        try {
            document.updateAndSaveAppDocStatus(CreditMemoStatuses.APPDOC_IN_PROCESS);

            if (document.isSourceDocumentPaymentRequest()) {
                document.setBankCode(document.getPaymentRequestDocument().getBankCode());
                document.setBank(document.getPaymentRequestDocument().getBank());
            } else {
                // set bank code to default bank code in the system parameter
                final Bank defaultBank = getDefaultBank(document);
                if (defaultBank != null) {
                    document.setBankCode(defaultBank.getBankCode());
                    document.setBank(defaultBank);
                }
            }

            final PaymentMethod paymentMethod = businessObjectService.findBySinglePrimaryKey(
                    PaymentMethod.class,
                    KFSConstants.PaymentSourceConstants.PAYMENT_METHOD_CHECK);
            if (paymentMethod != null) {
                document.setPaymentMethodCode(paymentMethod.getPaymentMethodCode());
                document.setPaymentMethod(paymentMethod);

                if (paymentMethod.isActive() && paymentMethod.getBank() != null) {
                    document.setBankCode(paymentMethod.getBankCode());
                    document.setBank(paymentMethod.getBank());
                }
            }

            documentService.saveDocument(document, AttributedContinuePurapEvent.class);
        } catch (final ValidationException ve) {
            // set the status back to initiate
            document.updateAndSaveAppDocStatus(CreditMemoStatuses.APPDOC_INITIATE);
        }
    }

    protected Bank getDefaultBank(final VendorCreditMemoDocument document) {
        return bankService.getDefaultBankByDocType(document.getClass());
    }

    @Override
    public void reopenClosedPO(final VendorCreditMemoDocument cmDocument) {
        // reopen PO if closed
        Integer purchaseOrderDocumentId = cmDocument.getPurchaseOrderIdentifier();
        if (cmDocument.isSourceDocumentPaymentRequest() && ObjectUtils.isNull(purchaseOrderDocumentId)) {
            final PaymentRequestDocument paymentRequestDocument = paymentRequestService.getPaymentRequestById(
                    cmDocument.getPaymentRequestIdentifier());
            purchaseOrderDocumentId = paymentRequestDocument.getPurchaseOrderIdentifier();
        }
    }

    @Override
    public VendorCreditMemoDocument addHoldOnCreditMemo(
            final VendorCreditMemoDocument cmDocument,
            final String note)
            throws Exception {
        // save the note
        final Note noteObj = documentService.createNoteFromDocument(cmDocument, note);
        cmDocument.addNote(noteObj);
        noteService.save(noteObj);

        // retrieve and save with hold indicator set to true
        final VendorCreditMemoDocument cmDoc = getCreditMemoDocumentById(cmDocument.getPurapDocumentIdentifier());
        cmDoc.setHoldIndicator(true);
        cmDoc.setLastActionPerformedByPersonId(GlobalVariables.getUserSession().getPerson().getPrincipalId());
        purapService.saveDocumentNoValidation(cmDoc);

        // must also save it on the incoming document
        cmDocument.setHoldIndicator(true);
        cmDocument.setLastActionPerformedByPersonId(GlobalVariables.getUserSession().getPerson().getPrincipalId());

        return cmDoc;
    }

    @Override
    public VendorCreditMemoDocument removeHoldOnCreditMemo(
            final VendorCreditMemoDocument cmDocument,
            final String note)
            throws Exception {
        // save the note
        final Note noteObj = documentService.createNoteFromDocument(cmDocument, note);
        cmDocument.addNote(noteObj);
        noteService.save(noteObj);

        // retrieve and save with hold indicator set to false
        final VendorCreditMemoDocument cmDoc = getCreditMemoDocumentById(cmDocument.getPurapDocumentIdentifier());
        cmDoc.setHoldIndicator(false);
        cmDoc.setLastActionPerformedByPersonId(null);
        purapService.saveDocumentNoValidation(cmDoc);

        // must also save it on the incoming document
        cmDocument.setHoldIndicator(false);
        cmDocument.setLastActionPerformedByPersonId(null);

        return cmDoc;
    }

    @Override
    public String updateStatusByNode(final String currentNodeName, final AccountsPayableDocument apDoc) {
        return updateStatusByNode(currentNodeName, (VendorCreditMemoDocument) apDoc);
    }

    /**
     * Updates the status of a credit memo document, currently this is used by the cancel action
     *
     * @param currentNodeName The string representing the current node to be used to obtain the canceled status code.
     * @param cmDoc           The credit memo document to be updated.
     * @return The string representing the canceledStatusCode, if empty it is assumed to be not from workflow.
     */
    protected String updateStatusByNode(final String currentNodeName, final VendorCreditMemoDocument cmDoc) {
        // update the status on the document

        final String cancelledStatusCode;
        if (StringUtils.isEmpty(currentNodeName)) {
            cancelledStatusCode = CreditMemoStatuses.APPDOC_CANCELLED_POST_AP_APPROVE;
        } else {
            cancelledStatusCode = CreditMemoStatuses.getCreditMemoAppDocDisapproveStatuses().get(currentNodeName);
        }

        if (StringUtils.isNotBlank(cancelledStatusCode)) {
            cmDoc.updateAndSaveAppDocStatus(cancelledStatusCode);
            purapService.saveDocumentNoValidation(cmDoc);
            return cancelledStatusCode;
        } else {
            logAndThrowRuntimeException("No status found to set for document being disapproved in node '" +
                    currentNodeName + "'");
        }
        return cancelledStatusCode;
    }

    @Override
    public void cancelExtractedCreditMemo(final VendorCreditMemoDocument cmDocument, final String note) {
        LOG.debug("cancelExtractedCreditMemo() started");
        if (CreditMemoStatuses.CANCELLED_STATUSES.contains(cmDocument.getApplicationDocumentStatus())) {
            LOG.debug("cancelExtractedCreditMemo() ended");
            return;
        }

        try {
            final Note noteObj = documentService.createNoteFromDocument(cmDocument, note);
            cmDocument.addNote(noteObj);
        } catch (final Exception e) {
            throw new RuntimeException(e.getMessage());
        }

        accountsPayableService.cancelAccountsPayableDocument(cmDocument, "");
        LOG.debug(
                "cancelExtractedCreditMemo() CM {} Cancelled Without Workflow",
                cmDocument::getPurapDocumentIdentifier
        );
        LOG.debug("cancelExtractedCreditMemo() ended");

    }

    @Override
    public void resetExtractedCreditMemo(final VendorCreditMemoDocument cmDocument, final String note) {
        LOG.debug("resetExtractedCreditMemo() started");
        if (CreditMemoStatuses.CANCELLED_STATUSES.contains(cmDocument.getApplicationDocumentStatus())) {
            LOG.debug("resetExtractedCreditMemo() ended");
            return;
        }
        cmDocument.setExtractedTimestamp(null);
        cmDocument.setCreditMemoPaidTimestamp(null);

        final Note noteObj;
        try {
            noteObj = documentService.createNoteFromDocument(cmDocument, note);
            cmDocument.addNote(noteObj);
        } catch (final Exception e) {
            throw new RuntimeException(e.getMessage());
        }
        purapService.saveDocumentNoValidation(cmDocument);

        LOG.debug(
                "resetExtractedCreditMemo() CM {} Cancelled Without Workflow",
                cmDocument::getPurapDocumentIdentifier
        );
        LOG.debug("resetExtractedCreditMemo() ended");
    }

    @Override
    public boolean shouldPurchaseOrderBeReversed(final AccountsPayableDocument apDoc) {
        // always return false, never reverse
        return false;
    }

    @Override
    public Person getPersonForCancel(final AccountsPayableDocument apDoc) {
        // return null, since superuser is fine for CM
        return null;
    }

    @Override
    public void takePurchaseOrderCancelAction(final AccountsPayableDocument apDoc) {
        final VendorCreditMemoDocument cmDocument = (VendorCreditMemoDocument) apDoc;
        if (cmDocument.isReopenPurchaseOrderIndicator()) {
            final String docType = PurapConstants.PurapDocTypeCodes.PURCHASE_ORDER_CLOSE_DOCUMENT;
            purchaseOrderService.createAndRoutePotentialChangeDocument(
                    cmDocument.getPurchaseOrderDocument().getDocumentNumber(), docType, "reopened by Payment Request " +
                            apDoc.getPurapDocumentIdentifier() + "cancel", new ArrayList(),
                    PurchaseOrderStatuses.APPDOC_PENDING_CLOSE);
        }
    }

    @Override
    public void markPaid(final VendorCreditMemoDocument cm, final Date processDate) {
        LOG.debug("markPaid() started");

        cm.setCreditMemoPaidTimestamp(new Timestamp(processDate.getTime()));
        purapService.saveDocumentNoValidation(cm);
    }

    @Override
    public boolean poItemEligibleForAp(final AccountsPayableDocument apDoc, final PurchaseOrderItem poItem) {
        poItem.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

        // if the po item is not active... skip it
        if (!poItem.isItemActiveIndicator()) {
            return false;
        }

        if (poItem.getItemType().isQuantityBasedGeneralLedgerIndicator()
                && poItem.getItemInvoicedTotalQuantity().isGreaterThan(KualiDecimal.ZERO)) {
            return true;
        } else {
            final BigDecimal unitPrice = poItem.getItemUnitPrice() == null ? new BigDecimal(0) :
                    poItem.getItemUnitPrice();
            return unitPrice.doubleValue() > poItem.getItemOutstandingEncumberedAmount().doubleValue();
        }
    }

    /**
     * The given document here needs to be a Credit Memo.
     */
    @Override
    public void generateGLEntriesCreateAccountsPayableDocument(final AccountsPayableDocument apDocument) {
        final VendorCreditMemoDocument creditMemo = (VendorCreditMemoDocument) apDocument;
        purapGeneralLedgerService.generateEntriesCreateCreditMemo(creditMemo);
    }

    /**
     * Records the specified error message into the Log file and throws a runtime exception.
     *
     * @param errorMessage the error message to be logged.
     */
    protected void logAndThrowRuntimeException(final String errorMessage) {
        this.logAndThrowRuntimeException(errorMessage, null);
    }

    /**
     * Records the specified error message into the Log file and throws the specified runtime exception.
     *
     * @param errorMessage the specified error message.
     * @param e            the specified runtime exception.
     */
    protected void logAndThrowRuntimeException(final String errorMessage, final Exception e) {
        if (ObjectUtils.isNotNull(e)) {
            LOG.error(errorMessage, e);
            throw new RuntimeException(errorMessage, e);
        } else {
            LOG.error(errorMessage);
            throw new RuntimeException(errorMessage);
        }
    }

    @Override
    public void populateDocumentAfterInit(final VendorCreditMemoDocument cmDocument) {
        // make a call to search for expired/closed accounts
        final HashMap<String, ExpiredOrClosedAccountEntry> expiredOrClosedAccountList =
                accountsPayableService.getExpiredOrClosedAccountList(cmDocument);

        if (cmDocument.isSourceDocumentPaymentRequest()) {
            populateDocumentFromPreq(cmDocument, expiredOrClosedAccountList);
        } else if (cmDocument.isSourceDocumentPurchaseOrder()) {
            populateDocumentFromPO(cmDocument, expiredOrClosedAccountList);
        } else {
            populateDocumentFromVendor(cmDocument);
        }

        populateDocumentDescription(cmDocument);

        // write a note for expired/closed accounts if any exist and add a message stating there were expired/closed
        // accounts at the top of the document
        accountsPayableService.generateExpiredOrClosedAccountNote(cmDocument, expiredOrClosedAccountList);

        // set indicator so a message is displayed for accounts that were replaced due to expired/closed status
        if (ObjectUtils.isNotNull(expiredOrClosedAccountList) && !expiredOrClosedAccountList.isEmpty()) {
            cmDocument.setContinuationAccountIndicator(true);
        }
    }

    /**
     * Populate Credit Memo of type Payment Request.
     *
     * @param cmDocument Credit Memo Document to Populate
     */
    protected void populateDocumentFromPreq(
            final VendorCreditMemoDocument cmDocument,
            final HashMap<String, ExpiredOrClosedAccountEntry> expiredOrClosedAccountList) {
        final PaymentRequestDocument paymentRequestDocument = paymentRequestService.getPaymentRequestById(
                cmDocument.getPaymentRequestIdentifier());
        cmDocument.getDocumentHeader().setOrganizationDocumentNumber(
                paymentRequestDocument.getDocumentHeader().getOrganizationDocumentNumber());
        cmDocument.setPaymentRequestDocument(paymentRequestDocument);
        cmDocument.setPurchaseOrderDocument(paymentRequestDocument.getPurchaseOrderDocument());
        cmDocument.setUseTaxIndicator(paymentRequestDocument.isUseTaxIndicator());

        // credit memo address taken directly from payment request
        cmDocument.setVendorHeaderGeneratedIdentifier(paymentRequestDocument.getVendorHeaderGeneratedIdentifier());
        cmDocument.setVendorDetailAssignedIdentifier(paymentRequestDocument.getVendorDetailAssignedIdentifier());
        cmDocument.setVendorAddressGeneratedIdentifier(paymentRequestDocument.getVendorAddressGeneratedIdentifier());
        cmDocument.setVendorCustomerNumber(paymentRequestDocument.getVendorCustomerNumber());
        cmDocument.setVendorName(paymentRequestDocument.getVendorName());
        cmDocument.setVendorLine1Address(paymentRequestDocument.getVendorLine1Address());
        cmDocument.setVendorLine2Address(paymentRequestDocument.getVendorLine2Address());
        cmDocument.setVendorCityName(paymentRequestDocument.getVendorCityName());
        cmDocument.setVendorStateCode(paymentRequestDocument.getVendorStateCode());
        cmDocument.setVendorPostalCode(paymentRequestDocument.getVendorPostalCode());
        cmDocument.setVendorCountryCode(paymentRequestDocument.getVendorCountryCode());
        cmDocument.setVendorAttentionName(paymentRequestDocument.getVendorAttentionName());
        cmDocument.setAccountsPayablePurchasingDocumentLinkIdentifier(
                paymentRequestDocument.getAccountsPayablePurchasingDocumentLinkIdentifier());

        // prep the item lines (also collect warnings for later display) this is only done on paymentRequest
        purapAccountingService.convertMoneyToPercent(paymentRequestDocument);
        populateItemLinesFromPreq(cmDocument, expiredOrClosedAccountList);
    }

    /**
     * Populates the credit memo items from the payment request items.
     *
     * @param cmDocument Credit Memo Document to Populate
     */
    protected void populateItemLinesFromPreq(
            final VendorCreditMemoDocument cmDocument,
            final HashMap<String, ExpiredOrClosedAccountEntry> expiredOrClosedAccountList) {
        final PaymentRequestDocument preqDocument = cmDocument.getPaymentRequestDocument();

        for (final PaymentRequestItem preqItemToTemplate : (List<PaymentRequestItem>) preqDocument.getItems()) {
            preqItemToTemplate.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (preqItemToTemplate.getItemType().isLineItemIndicator()
                    && (preqItemToTemplate.getItemType().isQuantityBasedGeneralLedgerIndicator()
                    && preqItemToTemplate.getItemQuantity().isNonZero()
                    || preqItemToTemplate.getItemType().isAmountBasedGeneralLedgerIndicator()
                    && preqItemToTemplate.getTotalAmount().isNonZero())) {
                cmDocument.getItems().add(new CreditMemoItem(cmDocument, preqItemToTemplate,
                        preqItemToTemplate.getPurchaseOrderItem(), expiredOrClosedAccountList));
            }
        }

        purapService.addBelowLineItems(cmDocument);
        cmDocument.fixItemReferences();
    }

    /**
     * Populate Credit Memo of type Purchase Order.
     *
     * @param cmDocument Credit Memo Document to Populate
     */
    protected void populateDocumentFromPO(
            final VendorCreditMemoDocument cmDocument,
            final HashMap<String, ExpiredOrClosedAccountEntry> expiredOrClosedAccountList) {
        final PurchaseOrderDocument purchaseOrderDocument = purchaseOrderService.getCurrentPurchaseOrder(
                cmDocument.getPurchaseOrderIdentifier());
        cmDocument.setPurchaseOrderDocument(purchaseOrderDocument);
        cmDocument.getDocumentHeader().setOrganizationDocumentNumber(
                purchaseOrderDocument.getDocumentHeader().getOrganizationDocumentNumber());
        cmDocument.setUseTaxIndicator(cmDocument.isUseTaxIndicator());

        cmDocument.setVendorHeaderGeneratedIdentifier(purchaseOrderDocument.getVendorHeaderGeneratedIdentifier());
        cmDocument.setVendorDetailAssignedIdentifier(purchaseOrderDocument.getVendorDetailAssignedIdentifier());
        cmDocument.setVendorCustomerNumber(purchaseOrderDocument.getVendorCustomerNumber());
        cmDocument.setVendorName(purchaseOrderDocument.getVendorName());
        cmDocument.setAccountsPayablePurchasingDocumentLinkIdentifier(
                purchaseOrderDocument.getAccountsPayablePurchasingDocumentLinkIdentifier());

        // populate cm vendor address with the default remit address type for the vendor if found
        final String userCampus = GlobalVariables.getUserSession().getPerson().getCampusCode();
        final VendorAddress vendorAddress = vendorService.getVendorDefaultAddress(
                purchaseOrderDocument.getVendorHeaderGeneratedIdentifier(),
                purchaseOrderDocument.getVendorDetailAssignedIdentifier(), VendorConstants.AddressTypes.REMIT,
                userCampus, false);
        if (vendorAddress != null) {
            cmDocument.templateVendorAddress(vendorAddress);
            cmDocument.setVendorAddressGeneratedIdentifier(vendorAddress.getVendorAddressGeneratedIdentifier());
            cmDocument.setVendorAttentionName(StringUtils.defaultString(vendorAddress.getVendorAttentionName()));
        } else {
            // set address from PO
            cmDocument.setVendorAddressGeneratedIdentifier(purchaseOrderDocument.getVendorAddressGeneratedIdentifier());
            cmDocument.setVendorLine1Address(purchaseOrderDocument.getVendorLine1Address());
            cmDocument.setVendorLine2Address(purchaseOrderDocument.getVendorLine2Address());
            cmDocument.setVendorCityName(purchaseOrderDocument.getVendorCityName());
            cmDocument.setVendorStateCode(purchaseOrderDocument.getVendorStateCode());
            cmDocument.setVendorPostalCode(purchaseOrderDocument.getVendorPostalCode());
            cmDocument.setVendorCountryCode(purchaseOrderDocument.getVendorCountryCode());

            final boolean blankAttentionLine = StringUtils.equalsIgnoreCase("Y", parameterService
                    .getParameterValueAsString(PurapConstants.PURAP_NAMESPACE, "Document",
                            PurapParameterConstants.CLEAR_ATTENTION_LINE_IND
                    ));
            if (blankAttentionLine) {
                cmDocument.setVendorAttentionName(StringUtils.EMPTY);
            } else {
                cmDocument.setVendorAttentionName(StringUtils.defaultString(
                        purchaseOrderDocument.getVendorAttentionName()));
            }
        }

        populateItemLinesFromPO(cmDocument, expiredOrClosedAccountList);
    }

    /**
     * Populates the credit memo items from the payment request items.
     *
     * @param cmDocument Credit Memo Document to Populate
     */
    protected void populateItemLinesFromPO(
            final VendorCreditMemoDocument cmDocument,
            final HashMap<String, ExpiredOrClosedAccountEntry> expiredOrClosedAccountList) {
        final List<PurchaseOrderItem> invoicedItems = getPOInvoicedItems(cmDocument.getPurchaseOrderDocument());
        for (final PurchaseOrderItem poItem : invoicedItems) {
            poItem.refreshReferenceObject(PurapPropertyConstants.ITEM_TYPE);

            if (poItem.getItemType().isQuantityBasedGeneralLedgerIndicator()
                    && poItem.getItemInvoicedTotalQuantity().isNonZero()
                || poItem.getItemType().isAmountBasedGeneralLedgerIndicator()
                && poItem.getItemInvoicedTotalAmount().isNonZero()) {
                final CreditMemoItem creditMemoItem = new CreditMemoItem(cmDocument, poItem, expiredOrClosedAccountList);
                cmDocument.getItems().add(creditMemoItem);
                final PurchasingCapitalAssetItem purchasingCAMSItem = cmDocument.getPurchaseOrderDocument()
                        .getPurchasingCapitalAssetItemByItemIdentifier(poItem.getItemIdentifier());
                if (purchasingCAMSItem != null) {
                    creditMemoItem.setCapitalAssetTransactionTypeCode(
                            purchasingCAMSItem.getCapitalAssetTransactionTypeCode());
                }
            }
        }

        purapService.addBelowLineItems(cmDocument);
        cmDocument.fixItemReferences();
    }

    /**
     * Populate Credit Memo of type Vendor.
     *
     * @param cmDocument Credit Memo Document to Populate
     */
    protected void populateDocumentFromVendor(final VendorCreditMemoDocument cmDocument) {
        final Integer vendorHeaderId = VendorUtils.getVendorHeaderId(cmDocument.getVendorNumber());
        final Integer vendorDetailId = VendorUtils.getVendorDetailId(cmDocument.getVendorNumber());

        final VendorDetail vendorDetail = vendorService.getVendorDetail(vendorHeaderId, vendorDetailId);
        cmDocument.setVendorDetail(vendorDetail);

        cmDocument.setVendorHeaderGeneratedIdentifier(vendorDetail.getVendorHeaderGeneratedIdentifier());
        cmDocument.setVendorDetailAssignedIdentifier(vendorDetail.getVendorDetailAssignedIdentifier());
        cmDocument.setVendorCustomerNumber(vendorDetail.getVendorNumber());
        cmDocument.setVendorName(vendorDetail.getVendorName());

        // credit memo type vendor uses the default remit type address for the vendor if found
        final String userCampus = GlobalVariables.getUserSession().getPerson().getCampusCode();
        VendorAddress vendorAddress = vendorService.getVendorDefaultAddress(vendorHeaderId, vendorDetailId,
                VendorConstants.AddressTypes.REMIT, userCampus, false);
        if (vendorAddress == null) {
            // pick up the default vendor po address type
            vendorAddress = vendorService.getVendorDefaultAddress(vendorHeaderId, vendorDetailId,
                    VendorConstants.AddressTypes.PURCHASE_ORDER, userCampus, false);
        }

        cmDocument.setVendorAddressGeneratedIdentifier(vendorAddress.getVendorAddressGeneratedIdentifier());
        cmDocument.templateVendorAddress(vendorAddress);

        purapService.addBelowLineItems(cmDocument);
    }

    /**
     * Defaults the document description based on the credit memo source type.
     *
     * @param cmDocument Credit Memo Document to Populate
     */
    protected void populateDocumentDescription(final VendorCreditMemoDocument cmDocument) {
        String description;
        if (cmDocument.isSourceVendor()) {
            description = "Vendor: " + cmDocument.getVendorName();
        } else {
            description = "PO: " + cmDocument.getPurchaseOrderDocument().getPurapDocumentIdentifier() + " Vendor: " +
                    cmDocument.getVendorName();
        }

        // trim description if longer than whats specified in the data dictionary
        final int noteTextMaxLength = dataDictionaryService.getAttributeMaxLength(DocumentHeader.class,
                KRADPropertyConstants.DOCUMENT_DESCRIPTION);
        if (noteTextMaxLength < description.length()) {
            description = description.substring(0, noteTextMaxLength);
        }

        cmDocument.getDocumentHeader().setDocumentDescription(description);
    }

    public void setAccountsPayableService(final AccountsPayableService accountsPayableService) {
        this.accountsPayableService = accountsPayableService;
    }

    public void setCreditMemoDao(final CreditMemoDao creditMemoDao) {
        this.creditMemoDao = creditMemoDao;
    }

    public void setDataDictionaryService(final DataDictionaryService dataDictionaryService) {
        this.dataDictionaryService = dataDictionaryService;
    }

    public void setDocumentService(final DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setConfigurationService(final ConfigurationService kualiConfigurationService) {
        this.kualiConfigurationService = kualiConfigurationService;
    }

    public void setNoteService(final NoteService noteService) {
        this.noteService = noteService;
    }

    public void setPaymentRequestService(final PaymentRequestService paymentRequestService) {
        this.paymentRequestService = paymentRequestService;
    }

    public void setPurapAccountingService(final PurapAccountingService purapAccountingService) {
        this.purapAccountingService = purapAccountingService;
    }

    public void setPurapGeneralLedgerService(final PurapGeneralLedgerService purapGeneralLedgerService) {
        this.purapGeneralLedgerService = purapGeneralLedgerService;
    }

    public void setPurapService(final PurapService purapService) {
        this.purapService = purapService;
    }

    public void setPurchaseOrderService(final PurchaseOrderService purchaseOrderService) {
        this.purchaseOrderService = purchaseOrderService;
    }

    public void setVendorService(final VendorService vendorService) {
        this.vendorService = vendorService;
    }

    public void setWorkflowDocumentService(final WorkflowDocumentService workflowDocumentService) {
        this.workflowDocumentService = workflowDocumentService;
    }

    public void setFinancialSystemDocumentService(final FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    protected BankService getBankService() {
        return bankService;
    }

    public void setBankService(final BankService bankService) {
        this.bankService = bankService;
    }

    public void setParameterService(final ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setBusinessObjectService(final BusinessObjectService businessObjectService) {
        this.businessObjectService = businessObjectService;
    }
}
