/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2021 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.kuali.kfs.module.purap.batch.service.impl;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.kuali.kfs.core.api.datetime.DateTimeService;
import org.kuali.kfs.core.api.util.type.KualiDecimal;
import org.kuali.kfs.coreservice.framework.parameter.ParameterService;
import org.kuali.kfs.coreservice.impl.parameter.Parameter;
import org.kuali.kfs.krad.bo.Note;
import org.kuali.kfs.krad.service.DocumentService;
import org.kuali.kfs.krad.service.KualiRuleService;
import org.kuali.kfs.krad.service.NoteService;
import org.kuali.kfs.module.purap.PurapConstants;
import org.kuali.kfs.module.purap.PurapParameterConstants;
import org.kuali.kfs.module.purap.PurchaseOrderStatuses;
import org.kuali.kfs.module.purap.batch.AutoCloseRecurringOrdersStep;
import org.kuali.kfs.module.purap.batch.service.AutoClosePurchaseOrderService;
import org.kuali.kfs.module.purap.businessobject.AutoClosePurchaseOrderView;
import org.kuali.kfs.module.purap.document.PurchaseOrderDocument;
import org.kuali.kfs.module.purap.document.dataaccess.PurchaseOrderDao;
import org.kuali.kfs.module.purap.document.service.PurchaseOrderService;
import org.kuali.kfs.sys.businessobject.DocumentHeader;
import org.kuali.kfs.sys.document.service.FinancialSystemDocumentService;
import org.kuali.kfs.sys.document.validation.event.AttributedRouteDocumentEvent;
import org.kuali.kfs.sys.mail.BodyMailMessage;
import org.kuali.kfs.sys.service.EmailService;
import org.springframework.transaction.annotation.Transactional;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.List;
import java.util.Objects;

public class AutoClosePurchaseOrderServiceImpl implements AutoClosePurchaseOrderService {

    private static final Logger LOG = LogManager.getLogger();

    protected PurchaseOrderService purchaseOrderService;
    protected ParameterService parameterService;
    protected DateTimeService dateTimeService;
    protected PurchaseOrderDao purchaseOrderDao;
    protected KualiRuleService kualiRuleService;
    protected FinancialSystemDocumentService financialSystemDocumentService;
    protected DocumentService documentService;
    protected NoteService noteService;
    protected EmailService emailService;

    @Override
    public boolean autoCloseFullyDisencumberedOrders() {
        LOG.debug("autoCloseFullyDisencumberedOrders() started");

        List<AutoClosePurchaseOrderView> autoCloseList = this.getAllOpenPurchaseOrdersForAutoClose();

        for (AutoClosePurchaseOrderView poAutoClose : autoCloseList) {
            autoClosePurchaseOrder(poAutoClose);
        }

        LOG.debug("autoCloseFullyDisencumberedOrders() ended");

        return true;
    }

    @Transactional
    public void autoClosePurchaseOrder(AutoClosePurchaseOrderView poAutoClose) {
        if (poAutoClose.getTotalAmount() != null && KualiDecimal.ZERO.compareTo(poAutoClose.getTotalAmount()) != 0) {
            LOG.info("autoCloseFullyDisencumberedOrders() PO ID " + poAutoClose.getPurapDocumentIdentifier() +
                    " with total " + poAutoClose.getTotalAmount().doubleValue() + " will be closed");
            String newStatus = PurchaseOrderStatuses.APPDOC_PENDING_CLOSE;
            String annotation = "This PO was automatically closed in batch.";
            String documentType = PurapConstants.PurapDocTypeCodes.PURCHASE_ORDER_CLOSE_DOCUMENT;
            PurchaseOrderDocument document = getPurchaseOrderService().getPurchaseOrderByDocumentNumber(
                    poAutoClose.getDocumentNumber());
            this.createNoteForAutoCloseOrders(document, annotation);
            getPurchaseOrderService().createAndRoutePotentialChangeDocument(poAutoClose.getDocumentNumber(),
                    documentType, annotation, null, newStatus);
        }
    }

    @Override
    @Transactional
    public boolean autoCloseRecurringOrders() {
        LOG.debug("autoCloseRecurringOrders() started");
        boolean shouldSendEmail = true;
        BodyMailMessage message = new BodyMailMessage();
        String parameterToEmail = parameterService.getParameterValueAsString(AutoCloseRecurringOrdersStep.class,
                PurapParameterConstants.AUTO_CLOSE_RECURRING_PO_TO_EMAIL_ADDRESSES);
        String parameterFromEmail = parameterService.getParameterValueAsString(AutoCloseRecurringOrdersStep.class,
                PurapParameterConstants.EMAIL_FROM);

        if (StringUtils.isEmpty(parameterToEmail)) {
            // Don't stop the show if the email address is wrong, log it and continue.
            LOG.error("autoCloseRecurringOrders(): parameterToEmail is missing, we'll not send out any emails for " +
                    "this job.");
            shouldSendEmail = false;
        }
        if (StringUtils.isEmpty(parameterFromEmail)) {
            // Don't stop the show if the email address is wrong, log it and continue.
            LOG.error("autoCloseRecurringOrders(): parameterFromEmail is missing, we'll not send out any emails for " +
                    "this job.");
            shouldSendEmail = false;
        }
        if (shouldSendEmail) {
            message = setMessageAddressesAndSubject(message, parameterToEmail, parameterFromEmail);
        }
        StringBuffer emailBody = new StringBuffer();
        // There should always be a "AUTO_CLOSE_RECURRING_ORDER_DT"
        // row in the table, this method sets it to "mm/dd/yyyy" after processing.
        String recurringOrderDateString = parameterService.getParameterValueAsString(
                AutoCloseRecurringOrdersStep.class, PurapParameterConstants.AUTO_CLOSE_RECURRING_PO_DATE);
        boolean validDate = true;
        java.util.Date recurringOrderDate = null;
        try {
            recurringOrderDate = dateTimeService.convertToDate(recurringOrderDateString);
        } catch (ParseException pe) {
            validDate = false;
        }
        if (StringUtils.isEmpty(recurringOrderDateString) || "mm/dd/yyyy".equalsIgnoreCase(recurringOrderDateString)
            || !validDate) {
            if ("mm/dd/yyyy".equalsIgnoreCase(recurringOrderDateString)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("autoCloseRecurringOrders(): mm/dd/yyyy was found in the Application Settings " +
                            "table. No orders will be closed, method will end.");
                }
                if (shouldSendEmail) {
                    emailBody.append("The AUTO_CLOSE_RECURRING_ORDER_DT found in the Application Settings table " +
                            "was mm/dd/yyyy. No recurring PO's were closed.");
                }
            } else {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("autoCloseRecurringOrders(): An invalid autoCloseRecurringOrdersDate was found in " +
                            "the Application Settings table: " + recurringOrderDateString + ". Method will end.");
                }
                if (shouldSendEmail) {
                    emailBody.append("An invalid AUTO_CLOSE_RECURRING_ORDER_DT was found in the Application Settings " +
                            "table: ").append(recurringOrderDateString).append(". No recurring PO's were closed.");
                }
            }
            if (shouldSendEmail) {
                sendMessage(message, emailBody.toString());
            }
            LOG.info("autoCloseRecurringOrders() ended");

            return false;
        }
        LOG.info("autoCloseRecurringOrders() The autoCloseRecurringOrdersDate found in the Application Settings " +
                "table was " + recurringOrderDateString);
        if (shouldSendEmail) {
            emailBody.append("The autoCloseRecurringOrdersDate found in the Application Settings table was ")
                    .append(recurringOrderDateString).append(".");
        }
        Calendar appSettingsDate = dateTimeService.getCalendar(recurringOrderDate);
        Timestamp appSettingsDay = new Timestamp(appSettingsDate.getTime().getTime());

        Calendar todayMinusThreeMonths = getTodayMinusThreeMonths();
        Timestamp threeMonthsAgo = new Timestamp(todayMinusThreeMonths.getTime().getTime());

        if (appSettingsDate.after(todayMinusThreeMonths)) {
            LOG.info("autoCloseRecurringOrders() The appSettingsDate: " + appSettingsDay +
                    " is after todayMinusThreeMonths: " + threeMonthsAgo + ". The program will end.");
            if (shouldSendEmail) {
                emailBody.append("\n\nThe autoCloseRecurringOrdersDate: ").append(appSettingsDay)
                        .append(" is after todayMinusThreeMonths: ").append(threeMonthsAgo)
                        .append(". The program will end.");
                sendMessage(message, emailBody.toString());
            }
            LOG.info("autoCloseRecurringOrders() ended");

            return false;
        }

        List<AutoClosePurchaseOrderView> closeList = purchaseOrderDao.getAutoCloseRecurringPurchaseOrders(
                getExcludedVendorChoiceCodes());

        // we need to eliminate the AutoClosePurchaseOrderView whose workflow document status is not OPEN.
        List<AutoClosePurchaseOrderView> purchaseOrderAutoCloseList = filterDocumentsForAppDocStatusOpen(closeList);

        LOG.info("autoCloseRecurringOrders(): " + purchaseOrderAutoCloseList.size() +
                " PO's were returned for processing.");
        int counter = 0;
        for (AutoClosePurchaseOrderView poAutoClose : purchaseOrderAutoCloseList) {
            LOG.info("autoCloseRecurringOrders(): Testing PO ID " + poAutoClose.getPurapDocumentIdentifier() +
                    ". recurringPaymentEndDate: " + poAutoClose.getRecurringPaymentEndDate());
            if (poAutoClose.getRecurringPaymentEndDate().before(threeMonthsAgo)) {
                String newStatus = PurchaseOrderStatuses.APPDOC_PENDING_CLOSE;
                String annotation = "This recurring PO was automatically closed in batch.";
                String documentType = PurapConstants.PurapDocTypeCodes.PURCHASE_ORDER_CLOSE_DOCUMENT;
                PurchaseOrderDocument document = purchaseOrderService.getPurchaseOrderByDocumentNumber(
                        poAutoClose.getDocumentNumber());
                kualiRuleService.applyRules(new AttributedRouteDocumentEvent("", document));

                ++counter;
                if (counter == 1) {
                    emailBody.append("\n\nThe following recurring Purchase Orders will be closed by auto close " +
                            "recurring batch job \n");
                }
                LOG.info("autoCloseRecurringOrders() PO ID " + poAutoClose.getPurapDocumentIdentifier() +
                        " will be closed.");
                createNoteForAutoCloseOrders(document, annotation);
                purchaseOrderService.createAndRoutePotentialChangeDocument(poAutoClose.getDocumentNumber(),
                        documentType, annotation, null, newStatus);
                if (shouldSendEmail) {
                    emailBody.append("\n\n").append(counter).append(" PO ID: ")
                            .append(poAutoClose.getPurapDocumentIdentifier()).append(", End Date: ")
                            .append(poAutoClose.getRecurringPaymentEndDate()).append(", Status: ")
                            .append(poAutoClose.getApplicationDocumentStatus()).append(", VendorChoice: ")
                            .append(poAutoClose.getVendorChoiceCode()).append(", RecurringPaymentType: ")
                            .append(poAutoClose.getRecurringPaymentTypeCode());
                }
            }
        }
        if (counter == 0) {
            LOG.info("\n\nNo recurring PO's fit the conditions for closing.");
            if (shouldSendEmail) {
                emailBody.append("\n\nNo recurring PO's fit the conditions for closing.");
            }
        }
        if (shouldSendEmail) {
            sendMessage(message, emailBody.toString());
        }
        resetAutoCloseRecurringOrderDateParameter();
        LOG.debug("autoCloseRecurringOrders() ended");

        return true;
    }

    /**
     * Filter out the auto close purchase order view documents for the appDocStatus with status open For each document
     * in the list, check if there is workflowdocument whose appdocstatus is open add add to the return list.
     *
     * @return filteredAutoClosePOView filtered auto close po view documents where appdocstatus is open
     */
    @Transactional
    protected List<AutoClosePurchaseOrderView> filterDocumentsForAppDocStatusOpen(
            List<AutoClosePurchaseOrderView> autoClosePurchaseOrderViews) {
        List<AutoClosePurchaseOrderView> filteredAutoClosePOView = new ArrayList<>();

        LOG.info("Start filtering " + autoClosePurchaseOrderViews.size() + " documents for app doc status open...");
        int count = 0;

        for (AutoClosePurchaseOrderView autoClosePurchaseOrderView : autoClosePurchaseOrderViews) {
            DocumentHeader documentHeader = financialSystemDocumentService.findByDocumentNumber(
                    autoClosePurchaseOrderView.getDocumentNumber());

            if (documentHeader != null) {
                if (PurchaseOrderStatuses.APPDOC_OPEN.equalsIgnoreCase(
                        documentHeader.getApplicationDocumentStatus())) {
                    // found the matched Awaiting Contract Manager Assignment status, retrieve the routeHeaderId and
                    // add to the list
                    filteredAutoClosePOView.add(autoClosePurchaseOrderView);
                }
            }
            count++;
            if (count % 10000 == 0) {
                LOG.debug("Filtered " + count + " documents.");
            }
        }
        LOG.info("Filtering is Complete. Found " + filteredAutoClosePOView.size() + " documents to be closed.");

        return filteredAutoClosePOView;
    }


    /**
     * Creates and add a note to the purchase order document using the annotation String in the input parameter. This
     * method is used by the autoCloseRecurringOrders() and autoCloseFullyDisencumberedOrders to add a note to the
     * purchase order to indicate that the purchase order was closed by the batch job.
     *
     * @param purchaseOrderDocument The purchase order document that is being closed by the batch job.
     * @param annotation            The string to appear on the note to be attached to the purchase order.
     */
    @Override
    @Transactional
    public void createNoteForAutoCloseOrders(PurchaseOrderDocument purchaseOrderDocument, String annotation) {
        try {
            Note noteObj = documentService.createNoteFromDocument(purchaseOrderDocument, annotation);
            noteService.save(noteObj);
        } catch (Exception e) {
            String errorMessage = "Error creating and saving close note for purchase order with document service";
            LOG.error("createNoteForAutoCloseRecurringOrders " + errorMessage, e);
            throw new RuntimeException(errorMessage, e);
        }
    }

    /**
     * Sets the to addresses, from address and the subject of the email.
     *
     * @param message        The MailMessage object of the email to be sent.
     * @param parameterToEmail The String of to email addresses with delimiters of ";" obtained from the system parameter.
     * @param parameterFromEmail The from email address obtained from the system parameter.
     * @return The MailMessage object after the to addresses, from address and the subject have been set.
     */
    @Transactional
    protected BodyMailMessage setMessageAddressesAndSubject(BodyMailMessage message, String parameterToEmail, String parameterFromEmail) {
        String[] toAddressList = parameterToEmail.split(";");

        if (toAddressList.length > 0) {
            Arrays.stream(toAddressList).filter(Objects::nonNull).map(String::trim).forEach(message::addToAddress);
        }

        message.setFromAddress(parameterFromEmail);
        message.setSubject("Auto Close Recurring Purchase Orders");
        return message;
    }

    /**
     * Sends the email by calling the sendMessage method in mailService and log error if exception occurs during the
     * attempt to send the message.
     *
     * @param message   The MailMessage object containing information to be sent.
     * @param emailBody The String containing the body of the email to be sent.
     */
    @Transactional
    protected void sendMessage(BodyMailMessage message, String emailBody) {
        message.setMessage(emailBody);
        try {
            emailService.sendMessage(message, false);
        } catch (Exception e) {
            // Don't stop the show if the email has problem, log it and continue.
            LOG.error("sendMessage(): email problem. Message not sent.", e);
        }
    }

    /**
     * Creates and returns a Calendar object of today minus three months.
     *
     * @return Calendar object of today minus three months.
     */
    @Transactional
    protected Calendar getTodayMinusThreeMonths() {
        // Set to today.
        Calendar todayMinusThreeMonths = Calendar.getInstance();
        // Back up 3 months.
        todayMinusThreeMonths.add(Calendar.MONTH, -3);
        todayMinusThreeMonths.set(Calendar.HOUR, 12);
        todayMinusThreeMonths.set(Calendar.MINUTE, 0);
        todayMinusThreeMonths.set(Calendar.SECOND, 0);
        todayMinusThreeMonths.set(Calendar.MILLISECOND, 0);
        todayMinusThreeMonths.set(Calendar.AM_PM, Calendar.AM);
        return todayMinusThreeMonths;
    }

    /**
     * Gets a List of excluded vendor choice codes from PurapConstants.
     *
     * @return a List of excluded vendor choice codes
     */
    @Transactional
    protected List<String> getExcludedVendorChoiceCodes() {
        return new ArrayList<>(Arrays.asList(PurapConstants.AUTO_CLOSE_EXCLUSION_VNDR_CHOICE_CODES));
    }

    /**
     * Resets the AUTO_CLOSE_RECURRING_ORDER_DT system parameter to "mm/dd/yyyy".
     */
    @Transactional
    protected void resetAutoCloseRecurringOrderDateParameter() {
        Parameter autoCloseRecurringPODate = parameterService.getParameter(AutoCloseRecurringOrdersStep.class,
                PurapParameterConstants.AUTO_CLOSE_RECURRING_PO_DATE);
        if (autoCloseRecurringPODate != null) {
            autoCloseRecurringPODate.setValue("mm/dd/yyyy");
            parameterService.updateParameter(autoCloseRecurringPODate);
        }
    }

    @Override
    @Transactional
    public List<AutoClosePurchaseOrderView> getAllOpenPurchaseOrdersForAutoClose() {
        return purchaseOrderDao.getAllOpenPurchaseOrders(getExcludedVendorChoiceCodes());
    }

    public PurchaseOrderService getPurchaseOrderService() {
        return purchaseOrderService;
    }

    public void setPurchaseOrderService(PurchaseOrderService purchaseOrderService) {
        this.purchaseOrderService = purchaseOrderService;
    }

    public void setParameterService(ParameterService parameterService) {
        this.parameterService = parameterService;
    }

    public void setDateTimeService(DateTimeService dateTimeService) {
        this.dateTimeService = dateTimeService;
    }

    public void setPurchaseOrderDao(PurchaseOrderDao purchaseOrderDao) {
        this.purchaseOrderDao = purchaseOrderDao;
    }

    public void setKualiRuleService(KualiRuleService kualiRuleService) {
        this.kualiRuleService = kualiRuleService;
    }

    public void setFinancialSystemDocumentService(FinancialSystemDocumentService financialSystemDocumentService) {
        this.financialSystemDocumentService = financialSystemDocumentService;
    }

    public void setDocumentService(DocumentService documentService) {
        this.documentService = documentService;
    }

    public void setNoteService(NoteService noteService) {
        this.noteService = noteService;
    }

    public void setEmailService(EmailService emailService) {
        this.emailService = emailService;
    }
}
