/**
 *	Copyright (c) 2015-2016 Vr Security Inc.
 *	All rights reserved.
 *	
 *	Redistribution and use in source and binary forms, with or without
 *	modification, are permitted provided that the following conditions are met:
 *	    * Redistributions of source code must retain the above copyright
 *	      notice, this list of conditions and the following disclaimer.
 *	    * Redistributions in binary form must reproduce the above copyright
 *	      notice, this list of conditions and the following disclaimer in the
 *	      documentation and/or other materials provided with the distribution.
 *	    * Neither the name of the <organization> nor the
 *	      names of its contributors may be used to endorse or promote products
 *	      derived from this software without specific prior written permission.
 *	
 *	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 *	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 *	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *	DISCLAIMED. IN NO EVENT SHALL <COPYRIGHT HOLDER> BE LIABLE FOR ANY
 *	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 *	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 *	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 *	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

// Provides simplified REST API access
var RestClient = require('node-rest-client').Client;

//RELEASE HOST
var ossindex = "https://ossindex.net";

//DEBUG HOST
//var ossindex = "http://localhost:8080";

//Instantiate the rest client
var client = new RestClient();

module.exports = {
	
	/** POST /v1.1/search/artifact/
	 * 
	 *   [{"pm": "<pm>", "name": "<name>", "version": "<version>"} [, ...]]
	 * 
	 * Get artifacts in bulk.
	 * 
	 * If an artifact cannot be found it leaves a null in the result array
	 * 
	 * @param pkgs An array of {name: pkgName, version: pkgVersion} objects
	 * @callback to call on completion
	 */
	getNpmArtifacts: function (pkgs, callback) {
		
		var data = [];
		
		for(var i = 0; i < pkgs.length; i++) {
			data.push({"pm": "npm", "name": pkgs[i].name, "version": pkgs[i].version});
		}
		
		var args = {
			data: data,
			headers:{"Content-Type": "application/json"}
		};
		
		var query = ossindex + "/v1.1/search/artifact/";
		client.post(query, args, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined) {
				callback(undefined, data);
			}
			else {
				callback(undefined, []);
			}
		});
	},
	
	/** GET /v1.1/search/artifact/npm/:name/:range
	 * 
	 * Return the artifact that best matches the given package/range
	 * 
	 * @param pkgs A package object: {name: pkgName, version: pkgVersion}
	 * @callback to call on completion
	 */
	getNpmArtifact: function (pkg, callback) {
		var query = ossindex + "/v1.1/search/artifact/npm/" + pkg.name + "/" + pkg.version;
		client.get(query, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined && data.length > 0) {
				callback(undefined, data[0]);
			}
			else {
				callback();
			}
		});
	},
	
	/** GET /v1.1/scm/:id
	 * 
	 * Return the SCM details for the SCM with the specified OSS Index ID.
	 * 
	 * @param List of SCM OSSIndex IDs
	 * @callback to call on completion
	 */
	getScms: function (scmIds, options, callback) {
		if(typeof options == "function" && callback == undefined) {
			callback = options;
			options = undefined;
		}
		var list = scmIds.join(",");
		var query = ossindex + "/v1.1/scm/" + list;
		
		if(options != null) {
			query += "?";
			Object.keys(options).forEach(function(key) {
				query += key + "=" + options[key] + "&";
			});
		}
		client.get(query, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined) {
				callback(undefined, data);
			}
			else {
				callback();
			}
		});
	},
	
	/** GET /v1.1/uri/:host/:path
	 * 
	 * Return the SCM details for the SCM with the specified OSS Index ID.
	 * 
	 * @param uri An SCM URI (eg. https://github.com/jquery/jquery.git)
	 * @callback to call on completion
	 */
	getScmByUri: function (uri, callback) {
		var index = uri.indexOf("://");
		var uriHostPath = uri.substring(index + 3, uri.length);
		client.get(ossindex + "/v1.1/uri/" + uriHostPath, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined) {
				callback(undefined, data);
			}
			else {
				callback();
			}
		});
	},
	
	/** GET /v1.1/project/:id
	 * 
	 * Return the SCM details for the project with the specified OSS Index ID.
	 * 
	 * @param List of project OSSIndex IDs
	 * @callback to call on completion
	 */
	getProjects: function (projectIds, options, callback) {
		if(typeof options == "function" && callback == undefined) {
			callback = options;
			options = undefined;
		}
		var list = projectIds.join(",");
		var query = ossindex + "/v1.1/project/" + list;
		
		if(options != null) {
			query += "?";
			Object.keys(options).forEach(function(key) {
				query += key + "=" + options[key] + "&";
			});
		}
		client.get(query, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined) {
				callback(undefined, data);
			}
			else {
				callback();
			}
		});
	},
	
	/** Given an SCM id, return a list of related CVEs
	 * 
	 * @param
	 */
	getScmVulnerabilities: function(scmId, callback) {
		var query = ossindex + "/v1.1/scm/" + scmId + "/vulnerabilities";
		this.getVulnerabilityList(query, callback);
	},
	
	/** Given a project id, return a list of related CVEs
	 * 
	 * @param
	 */
	getProjectVulnerabilities: function(projectId, callback) {
		var query = ossindex + "/v1.1/project/" + projectId + "/vulnerabilities";
		this.getVulnerabilityList(query, callback);
	},
	
	/** Given a vulnerability URL, collect the vulnerabilities
	 * 
	 * @param
	 */
	getVulnerabilityList: function(query, callback) {
		if(query != undefined) {
			client.get(query, function(data, response){
				// Handle the error response
				if(response.statusCode < 200 || response.statusCode > 299) {
					try {
						var json = JSON.parse(data);
						if(json != undefined && json.error != undefined){
							callback(json);
							return;
						}
					}
					catch(err) {}
					callback({error: "Server error", code: response.statusCode});
					return;
				}
				
				// Otherwise the data is considered good
				if(data != undefined) {
					callback(undefined, data);
				}
				else {
					callback();
				}
			});
		}
		else {
			callback();
		}
	},
	
	/** Given a list of CPE URIs, return a list of CPE details.
	 * 
	 * @param cpeList List of CPE URIs in the form cpe:/part/vendor/product
	 * @callback to call on completion
	 * @param results (internal use only)
	 */
	getCpeListDetails: function (cpeList, callback, results) {
		var that = this;
		if(results == undefined) {
			results = [];
		}
		if(cpeList.length == 0) {
			callback(undefined, results);
		}
		else {
			var cpe = cpeList.shift();
			this.getCpeFromUri(cpe, function(err, cpes) {
				if(err) {
					callback(err);
				}
				if(cpes != undefined) {
					results = results.concat(cpes);
				}
				that.getCpeListDetails(cpeList, callback, results);
			});
		}
	},
	
	/** Given a CPE URI, fetch the CPE details
	 *  
	 *  A CPE URI looks like this: cpe:/part/vendor/product
	 *  
	 * @param cpe A CPE URI in the form cpe:/part/vendor/product
	 * @callback to call on completion
	 */
	getCpeFromUri: function (cpe, callback) {
		var cpeId = cpe.substring(5);
		var tokens = cpeId.split(":");
		this.getCpe(tokens[0], tokens[1], tokens[2], callback);
	},
	
	/** GET /v1.1/cpe/:part/:vendor/:product
	 * 
	 * Given a part, vendor, and product, return the CPE details.
	 * 
	 * @param part field of a CPE (exact match required)
	 * @param vendor field of a CPE (exact match required)
	 * @param product field of a CPE (exact match required)
	 * @callback to call on completion
	 */
	getCpe: function (part, vendor, product, callback) {
		client.get(ossindex + "/v1.1/cpe/" + part + "/" + vendor + "/" + product, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined) {
				callback(undefined, data);
			}
			else {
				callback();
			}
		});
	},
	
	/** GET /v1.1/cve/:id
	 * 
	 * Given a CVE OSS Index ID, get all of the details which includes but
	 * is not limited to
	 *   o Score
	 *   o Impact information
	 *   o Affected CPEs with versions
	 *   o Reference information
	 * 
	 * @param cveIdList A list of CVE OSS Index Ids
	 * @callback to call on completion
	 */
	getCves: function (cveIdList, callback) {
		if(cveIdList.length == 0) {
			callback(undefined, []);
		}
		
		var ids = cveIdList.join(",");
		client.get(ossindex + "/v1.1/cve/" + ids, function(data, response){
			// Handle the error response
			if(response.statusCode < 200 || response.statusCode > 299) {
				try {
					var json = JSON.parse(data);
					if(json != undefined && json.error != undefined){
						callback(json);
						return;
					}
				}
				catch(err) {}
				callback({error: "Server error", code: response.statusCode});
				return;
			}
			
			// Otherwise the data is considered good
			if(data != undefined) {
				callback(undefined, data);
			}
			else {
				callback();
			}
		});
	}
};