const Promise = require('bluebird')
const fs = require('fs')
const glob = require('glob')
const { includes, map, partial } = require('lodash')
const path = require('path')

const globAsync = Promise.promisify(glob)
const writeFileAsync = Promise.promisify(fs.writeFile, fs)
const readFileAsync = Promise.promisify(fs.readFile, fs)

const globOpts = {
  root: process.cwd(),
  ignore: ['**/node_modules/**', '**/coverage/**', '**/coverage-client/**']
}

const existingCopyrightRegexp = /^\s*\/\*[\s\S]*?copyright[\s\S]*?\*\/\s*/i

exports.check = async function check (years, copyrightFile) {
  return loopOverFiles(checkFileForHeader, years, copyrightFile)
}

exports.license = async function license (years, copyrightFile) {
  console.info('Adding Copyright Headers')
  await loopOverFiles(addCopyrightHeaderToFile, years, copyrightFile)
  console.info('Finished Adding Copyright Headers')
}

async function loopOverFiles (fn, years, copyrightFile) {
  const header = (await readFileAsync(
    path.join(__dirname, copyrightFile),
    'utf8'
  )).replace(/{{YEARS}}/g, years)
  const paths = await globAsync('**/*.{js,css}', globOpts)
  const promises = map(paths, partial(fn, header))
  return Promise.all(promises)
}

async function addCopyrightHeaderToFile (header, filePath) {
  let contents = await readFileAsync(filePath, 'utf8')
  if (includes(contents, header)) return null
  console.info('Added Copyright Header to: ', filePath)
  contents = contents.replace(existingCopyrightRegexp, '')
  return writeFileAsync(filePath, `${header}\n${contents}`)
}

async function checkFileForHeader (header, filePath) {
  const contents = await readFileAsync(filePath, 'utf8')
  if (includes(contents, header)) return null
  throw new Error(`License missing in ${filePath}`)
}
