/* global describe, it, beforeEach */

const { exec } = require('child_process')
const expect = require('expect')
const path = require('path')
const snapshot = require('snap-shot')
const fs = require('fs')
const Promise = require('bluebird')
const readFileAsync = Promise.promisify(fs.readFile, fs)

const execAsync = (cmd, opts) => {
  return new Promise((resolve, reject) => {
    exec(cmd, opts, (err, stdout, stderr) => {
      if (!err) return resolve({ stdout, stderr })
      err.stdout = stdout
      err.stderr = stderr
      reject(err)
    })
  })
}

const failure = Symbol('EXPECT_THROW FAIL')
async function expectThrow (asyncFn) {
  try {
    await asyncFn()
    throw failure
  } catch (e) {
    if (e === failure) {
      throw new Error('Expected function to throw but it did not')
    }
    return e
  }
}

describe('Kuali-Utils CLI', () => {
  const cwd = path.resolve(__dirname, 'mock')

  async function cli (args) {
    const output = await execAsync(`node ../../src/cli.js ${args}`, { cwd })
    expect(output.stderr).toBeFalsy()
    return output
  }

  async function cliThrow (args) {
    const output = await expectThrow(async () => {
      await execAsync(`node ../../src/cli.js ${args}`, { cwd })
    })
    expect(output.stdout).toBeFalsy()
    return output
  }

  beforeEach(async () => {
    await execAsync(`rm -rf ${cwd}`)
    await execAsync(`mkdir -p ${cwd}`)
  })

  describe('help command', () => {
    it('kuali-utils', async () => {
      const output = await cli('')
      snapshot(output.stdout)
    })

    it('kuali-utils help', async () => {
      const output = await cli('help')
      snapshot(output.stdout)
    })

    it('kuali-utils h', async () => {
      const output = await cli('h')
      snapshot(output.stdout)
    })

    it('kuali-utils --help', async () => {
      const output = await cli('--help')
      snapshot(output.stdout)
    })

    it('kuali-utils -h', async () => {
      const output = await cli('-h')
      snapshot(output.stdout)
    })

    it('kuali-utils no-idea-what-i-am-doing', async () => {
      const output = await cli('no-idea-what-i-am-doing')
      snapshot(output.stdout)
    })
  })

  describe('version command', () => {
    it('kuali-utils version', async () => {
      const output = await cli('version')
      snapshot(output.stdout)
    })

    it('kuali-utils v', async () => {
      const output = await cli('v')
      snapshot(output.stdout)
    })

    it('kuali-utils --version', async () => {
      const output = await cli('--version')
      snapshot(output.stdout)
    })

    it('kuali-utils -v', async () => {
      const output = await cli('-v')
      snapshot(output.stdout)
    })
  })

  describe('license command', () => {
    it('kuali-utils license - adds the license if a js file has none', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - adds the license if a css file has none', async () => {
      await execAsync(`touch foo.css`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - works if no files are present', async () => {
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - does not add the license to a non js/css file', async () => {
      await execAsync(`touch foo.md`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - does not add the license if it already exists', async () => {
      await execAsync(`cat ../../src/prerelease-copyright-notice.txt | sed 's/{{YEARS}}/2016/g' > foo.js`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
    })

    it('kuali-utils license - uses years specified', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cli('license -y \'2005-2017\'')
      snapshot(output.stdout)
      const fileContents = await readFileAsync(path.resolve(cwd, 'foo.js'), 'utf-8')
      expect(fileContents).toInclude('2005-2017')
    })

    it('kuali-utils license - will use agpl license', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cli('license --years \'2005-2017\' -a')
      snapshot(output.stdout)
      const fileContents = await readFileAsync(path.resolve(cwd, 'foo.js'), 'utf-8')
      expect(fileContents).toInclude('2005-2017')
      expect(fileContents).toInclude('GNU Affero General Public License')
    })

    it('kuali-utils license - will use private license', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cli('license --years 2017 -p')
      snapshot(output.stdout)
      const fileContents = await readFileAsync(path.resolve(cwd, 'foo.js'), 'utf-8')
      expect(fileContents).toInclude(' 2017 ')
      expect(fileContents).toInclude('All Rights Reserved')
      expect(fileContents).toNotInclude('Pre-Release License Agreement')
    })

    it('kuali-utils license - replaces license if it already exists', async () => {
      await execAsync(`echo '/* My weird \n copyright notice that should be removed */' > foo.js`, { cwd })
      const output = await cli('license')
      snapshot(output.stdout)
      const fileContents = await readFileAsync(path.resolve(cwd, 'foo.js'), 'utf-8')
      expect(fileContents).toNotInclude('My weird')
    })
  })

  describe('check command', () => {
    it('kuali-utils check - works if no files are present', async () => {
      const output = await cli('check')
      expect(output.stdout).toBeFalsy()
    })

    it('kuali-utils check - works if a non js/css file is present', async () => {
      await execAsync(`touch foo.md`, { cwd })
      const output = await cli('check')
      expect(output.stdout).toBeFalsy()
    })

    it('kuali-utils check - works if the js/css file has a license', async () => {
      await execAsync(`cat ../../src/prerelease-copyright-notice.txt | sed 's/{{YEARS}}/2016/g' > foo.js`, { cwd })
      const output = await cli('check')
      expect(output.stdout).toBeFalsy()
    })

    it('kuali-utils check - works if the years in the file are incorrect', async () => {
      await execAsync(`cat ../../src/prerelease-copyright-notice.txt | sed 's/{{YEARS}}/2016/g' > foo.js`, { cwd })
      const output = await cliThrow('check -y 2017')
      snapshot(output.stderr)
    })

    it('kuali-utils check - throws an error if a js file has no license', async () => {
      await execAsync(`touch foo.js`, { cwd })
      const output = await cliThrow('check')
      snapshot(output.stderr)
    })

    it('kuali-utils check - throws an error if a css file has no license', async () => {
      await execAsync(`touch foo.css`, { cwd })
      const output = await cliThrow('check')
      snapshot(output.stderr)
    })
  })
})
