/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import cx from 'classnames'
import {
  cloneDeep,
  compact,
  keys,
  map,
  reduce,
  uniq,
  values,
  get,
  findIndex,
  find,
  last
} from 'lodash'
import React, { Component } from 'react'
import PropTypes from 'prop-types'
import { Paper, FontIcon, Button } from 'react-md'
import { utils } from '@kuali/formbot'

import styles from './config.css'
import Builder from './builder'
import { detectIE } from './utils'
const { gatherGadgetsComplex, findGadgetsInProgDiscFromList } = utils

export default class FormbotConfig extends Component {
  static displayName = 'FormbotConfig'

  static propTypes = {
    className: PropTypes.string,
    context: PropTypes.any,
    Formbot: PropTypes.object.isRequired,
    mode: PropTypes.string.isRequired,
    onChange: PropTypes.func.isRequired,
    onClickSettings: PropTypes.func.isRequired,
    onError: PropTypes.func,
    progDisc: PropTypes.bool.isRequired,
    template: PropTypes.object.isRequired,
    trashed: PropTypes.array.isRequired
  }

  constructor (props) {
    super(props)
    this.state = { error: null }
  }

  handleCloseError = () => {
    this.setState({ error: null })
  }
  handleOnError = errors => {
    if (this.props.onError) return this.props.onError(errors)
    return this.setState({ error: errors })
  }

  onMove = (gadget, target, direction, parentId) => {
    const { template: originalTemplate } = this.props
    const template = cloneDeep(originalTemplate)
    const gadgetPath = utils.findGadgetPathById(template, gadget.id)
    const gadgetPathIndex = parseInt(last(gadgetPath.split('.')), 10)
    const gadgetParentPath = gadgetPath
      .split('.')
      .slice(0, -1)
      .join('.')

    const sourceChildren = get(template, gadgetParentPath, [])
    sourceChildren.splice(gadgetPathIndex, 1)
    // If there is a target use that
    if (target) {
      const targetPath = utils.findGadgetPathById(template, target.id)
      const targetParentPath = targetPath
        .split('.')
        .slice(0, -1)
        .join('.')

      const targetChildren = get(template, targetParentPath, [])
      const targetIndex = findIndex(
        targetChildren,
        item => item.id === target.id
      )
      targetChildren.splice(targetIndex + direction, 0, gadget)
    }

    // if there is no target but there is a parentId add it to the parentId
    if (!target && parentId) {
      const targetPath = utils.findGadgetPathById(template, parentId)
      const targetChildren = get(template, `${targetPath}.children`)
      targetChildren.push(gadget)
    }

    return this.props.onChange(template.id, template)
  }

  onDelete = (gadgetIdToDelete, isDeleted) => {
    const { template, trashed: trash, onChange } = this.props
    const gadgetInstances = gatherGadgetsComplex(template, null, trash)
    const gadgetInstance = find(gadgetInstances, { id: gadgetIdToDelete })
    const trashed = isDeleted ? template : null
    if (!gadgetInstance.parent) {
      onChange({}, null, trashed)
    }
    // Don't delete gadgets that are used by, or have children used by others in prog. disclosure)
    const gadgetsUsedInProgressiveDisclosure = findGadgetsInProgDiscFromList(
      gadgetInstance,
      gadgetInstances
    )

    if (gadgetsUsedInProgressiveDisclosure.length > 0) {
      this.handleOnError(gadgetsUsedInProgressiveDisclosure)
      return
    }

    const newParent = cloneDeep(gadgetInstance.parent)
    const childIndex = findIndex(
      gadgetInstance.parent.children,
      child => child.id === gadgetIdToDelete
    )
    if (childIndex !== -1) newParent.children.splice(childIndex, 1)
    onChange(gadgetInstance.parent.id, newParent, trashed)
  }

  render () {
    const {
      context,
      Formbot,
      className,
      template,
      mode,
      progDisc,
      trashed,
      onChange
    } = this.props
    const { onClickSettings } = this.props
    const gadgets = utils.gatherGadgetsById(template)
    const data = reduce(
      gadgets,
      (memo, g) => {
        const gadgetTmpl = Formbot.getGadget(g.type)
        if (!gadgetTmpl) return memo
        memo[g.formKey] = gadgetTmpl.sampleValue || gadgetTmpl.defaultValue
        return memo
      },
      {}
    )
    const templateKeys = map(values(gadgets), 'formKey')
    const trashKeys = map(trashed, 'formKey')
    const presetKeys = keys(Formbot.context.presets)
    const takenKeys = uniq(
      compact([...templateKeys, ...trashKeys, ...presetKeys])
    )
    const classes = cx(className, { 'ie-fixes': detectIE() })
    const errorClasses = cx(className, styles.errorMessage, {
      [styles['errorMessage-show']]: this.state.error
    })
    return (
      <div className={classes}>
        <div className={errorClasses}>
          <Paper className={styles.errorPaper}>
            <FontIcon className={styles.errorIcon}>cancel</FontIcon>
            <div className={styles.errorText}>
              <h3>Gadget cannot be deleted.</h3>
              <p>
                The following gadgets/fields have progressive disclosure
                dependencies
              </p>
              <ul>
                {map(this.state.error, message => (
                  <li>
                    "{get(message, 'progDiscGadget.label')}"
                    <emphasis> is dependent on </emphasis>
                    "{get(message, 'targetGadget.label')}"
                  </li>
                ))}
              </ul>
            </div>
            <Button icon onClick={this.handleCloseError}>
              close
            </Button>
          </Paper>
        </div>
        {mode === 'view' &&
          Formbot.render('edit', template, trashed, data, {
            context,
            progDisc,
            takenKeys,
            onChange,
            onClickSettings,
            onError: this.handleOnError
          })}
        {mode === 'edit' && (
          <Builder
            template={template}
            trashed={trashed}
            onMove={this.onMove}
            onChange={onChange}
            onDelete={this.onDelete}
            onClickSettings={onClickSettings}
            gadgets={get(Formbot, 'context.gadgets', {})}
          />
        )}
      </div>
    )
  }
}
