/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import renderer from '../renderer'
import mFormBot from '../../../test-utils/mock-form-bot'
import { clone } from 'lodash'

describe('Renderer Tests', () => {
  const tMode = 'testMode'
  const tTemplate = { id: 'test' }
  const tProps = {
    progDisc: true,
    onChange: jest.fn(),
    onError: jest.fn(),
    onClickSettings: jest.fn()
  }

  it('should render an empty form if no template.id', () => {
    const testBot = {
      renderEmptyForm: jest.fn()
    }

    renderer(testBot, tMode, {})
    expect(testBot.renderEmptyForm).toHaveBeenCalledWith(testBot, tMode, {})
  })

  describe('renderGadget', () => {
    it('should call the correct render', () => {
      const formBot = clone(mFormBot)
      const customGadgetRenderer = jest.fn()
      formBot.getGadget = jest.fn(() => ({
        customGadgetRenderer,
        defaultValue: 'default'
      }))

      renderer(formBot, tMode, tTemplate, {})
      expect(customGadgetRenderer).toHaveBeenCalledWith(
        formBot,
        tMode,
        tTemplate,
        expect.any(Object)
      )
    })
  })

  describe('progressiveDisclosure', () => {
    let tTemplate

    beforeEach(() => {
      tTemplate = {
        id: true,
        progressiveDisclosure: {
          type: 'any',
          parts: [
            {
              formKey: 'test'
            }
          ]
        }
      }
    })

    it('should return defaults if no progressiveDisclosure', () => {
      tTemplate.progressiveDisclosure = false
      const formBot = clone(mFormBot)
      const customGadgetRenderer = jest.fn()
      formBot.getGadget = jest.fn(() => ({
        customGadgetRenderer,
        defaultValue: 'default'
      }))

      renderer(formBot, tMode, tTemplate, {})

      expect(customGadgetRenderer).toHaveBeenCalledWith(
        formBot,
        tMode,
        tTemplate,
        expect.objectContaining({
          shouldShow: true,
          shouldShowConfig: true
        })
      )
    })

    it('should return shouldShow as true', () => {
      const formBot = clone(mFormBot)
      const props = clone(tProps)
      const customGadgetRenderer = jest.fn()
      const testGadgetInstances = [
        {
          formKey: 'test'
        }
      ]
      const testGadget = {
        customGadgetRenderer,
        defaultValue: 'default',
        progressiveDisclosure: {
          check: jest.fn(() => true)
        }
      }

      formBot.getGadget = jest.fn(() => testGadget)

      renderer(formBot, tMode, tTemplate, {}, testGadgetInstances, props)

      expect(customGadgetRenderer).toHaveBeenCalledWith(
        formBot,
        tMode,
        tTemplate,
        expect.objectContaining({
          shouldShow: true,
          shouldShowConfig: true
        })
      )
    })
  })

  describe('addRemoveMeFn', () => {
    it('should not return removeMe', () => {
      const props = clone(tProps)
      const formBot = clone(mFormBot)
      const customGadgetRenderer = jest.fn()

      formBot.getGadget = jest.fn(() => ({
        customGadgetRenderer
      }))

      renderer(formBot, tMode, tTemplate, {}, {}, props)

      expect(customGadgetRenderer.mock.calls[0][3]).not.toHaveProperty(
        'removeMe',
        expect.any(Function)
      )
    })
  })
})
