/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

/* eslint import/max-dependencies: 0 */

import cx from 'classnames'
import { camelCase, cloneDeep, includes, noop } from 'lodash'
import React, { Component } from 'react'
import PropTypes from 'prop-types'
import { DragSource } from 'react-dnd'
import Button from 'react-md/lib/Buttons'
import FontIcon from 'react-md/lib/FontIcons'
import TextField from 'react-md/lib/TextFields'

import styles from './gadget-renderer.css'

export default function renderGadget (Formbot, mode, template, options) {
  let gadget = options.originalRender(Formbot, mode, template, {
    ...options,
    shouldShow: true,
    onChange: noop
  })
  const gadgetDefinition = Formbot.getGadget(template.type)
  if (!template.formKey && !gadgetDefinition.layout) {
    gadget = (
      <NoFormKey
        setKey={label => {
          let formKey = camelCase(label)
          for (let i = 1; includes(options.takenKeys, formKey); ++i) {
            formKey = camelCase(label) + i
          }
          options.onChange({ ...template, formKey, label })
        }}
      />
    )
  }
  return (
    <DraggableGadget
      key={template.id}
      options={options}
      gadget={gadget}
      template={template}
    />
  )
}

class _DraggableGadget extends Component {
  static displayName = 'DraggableGadget'

  static propTypes = {
    className: PropTypes.string,
    connectDragPreview: PropTypes.func.isRequired,
    connectDragSource: PropTypes.func.isRequired,
    gadget: PropTypes.node.isRequired,
    isDragging: PropTypes.bool.isRequired,
    options: PropTypes.object.isRequired
  }

  render () {
    const {
      gadget,
      options,
      connectDragPreview,
      connectDragSource,
      isDragging
    } = this.props
    const { shouldShowConfig, shouldShow, onClickSettings, removeMe } = options
    if (!shouldShowConfig) return null
    const wrapperStyles = cx(styles.configContainer, this.props.className, {
      [styles.shouldHide]: !shouldShow,
      [styles.isDragging]: isDragging
    })
    return connectDragPreview(
      <div className={wrapperStyles}>
        <div className={styles.configIcons}>
          {connectDragSource(
            <span className={cx(styles.draggerContainer, styles.smallIcon)}>
              <FontIcon className={styles.dragger}>drag_handle</FontIcon>
            </span>
          )}
          <Button
            icon
            onClick={onClickSettings}
            style={{ width: 16, height: 16, padding: 0 }}
            className={styles.smallIcon}
          >
            settings
          </Button>
          <Button
            icon
            onClick={removeMe}
            style={{ width: 16, height: 16, padding: 0 }}
            className={styles.smallIcon}
          >
            delete
          </Button>
        </div>
        {gadget}
      </div>
    )
  }
}

const gadgetSource = {
  beginDrag (props) {
    return {
      data: cloneDeep(props.template),
      cancelUpdate: true
    }
  },
  endDrag (props, monitor) {
    if (monitor.didDrop()) {
      props.options.removeMe()
    }
  }
}

function collect (connect, monitor) {
  return {
    connectDragSource: connect.dragSource(),
    connectDragPreview: connect.dragPreview(),
    isDragging: monitor.isDragging()
  }
}

export const DraggableGadget = DragSource('Gadget', gadgetSource, collect)(
  _DraggableGadget
)

class NoFormKey extends Component {
  static displayName = 'NoFormKey'

  static propTypes = {
    setKey: PropTypes.func.isRequired
  }

  constructor () {
    super()
    this.state = {
      key: '',
      error: null
    }
  }

  handleSaveOnEnter = e => {
    if (e.keyCode === 13) this.handleSave()
  }

  handleSave = () => {
    const { setKey } = this.props
    const { key } = this.state
    if (!key) {
      this.setState({ error: 'Label must not be blank.' })
      return
    }
    setKey(key)
  }

  render () {
    const { key, error } = this.state
    return (
      <div className={styles.noFormKey}>
        <TextField
          error={error}
          errorText={error}
          label='Label'
          placeholder='What do you want to call this field?'
          onChange={val => this.setState({ key: val })}
          onKeyDown={this.handleSaveOnEnter}
          fullWidth
          value={key}
        />
        <Button primary raised label='Save' onClick={this.handleSave} />
      </div>
    )
  }
}
