/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import { filter, map, mapValues, identity, reduce } from 'lodash'

import renderGadget, { buildHeadingParts } from './gadget-renderer'
import renderConfigGadget from './gadget-renderer-config'
import renderEmptyForm from './gadget-renderer-empty'
import renderer from './renderer'
import { gatherGadgetsComplex } from './utils'

export default function (options = {}) {
  const Formbot = {
    renderGadget,
    buildHeadingParts,
    renderConfigGadget,
    renderEmptyForm,
    options
  }

  const context = (Formbot.context = {
    progressiveDisclosures: [],
    decorators: [],
    gadgets: {},
    presets: {}
  })

  Formbot.registerGadget = (key, gadgetDefinition) => {
    if (context.gadgets[key]) {
      const msg = `A gadget is already registered at key ${key}.`
      throw Error(`DUPLICATE_GADGET_KEY: ${msg}`)
    }
    gadgetDefinition.meta = {
      icon: 'extension',
      category: 'Element',
      ...(gadgetDefinition.meta || {})
    }
    context.gadgets[key] = gadgetDefinition
  }

  Formbot.unregisterGadget = key => {
    delete context.gadgets[key]
  }

  Formbot.registerPreset = (key, type, lbl, data = {}, meta = {}) => {
    if (context.presets[key]) {
      const msg = `A preset is already registered at key ${key}.`
      throw Error(`DUPLICATE_PRESET_KEY: ${msg}`)
    }
    if (!context.gadgets[type]) {
      throw Error(`GADGET_NOT_FOUND: There is no gadget registered as ${type}.`)
    }
    context.presets[key] = {
      data: {
        ...data,
        formKey: key,
        type
      },
      meta: {
        icon: context.gadgets[type].meta.icon,
        ...meta,
        lbl
      }
    }
  }

  Formbot.middlewareTransformer = identity

  Formbot.getGadget = key => context.gadgets[key] || context.gadgets.NotFound

  Formbot.registerProgressiveDisclosure = progressiveDisclosure =>
    context.progressiveDisclosures.push(progressiveDisclosure)

  Formbot.getProgressiveDisclosures = () => [...context.progressiveDisclosures]

  Formbot.registerDecorator = (pattern, val) =>
    context.decorators.push({ pattern, val })

  Formbot.getDecorators = _pattern => {
    const matching = filter(context.decorators, ({ pattern }) => {
      for (const key in pattern) {
        if (!(key in _pattern) || _pattern[key] !== pattern[key]) return false
      }
      return true
    })
    return map(matching, 'val')
  }

  Formbot.decorate = (
    decorators,
    parts,
    props,
    gadgetDefinition,
    createComponent
  ) =>
    reduce(
      decorators,
      (memo, dec) => {
        const decorations = dec(props, gadgetDefinition)
        return mapValues(memo, (part, key) =>
          (decorations[key] || identity)(part, createComponent)
        )
      },
      parts
    )

  Formbot.render = (mode, template, trashed, data, props = {}) => {
    const gadgetInstances = gatherGadgetsComplex(template, null, trashed)
    const toRender = props.multipageNum
      ? template.children[props.multipageNum - 1]
      : template
    return renderer(Formbot, mode, toRender, data, gadgetInstances, props)
  }

  return Formbot
}
