/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import { cloneDeep, find, map } from 'lodash'
import React, { Component } from 'react'
import PropTypes from 'prop-types'
import {
  FlatButton,
  Icon,
  IconButton,
  Form,
  FieldGroup,
  TextField,
  Label
} from '@kuali/kuali-ui'
import shortid from 'shortid'

import styles from './keyed-list.css'

export default class KeyedList extends Component {
  static displayName = 'KeyedList'

  static propTypes = {
    onChange: PropTypes.func.isRequired,
    value: PropTypes.array.isRequired
  }

  addValue (key, lbl) {
    const { value, onChange } = this.props
    const newValue = cloneDeep(value)
    const singleVal = find(newValue, { key })
    if (!singleVal) return
    singleVal.lbl = lbl
    onChange(newValue)
  }

  addItem () {
    const { value, onChange } = this.props
    const newValue = [...value, { key: shortid.generate(), lbl: '' }]
    onChange(newValue)
  }

  handleKeyDown (e) {
    if (e.keyCode === 13) {
      e.preventDefault()
      this.addItem()
    }
  }

  removeValue (i) {
    const { value, onChange } = this.props
    onChange([...value.slice(0, i), ...value.slice(i + 1, value.length)])
  }

  render () {
    const { value } = this.props
    return (
      <div className={styles['keyed-list']}>
        <Form showGrid={!!value.length}>
          {map(value, ({ key, lbl }, i) => (
            <FieldGroup
              key={`keyed-list-row-${key}`}
              className={styles['selection-option']}
            >
              <div className='kuali-form__cell'>
                <Label htmlFor={key}>{`Selection ${i + 1}`}</Label>
                <TextField
                  autoFocus
                  id={key}
                  placeholder='Enter a value for this selection'
                  onChange={val => this.addValue(key, val)}
                  onKeyDown={e => this.handleKeyDown(e)}
                  value={lbl}
                />
              </div>
              <div className='kuali-form__cell'>
                <IconButton onClick={() => this.removeValue(i)}>
                  <Icon name='delete' />
                </IconButton>
              </div>
            </FieldGroup>
          ))}
        </Form>
        <FlatButton
          className={styles['add-item']}
          label='Add Item'
          onClick={() => this.addItem()}
        >
          <Icon name='add' />
        </FlatButton>
      </div>
    )
  }
}
