/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import {
  compact,
  every,
  filter,
  find,
  get,
  isFunction,
  keyBy,
  map,
  noop,
  reduce,
  set,
  some
} from 'lodash'

export default function fbRender (
  Formbot,
  mode,
  template,
  data,
  gadgetInstances,
  props = {}
) {
  if (!template.id) {
    return Formbot.renderEmptyForm(Formbot, mode, props)
  }
  if (!template.root) template.root = true
  return renderGadget(Formbot, gadgetInstances, mode, data, props)(template)
}

function renderGadget (Formbot, gadgetInstances, mode, data, props) {
  function renderGadgetInternal (template) {
    const middlewares = Formbot.middlewareTransformer([
      progressiveDisclosure,
      attachTakenKeys,
      renderChildren,
      bindDataHandlers,
      linkDependentData,
      chooseRenderer
    ])
    const args = [
      Formbot,
      mode,
      template,
      data,
      props,
      gadgetInstances,
      renderGadgetInternal
    ]
    const options = reduce(
      middlewares,
      (memo, mw) => {
        return {
          ...memo,
          ...mw(...args)
        }
      },
      {
        fbRender: fbRender.bind(null, Formbot),
        context: props.context,
        noPad: props.noPad
      }
    )

    return options.render(Formbot, mode, template, options)
  }
  return renderGadgetInternal
}

function progressiveDisclosure (
  Formbot,
  _mode,
  template,
  data,
  props,
  gadgetInstances
) {
  if (!template.progressiveDisclosure) {
    return { shouldShow: true, shouldShowConfig: true }
  }
  const combineFn = template.progressiveDisclosure.type === 'any' ? some : every
  const parts = filter(template.progressiveDisclosure.parts, p => p.formKey)
  const shouldShow = combineFn(parts, part => {
    const { formKey } = part
    const progDiscDef = find(Formbot.getProgressiveDisclosures(), { formKey })
    if (progDiscDef) return progDiscDef.check(get(data, formKey), part.data)
    const gadgetInstance = find(gadgetInstances, { formKey })
    const gadgetDefinition = Formbot.getGadget(gadgetInstance.type)
    const actual = get(data, formKey) || gadgetDefinition.defaultValue
    const expected = part.data
    return gadgetDefinition.progressiveDisclosure.check(actual, expected)
  })
  return { shouldShow, shouldShowConfig: shouldShow || props.progDisc }
}

function attachTakenKeys (_Formbot, _mode, _template, _data, props) {
  if (!props.takenKeys) return {}
  return { takenKeys: props.takenKeys }
}

function renderChildren (
  _Formbot,
  _mode,
  template,
  _data,
  _props,
  _gadgetInstances,
  _renderGadget
) {
  if (!template.children) return {}
  return {
    children: compact(map(template.children, _renderGadget)),
    childrenTemplates: template.children
  }
}

function bindDataHandlers (Formbot, _mode, template, data, props) {
  const obj = {}
  const gadgetDefinition = Formbot.getGadget(template.type)
  obj.value = data[template.formKey] || gadgetDefinition.defaultValue
  obj.onChange = noop
  if (props.onChange) {
    obj.onChange = (newVal, path) => {
      const newPath =
        path || path === 0 ? `${template.formKey}.${path}` : template.formKey
      return props.onChange(newPath, newVal)
    }
  }
  return obj
}

function linkDependentData (
  Formbot,
  _mode,
  template,
  data,
  _props,
  gadgetInstances
) {
  const { getDependentKeys } = Formbot.getGadget(template.type)
  if (!isFunction(getDependentKeys)) return {}
  const dependentKeys = getDependentKeys(template.details)
  const dependentData = reduce(
    dependentKeys.data,
    (prev, key) => ({
      ...prev,
      [key]: data[key]
    }),
    {}
  )
  let dataGadgets = filter(gadgetInstances, 'formKey')
  dataGadgets = keyBy(dataGadgets, 'formKey')
  const dependentTemplate = reduce(
    dependentKeys.template,
    (prev, key) => {
      set(prev, key, get(dataGadgets, key))
      return prev
    },
    {}
  )
  return {
    dependent: { data: dependentData, template: dependentTemplate }
  }
}

function chooseRenderer (Formbot, _mode, template, _data, props) {
  const GadgetDef = Formbot.getGadget(template.type)
  const originalRender = GadgetDef.customGadgetRenderer || Formbot.renderGadget
  const render = originalRender
  return { originalRender, render }
}
