/* Copyright © 2016 Kuali, Inc. - All Rights Reserved
 * You may use and modify this code under the terms of the Kuali, Inc.
 * Pre-Release License Agreement. You may not distribute it.
 *
 * You should have received a copy of the Kuali, Inc. Pre-Release License
 * Agreement with this file. If not, please write to license@kuali.co.
 */

import React from 'react'
import Section from '../lib/section/manifest'
import * as GadgetTests from '../../../test-utils/gadget'

describe('Section', () => {
  GadgetTests.structure(Section, {
    layout: true,
    meta: {
      lbl: 'Section',
      icon: 'tab',
      category: 'Layout'
    }
  })

  GadgetTests.render(Section, { view: {}, edit: {} })

  describe('Meta', () => {
    it('should generate an initial template', () => {
      const template = Section.meta.initialTemplate()
      expect(template).toMatchObject({
        children: []
      })
    })
  })

  describe('Custom Renderers', () => {
    const mode = 'view'
    const template = {
      id: 'template id',
      label: 'template label',
      type: 'view',
      children: []
    }
    const opts = {
      shouldShow: true,
      children: [<div key='child'>this is a child</div>],
      context: { name: 'a context' }
    }

    let Formbot
    beforeEach(() => {
      Formbot = {
        decorate: jest.fn(),
        getGadget: jest.fn(),
        getDecorators: jest.fn(),
        options: {}
      }
    })

    describe('Custom Gadget Renderer', () => {
      const GadgetRenderer = Section.customGadgetRenderer
      it('should return null if shouldShow is false', () => {
        expect(
          GadgetRenderer(Formbot, null, null, { shouldShow: false })
        ).toBeNull()
      })

      it('should return a regular component if no decorators are provided', () => {
        Formbot.getDecorators.mockReturnValue([])
        const template = { id: 'template id', label: 'template label' }
        const gadget = GadgetRenderer(Formbot, null, template, opts)
        expect(gadget).toMatchSnapshot()
      })

      it('should properly decorate a component if they are provided via Formbot', () => {
        const decorators = ['something']
        const mockGadget = <div>A Gadget</div>
        Formbot.getDecorators.mockReturnValue(decorators)
        Formbot.getGadget.mockReturnValue(mockGadget)
        Formbot.decorate
          .mockReturnValueOnce({
            label: template.label,
            gadget: mockGadget
          })
          .mockReturnValueOnce({
            component: <div>This is a fake component</div>
          })

        const gadget = GadgetRenderer(Formbot, mode, template, opts)
        expect(Formbot.getDecorators).toHaveBeenCalledWith({
          type: template.type,
          mode
        })

        const props = {
          children: opts.children,
          context: opts.context,
          template
        }

        const calls = Formbot.decorate.mock.calls
        // first call from within create component
        const firstCall = calls[0]
        expect(firstCall).toEqual([
          decorators,
          { label: expect.anything(), gadget: expect.anything() },
          props,
          mockGadget
        ])
        expect(firstCall[1].label).toMatchSnapshot()
        expect(firstCall[1].gadget).toMatchSnapshot()

        // second call to decorate component
        const secondCall = calls[1]
        expect(secondCall).toEqual([
          decorators,
          { component: expect.anything() },
          props,
          mockGadget,
          expect.any(Function)
        ])
        expect(secondCall[1].component).toMatchSnapshot()

        // Ensure that the formbot returns the component
        // returned from the second decorate call
        expect(gadget).toMatchSnapshot()
      })
    })
  })
})
