const chalk = require('chalk')
const commander = require('commander')
const open = require('open')
const pkg = require('../package.json')
const CircleCI = require('./util/circleci')
const github = require('./util/github')
const util = require('util')

async function publish () {
  commander
    .option(
      '-v, --npmversion [version]',
      'semver or version string (see https://docs.npmjs.com/cli/version)'
    )
    .option('-j, --circlejob [job]', 'Circle job to execute')
    .option('-o, --owner <owner>', 'Github organization (default: KualiCo)')
    .option('-p, --project <prj>', 'Github project')
    .option('-b, --branch [branch]', 'Github branch (defaults to master)')
    .option(
      '-P, --publishbranch [publishbranch]',
      'Github branch containing published version (defaults to ci_publish)'
    )
    .option('--verbose', 'Output more detail')
    .parse(process.argv)

  const {
    owner = 'KualiCo',
    project,
    branch = 'master',
    publishbranch: publishBranch = 'ci_publish',
    npmversion: npmVersion = 'patch',
    circlejob: circleJob = 'publish'
  } = commander

  console.log(chalk.green(`--- kuali-build-helper --- (v${pkg.version})`))

  try {
    const gh = await github.getAuthenticatedClient()
    const circleci = await CircleCI.getAuthenticatedClient()

    const result = await circleci.startBuild({
      username: owner,
      project,
      branch,
      body: {
        build_parameters: {
          PUBLISH_VERSION: npmVersion,
          CIRCLE_JOB: circleJob
        }
      }
    })

    if (result.failed) {
      result.messages.forEach(msg => {
        console.error(chalk.red(msg))
      })
    }

    console.log(
      chalk.green(`CircleCI build [${result.build_num}] triggered`),
      chalk.yellow('waiting...')
    )

    setInterval(async () => {
      try {
        const status = await circleci.getBuild({
          username: owner,
          project,
          build_num: result.build_num
        })

        if (status.lifecycle === 'finished') {
          if (status.outcome === 'success') {
            console.log(
              chalk.green(`CircleCI build [${result.build_num}] succeeded`),
              chalk.green('Creating pull request...')
            )

            try {
              const res = await gh.pullRequests.create({
                owner,
                repo: project,
                title: 'Publish new version',
                head: publishBranch,
                base: branch
              })

              if (res.meta.status === '201 Created') {
                const pull = res.meta.location.split('/').pop()
                open(`https://github.com/${owner}/${project}/pull/${pull}`)
                console.log(chalk.green(`Pull request ${pull} opened`))
                process.exit(0)
              } else {
                console.error(
                  chalk.red(`Failed to open PR with status: ${res.meta.status}`)
                )
                process.exit(1)
              }
            } catch (e2) {
              console.error(chalk.red('Unexpected error creating pull request'))
              console.log(e2)
            }
          } else {
            console.error(
              chalk.red(
                `CircleCI build [${result.build_num}] finished with status: ${
                  status.status
                }`
              )
            )
            process.exit(1)
          }
        }
      } catch (e1) {
        console.error(chalk.red('Unexpected error checking status of build'))
        console.log(e1)
      }
    }, 5000)
  } catch (err) {
    console.log('-----')
    console.log(util.inspect(err, { depth: 9 }))
    console.log('-----')
  }
}

publish()
