/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* eslint-disable import/first */
jest.mock('../../../sys/utils')

import DomUtils from '../../../sys/dom_utils'
import KfsUtils from '../../../sys/utils'
import React from 'react'
import { setupContainer } from '../../../sys/test-utils'
import { shallow } from 'enzyme'
import { URLS } from '../../../sys/constants'

describe('header', () => {
  let Header
  let wrapper

  beforeAll(() => {
    setupContainer('header')
  })

  beforeEach(async () => {
    global.stayOnPage = jest.fn()
    KfsUtils.apiCall = jest.fn().mockReturnValue(Promise.resolve({ data: {} }))
    KfsUtils.getUrlPathPrefix = jest.fn().mockReturnValue('')
    KfsUtils.buildBackdoorIdAppender = jest.fn().mockReturnValue(jest.fn())
    jest.spyOn(DomUtils, 'replaceWindowLocation').mockReturnThis()
    Header = require('../header')
    Header = Header.default
    wrapper = await shallow(<Header />)
  })

  afterAll(() => {
    jest.unmock('../../../sys/utils')
  })

  it('should match the snapshot', () => {
    const user = { firstName: 'keisha' }
    wrapper.setState({ user })
    expect(wrapper).toMatchSnapshot()
  })

  it('should render header items', () => {
    const headerItems = ['action-list', 'doc-search', 'user', 'menu']
    headerItems.forEach(item => {
      expect(wrapper.find(`[data-test-id="header-${item}"]`)).toHaveLength(1)
    })
  })

  it('should trigger stayOnPage on sign out', () => {
    const signoutUrl = 'http://foo.bar'
    wrapper.setState({ preferences: { signoutUrl } })

    const link = wrapper.find('[data-test-id="header-sign-out"]')
    expect(link.prop('href')).toEqual(signoutUrl)

    link.simulate('click')
    expect(global.stayOnPage).toHaveBeenCalled()
  })

  describe('menu', () => {
    it('should render menu links', () => {
      const menuItems = ['foo', 'bar', 'baz']
      const menu = menuItems.map(item => {
        return { label: item, link: `http://${item}.${item}` }
      })

      wrapper.setState({ preferences: { menu } })
      menuItems.forEach(item => {
        const selector = `[data-test-id="header-menu-link-${item}"]`
        expect(wrapper.find(selector)).toHaveLength(1)
      })
    })

    it('should not render "About" if no version specified in preferences"', () => {
      expect(wrapper.find('[data-test-id="header-about"]')).toHaveLength(0)
    })

    it('should render "About" if version specified in preferences', () => {
      wrapper.setState({ preferences: { versions: 1 } })
      expect(wrapper.find('[data-test-id="header-about"]')).toHaveLength(1)
    })

    it('should show the about modal if the link is clicked', () => {
      setupContainer('remodal-content')
      const div = document.createElement('div')
      div.className = 'remodal-content'
      document.body.appendChild(div)
      wrapper.setState({
        preferences: { versions: { first: 'The First Version' } }
      })
      wrapper.find('[data-test-id="header-about"]').simulate('click')
      expect(div.innerHTML).toMatchSnapshot()
    })
  })

  describe('logo', () => {
    it('should render logo from logo url preference', () => {
      const logoUrl = 'http://foo.bar'
      wrapper.setState({ preferences: { logoUrl } })
      expect(wrapper.find('.navbar-brand img').prop('src')).toEqual(logoUrl)
    })

    it('should append logoUrl to prefix if data is present in preference', () => {
      const logoUrl = 'foo?data:bar.baz'
      KfsUtils.getUrlPathPrefix = jest
        .fn()
        .mockReturnValue('http://monsters-tst/')
      wrapper.setState({ preferences: { logoUrl } })
      expect(wrapper.find('.navbar-brand img').prop('src')).toEqual(
        'http://monsters-tst/foo?data:bar.baz'
      )
    })
  })

  describe('Back Door Header', () => {
    const selector = '#test-header'
    const impersonationLabelSelector = '[data-test-id="impersonation-label"]'

    it('should not display if in prod', () => {
      wrapper.setState({ environment: { prodMode: true } })
      expect(wrapper.find(selector)).toHaveLength(0)
    })

    it('should display if in test env', () => {
      expect(wrapper.find(selector)).toHaveLength(1)
    })

    it('should display "impersonating" if backdoor id is present', () => {
      wrapper.setState({ backdoorId: 1 })
      const label = wrapper.find(impersonationLabelSelector)

      expect(label).toHaveLength(1)
      expect(label.text()).toMatch(/Impersonating: 1\s/)
    })

    it('should not display "impersonating" if backdoor id is not present', () => {
      const label = wrapper.find(impersonationLabelSelector)
      expect(label).toHaveLength(1)
      expect(label.text()).toMatch(/\s/)
    })

    it('selecting backdoor logout triggers a logout', () => {
      KfsUtils.apiCall.mockClear()
      wrapper.setState({ backdoorId: 1 })
      wrapper.find('[data-test-id="backdoor-logout"]').simulate('click')
      expect(KfsUtils.apiCall.mock.calls[0][0]).toBe(
        URLS.API.SYS.BACKDOOR_LOGOUT
      )
    })
  })
})
