/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import React from 'react'
import { shallow } from 'enzyme'
import SidebarMenu from '../SidebarMenu'

describe('Sidebar Menu', () => {
  let props
  beforeEach(() => {
    props = {
      id: 'some id',
      label: 'This is a label',
      style: { margin: '10px' },
      className: 'someClass',
      'aria-label': 'Some aria label',
      'aria-expanded': true
    }
  })

  it('should render without issue', () => {
    const wrapper = shallow(
      <SidebarMenu {...props}>
        <div>These are some childre</div>
      </SidebarMenu>
    )
    expect(wrapper).toMatchSnapshot()
  })

  describe('Close click handler', () => {
    it('should call onCloseClick if the close button is clicked', () => {
      props.onCloseClick = jest.fn()
      const wrapper = shallow(<SidebarMenu {...props} />)
      wrapper.find('button.close').simulate('click')
      expect(props.onCloseClick).toHaveBeenCalledWith(props.label, props.id)
    })

    it('should not crash if a onCloseClick is null', () => {
      const wrapper = shallow(<SidebarMenu {...props} />)
      expect(() => {
        wrapper.find('button.close').simulate('click')
      }).not.toThrow()
    })
  })

  describe('Header vs Label', () => {
    it('should render the label within a header tag if present', () => {
      const wrapper = shallow(<SidebarMenu {...props} />)
      expect(wrapper.find('h3').text()).toEqual(props.label)
    })

    it('should render a header in favor of label if provided', () => {
      props.header = <div className='myHeader'>My Header</div>
      const wrapper = shallow(<SidebarMenu {...props} />)
      expect(wrapper.find('h3')).toHaveLength(0)
      expect(wrapper.find('div.myHeader').text()).toEqual('My Header')
    })
  })
})
