/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import classnames from 'classnames'
import PropTypes from 'prop-types'
import { searchLinkGroups } from '../../../sys/search_utils'
import SearchResults from '../SearchResults'
import SelectableLink from './SelectableLink.js'
import SelectableLinkGroup from './SelectableLinkGroup.js'
import SidebarMenu from '../SidebarMenu'
import styles from './style.css'
import {
  determineMenuWidth,
  determineMenuWidthClass,
  findLongestLinkSectionCount,
  getLinkCounts
} from '../../../sys/sidebar_utils'
import { isEqual, pick } from 'lodash'
import React, { Component } from 'react'

export default class FavoritesLinkConfig extends Component {
  static propTypes = {
    id: PropTypes.string,
    handleClick: PropTypes.func,
    addFavorite: PropTypes.func,
    removeFavorite: PropTypes.func,
    header: PropTypes.node,
    label: PropTypes.string,
    expanded: PropTypes.bool,
    linkGroups: PropTypes.array,
    layoutActivePanel: PropTypes.func
  }

  constructor (props) {
    super(props)
    this.state = {
      expandedLabel: null,
      searchResults: null,
      searchTerm: '',
      menuWidth: 1
    }
  }

  componentDidMount () {
    if (this.searchInput) {
      this.searchInput.focus()
    }
  }

  componentDidUpdate (prevProps, prevState) {
    if (!isEqual(prevProps.linkGroups, this.props.linkGroups)) {
      this.updateSearchResults()
    }
    // If the layout went from non-searching to searching (or vice versa), re-layout the sidebar panel
    if (!!prevState.searchResults !== !!this.state.searchResults) {
      this.props.layoutActivePanel()
    }
  }

  handleToggleFavorite = link => {
    const { removeFavorite, addFavorite } = this.props
    const favoriteMethod = link.favorite ? removeFavorite : addFavorite
    favoriteMethod(link.navLinkId)
  }

  handleExpandedChanged = label => {
    this.setState(prevState => ({
      expandedLabel: prevState.expandedLabel !== label ? label : null
    }))
  }

  handleSearch = event => {
    const searchTerm = event.target.value
    this.setState(
      {
        searchTerm
      },
      this.updateSearchResults
    )
  }

  updateSearchResults = () => {
    const { searchTerm } = this.state
    const shouldSearch = searchTerm.length > 2
    const linkGroups = this.getFilteredLinkGroups()
    this.setState({
      searchResults: shouldSearch
        ? searchLinkGroups(searchTerm, linkGroups)
        : null
    })
  }

  getFilteredLinkGroups = () => {
    const { linkGroups, checkedLinkFilters } = this.props
    return linkGroups.map(linkGroup => ({
      ...linkGroup,
      links: pick(linkGroup.links, checkedLinkFilters)
    }))
  }

  clearSearch = () => {
    this.setState({
      searchTerm: '',
      searchResults: null
    })
  }

  getMenuWidthClass = () => {
    const linkGroups = this.getFilteredLinkGroups()
    const { searchResults } = this.state
    const { linkCount, headerCount } = searchResults
      ? getLinkCounts(searchResults)
      : findLongestLinkSectionCount(linkGroups)
    return determineMenuWidthClass(linkCount, headerCount)
  }

  getTextLabel = elements => {
    return elements.map(element => element.textContent).join()
  }

  render () {
    const { id, expanded, handleClick, header, label } = this.props
    const { expandedLabel, searchTerm, searchResults } = this.state
    const linkGroups = this.getFilteredLinkGroups()

    const searchBar = (
      <div className={classnames('search', styles['favorites-search'])}>
        <input
          aria-label='favorites menu search'
          aria-controls='favorites-search-results'
          aria-describedby='favorites-search-input-description'
          id='search-input'
          type='search'
          placeholder='Find Links'
          className={styles['favorites-search-input']}
          onChange={this.handleSearch}
          value={searchTerm}
          onFocus={this.handleSearch}
        />
        <span
          className='screen-reader-only'
          id='favorites-search-input-description'
          aria-hidden='true'
        >
          You will be notified of results.
        </span>
        <span
          className='glyphicon glyphicon-remove remove'
          aria-hidden='true'
          onClick={this.clearSearch}
        />
      </div>
    )

    const content = searchResults ? (
      <SearchResults
        id='favorites-search-results'
        searchTerm={searchTerm}
        results={searchResults}
        expanded={expanded}
        linkGroups={linkGroups}
        renderResultItem={(link, type, group) => (
          <SelectableLink
            key={`${type}-${link.navLinkId}`}
            toggleFavorite={this.handleToggleFavorite}
            link={link}
          />
        )}
      />
    ) : (
      linkGroups.map(linkGroup => {
        const { label, links } = linkGroup
        const { linkCount, headerCount } = getLinkCounts(links)
        const groupWidth = determineMenuWidth(linkCount, headerCount)
        return (
          <SelectableLinkGroup
            linksStyle={{ columnCount: groupWidth }}
            key={label}
            label={label}
            links={links}
            expanded={label === expandedLabel}
            onExpandClicked={this.handleExpandedChanged}
            toggleFavorite={this.handleToggleFavorite}
          />
        )
      })
    )

    return (
      <SidebarMenu
        id={id}
        className={classnames([
          styles['favorites-menu'],
          this.getMenuWidthClass()
        ])}
        header={header}
        label={label}
        onCloseClick={handleClick}
        style={{ display: 'flex', flexDirection: 'column' }}
        expanded={expanded}
      >
        {searchBar}
        <div className={styles['favorites-body']}>{content}</div>
      </SidebarMenu>
    )
  }
}
