/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import classnames from 'classnames'
import { getFirstElementBySelector } from '../../../sys/dom_utils'
import PropTypes from 'prop-types'
import SelectableLink from './SelectableLink.js'
import styles from './style.css'
import { get, isEmpty } from 'lodash'
import React, { Component } from 'react'

export default class SelectableLinkGroup extends Component {
  static propTypes = {
    label: PropTypes.string,
    links: PropTypes.object,
    expanded: PropTypes.bool,
    toggleFavorite: PropTypes.func,
    onExpandClicked: PropTypes.func
  }

  handleToggleClick = _ => {
    const { label, onExpandClicked } = this.props
    if (onExpandClicked) {
      onExpandClicked(label)
    }
  }

  componentDidUpdate (prevProps) {
    // Focus heading when section is expanded
    if (this.props.expanded && !prevProps.expanded) {
      const firstHeader = getFirstElementBySelector('h4', this.links)
      if (firstHeader) {
        firstHeader.focus()
      }
    }
  }

  buildLinkCategories = () => {
    const { links, toggleFavorite } = this.props

    const getCategoryLinks = category => {
      const categoryLinks = get(links, category, []).map(link => (
        <SelectableLink
          key={link.label}
          toggleFavorite={toggleFavorite}
          link={link}
        />
      ))
      return !isEmpty(categoryLinks) ? [
        <h4 tabIndex={-1} key={category}>{category}</h4>,
        ...categoryLinks
      ] : []
    }

    return [
      ...getCategoryLinks('activities'),
      ...getCategoryLinks('reference'),
      ...getCategoryLinks('administration')
    ]
  }

  render () {
    const { label, expanded, linksStyle } = this.props

    const linkCategories = this.buildLinkCategories()

    return (
      <div>
        <button
          ref={ref => {
            this.tabButton = ref
          }}
          className={styles['link-group-toggle-button']}
          onClick={this.handleToggleClick}
          aria-expanded={expanded}
          aria-label={label}
        >
          <span className={styles['link-group-header-label']}>{label}</span>
          <i className='material-icons'>
            {expanded ? 'keyboard_arrow_up' : 'keyboard_arrow_down'}
          </i>
        </button>
        <div
          ref={ref => {
            this.links = ref
          }}
          className={classnames(
            ['favorites-links-container', styles['link-group-body']],
            {
              [styles['hidden']]: !expanded
            }
          )}
          style={linksStyle}
        >
          {expanded && (
            linkCategories.length > 0 ? linkCategories : 'No links found.'
          )}
        </div>
      </div>
    )
  }
}
