/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import React from 'react'
import SelectableLink from '../SelectableLink'
import { shallow } from 'enzyme'

describe('Selectable Link', () => {
  let props
  beforeEach(() => {
    props = {
      link: {
        label: 'label',
        favorite: false
      },
      toggleFavorite: jest.fn()
    }
  })

  it('should render without issue', () => {
    const wrapper = shallow(<SelectableLink {...props} />)
    expect(wrapper).toMatchSnapshot()
  })

  it('should render the input as checked', () => {
    props.link.favorite = true
    const wrapper = shallow(<SelectableLink {...props} />)
    expect(wrapper.find('input').prop('checked')).toBe(true)
  })

  it('should render the input as unchecked', () => {
    const wrapper = shallow(<SelectableLink {...props} />)
    expect(wrapper.find('input').prop('checked')).toBe(false)
  })

  it('should render the input as unchecked when favorite is undefined', () => {
    props.link.favorite = undefined
    const wrapper = shallow(<SelectableLink {...props} />)
    expect(wrapper.find('input').prop('checked')).toBe(false)
  })

  describe('Toggling value', () => {
    it('should call toggleFavorite when clicked', () => {
      const wrapper = shallow(<SelectableLink {...props} />)
      wrapper.find('input').simulate('change')
      expect(props.toggleFavorite).toHaveBeenCalledWith(props.link)
    })

    it('should call toggleFavorite when enter is pressed', () => {
      const wrapper = shallow(<SelectableLink {...props} />)
      const input = wrapper.find('input')

      input.simulate('keyPress', { keyCode: 13 })
      expect(props.toggleFavorite).toHaveBeenCalledWith(props.link)

      props.toggleFavorite.mockClear()
      input.simulate('keyPress', { which: '13' })
      expect(props.toggleFavorite).toHaveBeenCalledWith(props.link)
    })

    it('should not calls toggleFavorite on non-enter key presses', () => {
      const wrapper = shallow(<SelectableLink {...props} />)
      wrapper.find('input').simulate('keyPress', { keyCode: 22 })
      expect(props.toggleFavorite).not.toHaveBeenCalled()
    })
  })
})
