/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import React from 'react'
import SelectableLinkGroup from '../SelectableLinkGroup'
import { shallow } from 'enzyme'

describe('Selectable Link Group', () => {
  let props
  beforeEach(() => {
    props = {
      label: 'group label',
      links: {
        administration: [
          {
            label: 'link label 4',
            favorite: false
          }
        ],
        activities: [
          {
            label: 'link label',
            favorite: false
          },
          {
            label: 'link label 2',
            favorite: false
          }
        ],
        reference: [
          {
            label: 'link label 3',
            favorite: false
          }
        ]
      },
      expanded: false,
      toggleFavorite: jest.fn(),
      onExpandClicked: jest.fn()
    }
  })

  it('should render without issue', () => {
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    expect(wrapper).toMatchSnapshot()
  })

  it('should render without issue with no links', () => {
    props.links = {
      activities: []
    }
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    expect(wrapper).toMatchSnapshot()
  })

  it('should render in order of Activities, Reference, Administration', () => {
    props.expanded = true
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    const headers = wrapper.find('h4')
    expect(headers.length).toBe(3)
    expect(headers.at(0).text()).toBe('activities')
    expect(headers.at(1).text()).toBe('reference')
    expect(headers.at(2).text()).toBe('administration')
  })

  it('should render collapsed when expanded prop is false', () => {
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    expect(wrapper.find('button').prop('aria-expanded')).toBe(false)
    expect(wrapper.find('i.material-icons').text()).toBe('keyboard_arrow_down')
    expect(wrapper.find('.hidden')).toHaveLength(1)
  })

  it('should render expanded when expanded prop is true', () => {
    props.expanded = true
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    expect(wrapper.find('button').prop('aria-expanded')).toBe(true)
    expect(wrapper.find('i.material-icons').text()).toBe('keyboard_arrow_up')
    expect(wrapper.find('.hidden')).toHaveLength(0)
  })

  it('should call onExpandClicked when link group toggle button clicked', () => {
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    wrapper.find('button').simulate('click')
    expect(props.onExpandClicked).toHaveBeenCalledWith(props.label)
  })

  it('should not throw an exception if onExpandClicked is undefined when link group toggle button clicked', () => {
    props.onExpandClicked = undefined
    const wrapper = shallow(<SelectableLinkGroup {...props} />)
    expect(() => {
      wrapper.find('button').simulate('click')
    }).not.toThrow()
  })
})
