/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import classnames from 'classnames'
import FavoriteButton from './favorites/FavoriteButton'
import FavoritesLinkGroupItem from './favorites/FavoritesLinkGroupItem'
import KfsUtils from '../../sys/utils.js'
import LinkFilter from './LinkFilter.jsx'
import LinkGroupItem from './LinkGroupItem'
import React from 'react'
import { searchLinkGroups } from '../../sys/search_utils'
import SearchResults from './SearchResults'
import SidebarError from './SidebarError.jsx'
import SidebarMenu from './SidebarMenu'
import SidebarWaiting from './SidebarWaiting.jsx'
import { URLS } from '../../sys/constants'
import UserFavorites from '../../sys/user_favorites'
import UserPrefs from '../../sys/user_preferences'
import {
  addClassToElement,
  closest,
  elementHasClass,
  getElementOffset,
  getElementOuterHeight,
  getFirstElementBySelector,
  removeClassFromElement,
  setStyleRule,
  toggleClass
} from '../../sys/dom_utils'
import {
  convertLink,
  determineMenuWidthClass,
  getLinkCounts,
  localStorageGet,
  localStorageRemove,
  localStorageSet
} from '../../sys/sidebar_utils.js'
import { cloneDeep, extend, filter, get, pull, sortBy } from 'lodash'

class Sidebar extends React.Component {
  constructor () {
    super()
    const sidebarOut = !elementHasClass(
      document.getElementById('sidebar'),
      'collapsed'
    )

    this.state = {
      loadingData: true,
      loadError: false,
      principalName: '',
      institutionPreferences: {},
      userPreferences: {
        checkedLinkFilters: ['activities', 'reference', 'administration']
      },
      expandedLinkGroup: '',
      expandedSearch: false,
      search: '',
      searchResults: undefined,
      backdoorId: '',
      sidebarOut
    }
  }

  buildFavorites = preferences => {
    let { favorites, linkGroups } = preferences

    if (!favorites || linkGroups.length <= 0) {
      return
    }

    if (get(linkGroups, '0.favorites')) {
      linkGroups.shift()
    }

    let favoritesGroup = {
      links: { activities: [], administration: [], reference: [] },
      label: 'My Favorites',
      favorites: true
    }

    let links = this.getLinks(preferences)
    links = links.map(link =>
      extend(link, { favorite: favorites.includes(link.navLinkId) })
    )
    favoritesGroup.links.activities = sortBy(
      filter(links, { favorite: true, subsection: 'activities' }),
      'label'
    )
    favoritesGroup.links.administration = sortBy(
      filter(links, { favorite: true, subsection: 'administration' }),
      'label'
    )
    favoritesGroup.links.reference = sortBy(
      filter(links, { favorite: true, subsection: 'reference' }),
      'label'
    )

    linkGroups.splice(0, 0, favoritesGroup)
  }

  getLinks = preferences => {
    let links = []
    if (preferences && preferences.linkGroups) {
      preferences.linkGroups.forEach(linkGroup => {
        links = links.concat(
          (linkGroup.links.activities || []).map(link =>
            extend(link, { subsection: 'activities' })
          )
        )
        links = links.concat(
          (linkGroup.links.administration || []).map(link =>
            extend(link, { subsection: 'administration' })
          )
        )
        links = links.concat(
          (linkGroup.links.reference || []).map(link =>
            extend(link, { subsection: 'reference' })
          )
        )
      })
    }
    return links
  }

  removeFavorite = navLinkId => {
    UserFavorites.removeUserFavorite(
      this.state.principalName,
      navLinkId,
      () => {
        let institutionPreferences = cloneDeep(
          this.state.institutionPreferences
        )
        pull(institutionPreferences.favorites, navLinkId)
        this.buildFavorites(institutionPreferences)
        this.refreshSearchResults({ institutionPreferences })
      },
      error => {
        console.log('error removing favorite: ' + error)
        this.setState({ loadError: true })
      }
    )
  }

  addFavorite = navLinkId => {
    UserFavorites.addUserFavorite(
      this.state.principalName,
      navLinkId,
      () => {
        let institutionPreferences = cloneDeep(
          this.state.institutionPreferences
        )
        institutionPreferences.favorites.push(navLinkId)
        this.buildFavorites(institutionPreferences)
        this.refreshSearchResults({ institutionPreferences })
      },
      error => {
        console.log('error add favorite: ' + error)
        this.setState({ loadError: true })
      }
    )
  }

  componentDidMount () {
    return Promise.all([
      UserPrefs.getUserPreferences(),
      UserPrefs.getBackdoorId()
    ])
      .then(([userPreferences, backdoorId]) => {
        this.setState({ userPreferences })
        return backdoorId
      })
      .then(backdoorId => {
        return UserPrefs.getPrincipalName()
          .then(principalName => {
            let found = false
            this.setState({ principalName })
            const sessionId = KfsUtils.getKualiSessionId()
            let preferencesString = localStorageGet('institutionPreferences')

            if (preferencesString !== null) {
              const prefs = JSON.parse(preferencesString)
              this.buildFavorites(prefs)
              if (
                prefs.sessionId === sessionId &&
                prefs.principalName === principalName
              ) {
                found = true
                this.setState({
                  loadingData: false,
                  loadError: false,
                  backdoorId,
                  institutionPreferences: prefs
                })
              } else {
                localStorageRemove('institutionPreferences')
              }
            }

            if (!found) {
              const path = `${URLS.API.SYS.INSTITUTION_LINKS}/${principalName}`
              KfsUtils.apiCall(path, {
                method: 'GET'
              })
                .then(result => {
                  const preferences = result.data
                  this.buildFavorites(preferences)
                  this.setState({
                    loadingData: false,
                    loadingError: false,
                    backdoorId,
                    institutionPreferences: preferences
                  })
                  preferences.sessionId = sessionId
                  localStorageSet(
                    'institutionPreferences',
                    JSON.stringify(preferences)
                  )
                })
                .catch(error => {
                  console.error(get(error, 'response.status'), error.message)
                  this.setState({ loadError: true })
                })
            }
          })
          .catch(error => {
            console.error('Error retrieving principalName: ' + error.message)
            this.setState({ loadError: true })
          })
      })
      .catch(error => {
        console.error(error)
      })
  }

  componentDidUpdate () {
    this.layoutActivePanel()
  }

  layoutActivePanel = () => {
    const testEnvDiv = getFirstElementBySelector('div.body.test-env')
    const HEADER_HEIGHT = testEnvDiv ? 100 : 60

    let newClass
    let activePanelId =
      KfsUtils.buildKeyFromLabel(this.state.expandedLinkGroup) + '-menu'
    let activePanel = document.getElementById(activePanelId)
    if (
      activePanel &&
      activePanel.parentNode &&
      !elementHasClass(activePanel.parentNode, 'search') &&
      getElementOffset(activePanel)
    ) {
      let panelHeight = getElementOuterHeight(activePanel)
      let sidebarScroll = document.getElementById('sidebar-scroll')
      let linkGroups = document.getElementById('linkgroups')
      let sidebarScrollRefresh = getFirstElementBySelector(
        '#sidebar-scroll>div.refresh'
      )
      if (
        getElementOuterHeight(sidebarScroll) <
        getElementOuterHeight(linkGroups) +
          getElementOuterHeight(sidebarScrollRefresh, true)
      ) {
        let sidebarMiddle = (getElementOuterHeight(sidebarScroll) + 170) / 2
        let topPosition = sidebarMiddle - panelHeight / 2
        setStyleRule(activePanel, 'top', `${topPosition + HEADER_HEIGHT}px`)
        addClassToElement(activePanel, 'floating')
      } else {
        let activeLink = getFirstElementBySelector('#linkgroups .active')
        let sideBarDiv = getFirstElementBySelector('#sidebar>div')
        let sidebarHeight = getElementOuterHeight(sideBarDiv)
        let distanceFromTop =
          getElementOffset(activeLink).top - getElementOffset(sideBarDiv).top
        let distanceFromBottom = sidebarHeight - distanceFromTop
        let centerY = getElementOuterHeight(activeLink) / 2 + panelHeight / 2
        if (
          distanceFromBottom < distanceFromTop &&
          distanceFromBottom < panelHeight
        ) {
          if (
            distanceFromTop > distanceFromBottom &&
            distanceFromBottom > centerY
          ) {
            newClass = 'flowCenter'
          } else {
            newClass = 'flowUp'
          }
        } else if (
          distanceFromBottom < panelHeight &&
          distanceFromBottom > centerY
        ) {
          newClass = 'flowCenter'
        }
        if (newClass) {
          addClassToElement(activePanel, newClass)
          if (newClass === 'flowCenter') {
            let centeredTop =
              distanceFromTop +
              getElementOuterHeight(activeLink) / 2 -
              panelHeight / 2
            if (centeredTop < 0) {
              centeredTop = 10
            }

            setStyleRule(activePanel, 'top', `${centeredTop + HEADER_HEIGHT}px`)
          } else if (newClass === 'flowUp') {
            setStyleRule(
              activePanel,
              'bottom',
              `${distanceFromBottom - getElementOuterHeight(activeLink) + 1}px`
            )
          }
        } else {
          setStyleRule(
            activePanel,
            'top',
            `${distanceFromTop + HEADER_HEIGHT}px`
          )
        }
      }
    }
    if (activePanel) {
      activePanel.scrollTop = 0
    }
  }

  modifyLinkFilter = type => {
    let userPreferences = cloneDeep(this.state.userPreferences)
    let newChecked = userPreferences.checkedLinkFilters
    let index = newChecked.indexOf(type)
    if (index === -1) {
      newChecked.push(type)
    } else {
      newChecked.splice(index, 1)
    }
    this.setState({ userPreferences: userPreferences })
    UserPrefs.putUserPreferences(userPreferences)
  }

  toggleLinkGroup = (label, sublinksId) => {
    let sidebar = this
    let toggleLinkGroupClick = (event) => {
      if (
        !closest(event.target, '.sublinks') &&
        !closest(event.target, 'li.panel.active') &&
        !closest(event.target, '#linkFilter') &&
        !closest(event.target, '.favorite-button')
      ) {
        sidebar.closeLinkGroup(sublinksId, sidebar)
      }
    }

    const refocusSidebar = sublinks => {
      // Place focus back on menu
      const menuItem = getFirstElementBySelector('a', sublinks.parentElement)
      if (menuItem) {
        menuItem.focus()
      }
    }

    let toggleLinkGroupKeyUp = () => {
      if (event.keyCode === 27) {
        let sublinks = document.getElementById(sublinksId)
        refocusSidebar(sublinks)
        sidebar.closeLinkGroup(sublinksId, sidebar)
      }
    }
    // Close the link group if it's open
    if (this.state.expandedLinkGroup === label) {
      this.setState({ expandedLinkGroup: '' })
      let sublinks = document.getElementById(sublinksId)
      this.removeLinkGroupStyling(sublinks)
      getFirstElementBySelector('html').removeEventListener(
        'click',
        toggleLinkGroupClick
      )
      document.removeEventListener('keyup', toggleLinkGroupKeyUp)
      refocusSidebar(sublinks)
    } else {
      this.setState({ expandedLinkGroup: label })
      addClassToElement(document.getElementById('content-overlay'), 'visible')
      getFirstElementBySelector('html').addEventListener(
        'click',
        toggleLinkGroupClick
      )
      document.addEventListener('keyup', toggleLinkGroupKeyUp)
    }
  }

  removeLinkGroupStyling = sublinks => {
    removeClassFromElement(sublinks, 'flowUp')
    removeClassFromElement(sublinks, 'flowCenter')
    setStyleRule(sublinks, 'bottom', 'inherit')
    setStyleRule(sublinks, 'top', 'inherit')
    removeClassFromElement(
      document.getElementById('content-overlay'),
      'visible'
    )
  }

  closeLinkGroup = (sublinksId, sidebar) => {
    let sublinks = document.getElementById(sublinksId)
    this.removeLinkGroupStyling(sublinks)
    sidebar.setState({ expandedLinkGroup: '' })
  }

  toggleSidebar = () => {
    toggleClass(document.getElementById('menu-toggle'), 'rotated')
    toggleClass(document.getElementById('sidebar'), 'collapsed')
    toggleClass(getFirstElementBySelector('main.content'), 'fullwidth')

    if (!this.state.sidebarOut) {
      getFirstElementBySelector('li.search>input').focus()
    }
    this.setState({ sidebarOut: !this.state.sidebarOut, expandedLinkGroup: '' })
  }

  refreshLinks = () => {
    const path = `${URLS.API.SYS.INSTITUTION_LINKS}/${this.state.principalName}`

    this.setState({ loadingData: true })

    KfsUtils.apiCall(path, {
      method: 'GET',
      headers: {
        'cache-control': 'must-revalidate'
      }
    })
      .then(result => {
        const preferences = result.data
        this.buildFavorites(preferences)
        this.setState({
          loadingData: false,
          loadError: false,
          institutionPreferences: preferences
        })
        preferences.sessionId = KfsUtils.getKualiSessionId()
        localStorageSet('institutionPreferences', JSON.stringify(preferences))
      })
      .catch(error => {
        console.error(get(error, 'response.status'), error.message)
        this.setState({ loadError: true })
      })
  }

  autocompleteSearchEventHandler = event => {
    if (
      !closest(event.target, 'li.panel.active') &&
      !closest(event.target, '#linkFilter') &&
      !closest(event.target, '.favorite-button')
    ) {
      removeClassFromElement(
        getFirstElementBySelector('li.panel.active'),
        'active'
      )
      removeClassFromElement(
        document.getElementById('content-overlay'),
        'visible'
      )
      this.setState({ expandedSearch: false })
    }
  }

  autocompleteSearch = event => {
    let searchString = event.target.value
    let expandedSearch = searchString.length > 2

    let newState = {
      search: searchString,
      expandedSearch,
      expandedLinkGroup: ''
    }

    if (!expandedSearch) {
      removeClassFromElement(
        document.getElementById('content-overlay'),
        'visible'
      )
      getFirstElementBySelector('html').removeEventListener(
        'click',
        this.autocompleteSearchEventHandler
      )
      this.setState(newState)
    } else {
      addClassToElement(document.getElementById('content-overlay'), 'visible')
      getFirstElementBySelector('html').addEventListener(
        'click',
        this.autocompleteSearchEventHandler
      )

      this.refreshSearchResults(newState)
    }
  }

  refreshSearchResults = updatedState => {
    const newState = { ...this.state, ...updatedState }
    const searchString = newState.search || ''
    const expandedSearch = searchString.length > 2
    if (expandedSearch) {
      newState.searchResults = searchLinkGroups(
        searchString,
        newState.institutionPreferences.linkGroups
      )
    }
    this.setState(newState)
  }

  clearSearch = () => {
    getFirstElementBySelector('li.search>input').focus()
    this.setState({
      search: '',
      searchResults: undefined,
      expandedSearch: false
    })
    removeClassFromElement(
      document.getElementById('content-overlay'),
      'visible'
    )
    getFirstElementBySelector('html').removeEventListener(
      'click',
      this.autocompleteSearchEventHandler
    )
  }

  closeSearch = () => {
    removeClassFromElement(
      getFirstElementBySelector('li.search.panel.active div.sublinks.active'),
      'active'
    )
    removeClassFromElement(
      getFirstElementBySelector('li.search.panel.active'),
      'active'
    )
    removeClassFromElement(
      document.getElementById('content-overlay'),
      'visible'
    )
    getFirstElementBySelector('html').removeEventListener(
      'click',
      this.autocompleteSearchEventHandler
    )
    this.setState({ expandedSearch: false })
  }

  handleConfigureModeChange = _ => {
    this.layoutActivePanel()
  }

  buildLinkComponents () {
    const {
      institutionPreferences,
      backdoorId,
      expandedLinkGroup,
      userPreferences
    } = this.state

    const { linkGroups = [] } = institutionPreferences
    return linkGroups.map((group, groupIndex) => {
      const { label } = group
      const expanded = expandedLinkGroup === label
      const props = {
        menuId: `${KfsUtils.buildKeyFromLabel(label)}-menu`,
        group,
        key: groupIndex,
        index: groupIndex,
        backdoorId: backdoorId,
        addFavorite: this.addFavorite,
        handleClick: this.toggleLinkGroup,
        removeFavorite: this.removeFavorite,
        expanded,
        className: classnames(['panel', 'list-item'], { active: expanded }),
        checkedLinkFilters: userPreferences.checkedLinkFilters,
        layoutActivePanel: this.layoutActivePanel
      }

      if (group.favorites) {
        return (
          <FavoritesLinkGroupItem
            {...props}
            onConfigureModeChange={this.handleConfigureModeChange}
            linkGroups={cloneDeep(linkGroups)}
          />
        )
      } else {
        return <LinkGroupItem {...props} />
      }
    })
  }

  handleHomeClick = () => {
    if (typeof stayOnPage === 'function') {
      stayOnPage()
    }
  }

  handleEscapeFromSearchResult = event => {
    if (event.keyCode === 27) {
      getFirstElementBySelector('li.search>input').focus()
    }
  }

  render () {
    const {
      expandedSearch,
      loadError,
      loadingData,
      search,
      sidebarOut,
      userPreferences,
      searchResults
    } = this.state

    if (sidebarOut && loadError) {
      return <SidebarError />
    }
    if (sidebarOut && loadingData) {
      return <SidebarWaiting />
    }

    let rootPath = KfsUtils.getUrlPathPrefix()
    const links = this.buildLinkComponents()

    if (!sidebarOut) {
      addClassToElement(document.getElementById('sidebar'), 'collapsed')
    }

    const backdoorIdAppender = KfsUtils.buildBackdoorIdAppender(
      this.state.backdoorId
    )
    const { linkCount, headerCount } = getLinkCounts(searchResults)
    const menuWidthClass = determineMenuWidthClass(linkCount, headerCount)

    const searchResultsLabel = `${linkCount - headerCount} Results Found`

    return (
      <div>
        <ul id='filters' className='nav list-group'>
          <li id='home-item'>
            <span id='home'>
              <a href={rootPath} onClick={this.handleHomeClick}>
                <span className='fa fa-home home-icon' />Home
              </a>
            </span>
            <span
              id='menu-toggle'
              className={classnames(['glyphicon glyphicon-menu-left'], {
                rotated: !sidebarOut
              })}
              aria-hidden='true'
              onClick={this.toggleSidebar}
            />
          </li>
          <li
            className={classnames(['search', 'list-item', 'panel'], {
              active: expandedSearch
            })}
            role='navigation'
          >
            <input
              aria-label='sidebar menu search'
              aria-controls='sidebar-search-results'
              aria-describedby='sidebar-search-input-description'
              id='search-input'
              type='search'
              placeholder='Search'
              onChange={this.autocompleteSearch}
              value={search}
              onFocus={this.autocompleteSearch}
            />
            <span
              className='screen-reader-only'
              id='sidebar-search-input-description'
              aria-hidden='true'
            >
              You will be notified of results.
            </span>
            <span
              className='glyphicon glyphicon-remove remove'
              aria-hidden='true'
              onClick={this.clearSearch}
            />
            {expandedSearch && (
              <SidebarMenu
                className={menuWidthClass}
                expanded
                id='sidebar-search-results'
                onCloseClick={this.closeSearch}
                label={searchResultsLabel}
                aria-label='search results submenu'
                aria-expanded={expandedSearch}
                header={<h3 aria-hidden>{searchResultsLabel}</h3>}
              >
                <div
                  className='links-container'
                  onKeyUp={this.handleEscapeFromSearchResult}
                >
                  <SearchResults
                    searchTerm={search}
                    results={searchResults}
                    expanded={expandedSearch}
                    linkGroups={this.state.institutionPreferences.linkGroups}
                    renderResultItem={(link, type, group) => {
                      const linkComponent = convertLink(
                        link,
                        type,
                        backdoorIdAppender,
                        null,
                        group
                      )
                      return (
                        <FavoriteButton
                          key={`${type}-${link.navLinkId}`}
                          link={link}
                          linkComponent={linkComponent}
                          group={group}
                          addFavorite={this.addFavorite}
                          removeFavorite={this.removeFavorite}
                        />
                      )
                    }}
                  />
                </div>
              </SidebarMenu>
            )}
          </li>
          <li className='list-item'>
            <LinkFilter
              checkedLinkFilters={userPreferences.checkedLinkFilters}
              modifyLinkFilter={this.modifyLinkFilter}
            />
          </li>
        </ul>
        <div id='sidebar-scroll'>
          <ul id='linkgroups' className='nav list-group'>
            {links}
          </ul>
          <div className='refresh'>
            Missing something?{' '}
            <a href='#d' onClick={this.refreshLinks}>
              <span> Refresh Menu </span>
            </a>{' '}
            to make sure your permissions are up to date.
          </div>
        </div>
      </div>
    )
  }
}

export default Sidebar
