/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import Dropzone from 'react-dropzone'
import { get } from 'lodash'
import KfsUtils from '../../../sys/utils'
import toast from '../../../sys/toast'
import { URLS } from '../../../sys/constants'
import React, { Component } from 'react'

export default class LogoUpload extends Component {
  constructor (props) {
    super(props)

    this.state = {
      logo: '',
      hasChanges: false,
      saveButtonText: 'SAVE CHANGES'
    }

    this.onDrop = this.onDrop.bind(this)
    this.saveChanges = this.saveChanges.bind(this)
  }

  componentDidMount () {
    return KfsUtils.apiCall(URLS.API.SYS.LOGO_CONFIG, {
        method: 'GET'
    }).then(result => {
        this.setState({ logo: get(result, ['data', 'logoUrl'], '') })
    })
    .catch(error => {
        console.error(error.response.status, error.message)
    })
  }

  onDrop (files) {
    const data = new FormData()
    data.append('logo', files[0])

    return KfsUtils.apiCall(URLS.API.SYS.LOGO_CONFIG, {
      method: 'POST',
      data
    }).then(result => {
      this.setState({
        hasChanges: true,
        logo: get(result, ['data', 'logoUrl'], ''),
        error: undefined
      })
    }).catch(error => {
      console.error(error.response.status, error.message)
      toast.error(error.message || 'Upload failed.')
      this.setState({ error: error.message })
    })
  }

  saveChanges () {
    const data = { logoUrl: this.state.logo }

    return KfsUtils.apiCall(URLS.API.SYS.LOGO_CONFIG, {
      method: 'PUT',
      data
    }).then(result => {
      const spanStyle = { color: '#6DA487' }
      this.setState({
        hasChanges: false,
        saveButtonText: (
            <span style={spanStyle}>
              <span className='glyphicon glyphicon-ok' />SAVED
            </span>
          )
      })
      toast.success('Save Successful!')
    }).catch(error => {
      console.error(error.response.status, error.message)
      toast.error(error.message || 'Save failed.')
      this.setState({ error: error.message })
    })
  }

  render () {
    let logoUrl = this.state.logo
    if (
      logoUrl &&
      logoUrl.indexOf('data:') !== 0 &&
      !logoUrl.startsWith('http')
    ) {
      logoUrl = KfsUtils.getUrlPathPrefix() + logoUrl
    }

    let saveDisabled
    let saveButtonClass = 'btn btn-green'
    let saveButtonText = this.state.saveButtonText
    if (!this.state.hasChanges) {
      saveDisabled = true
      saveButtonClass += ' disabled'
    } else {
      saveButtonText = 'SAVE CHANGES'
    }

    return (
      <div>
        <div className='headerarea-small' id='headerarea-small'>
          <h1>Logo Upload</h1>
        </div>

        <div className='logo-upload main'>
          <h4>Preview</h4>
          <div className='preview-box'>
            <img src={logoUrl} height='35px' />
            <span className='logo-right'>Financials</span>
          </div>
          <div className='instructions'>
            If you would like to brand Kuali Financials for your institution, we
            recommend that you upload an image that contains your institution's
            insignia. We require that the image you upload be at least 70 pixels
            tall, be no larger than 100kb, and have a transparent background.
          </div>

          <div className='dropzone'>
            <Dropzone onDrop={this.onDrop} multiple={false}>
              <div>
                Drag &amp; drop logo here or click to select a logo to upload.
              </div>
            </Dropzone>
          </div>

          <div className='error'>{this.state.error}</div>

          <div className='buttonbar'>
            <button
              disabled={saveDisabled}
              className={saveButtonClass}
              onClick={this.saveChanges}
            >
              {saveButtonText}
            </button>
          </div>
        </div>
      </div>
    )
  }
}
