/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
/* eslint-disable import/first */
jest.mock('../../../sys/utils')

import { ButtonActions } from '../../general/button-bar'
import KfsUtils from '../../../sys/utils'
import Lookup from '../lookup'
import React from 'react'
import { setupContainer } from '../../../sys/test-utils'
import { mount, shallow } from 'enzyme'

describe('Lookup', () => {
  let props

  beforeAll(() => {
    setupContainer(['header', 'sidebar', 'page-content'])
  })

  beforeEach(() => {
    global.stayOnPage = jest.fn()
    KfsUtils.apiCall = jest.fn().mockResolvedValue({ data: {} })
    props = {
      getFormTemplate: jest.fn(),
      businessObjectName: 'BatchFile'
    }
  })

  it('should render', () => {
    const wrapper = shallow(<Lookup {...props} />)
    expect(wrapper).toMatchSnapshot()
  })

  describe('Headers', () => {
    it('should render the BO name in the header if available', () => {
      const wrapper = shallow(<Lookup {...props} />)
      expect(wrapper.find('h1').text()).toEqual('Batch File Lookup')
    })

    it('should render Lookup if no BO name is available', () => {
      props.businessObjectName = null
      const wrapper = shallow(<Lookup {...props} />)
      expect(wrapper.find('h1').text()).toEqual('Lookup')
    })
  })

  describe('Button Handlers', () => {
    it('should clear data state when clear is clicked', () => {
      const wrapper = shallow(<Lookup {...props} />)
      wrapper
        .find('ButtonBar')
        .getElement()
        .props.onButtonClick(ButtonActions.Clear)
      expect(wrapper.state('data')).toEqual({})
    })
  })

  describe('Formbot Integration', () => {
    it('should update its data when formbot updates its data', () => {
      const wrapper = mount(<Lookup {...props} />)
      expect(wrapper.state().data).toMatchObject({})
      wrapper.find('FormbotEdit').prop('onChange')('myKey', 'aValue')
      expect(wrapper.state().data).toMatchObject({ myKey: 'aValue' })
    })
  })

  describe('Data Fetching', () => {
    it('should request the right business object on mount', () => {
      shallow(<Lookup {...props} />)
      expect(props.getFormTemplate).toHaveBeenCalledWith(
        props.businessObjectName
      )
    })

    it('should fetch the form definition and update its template upon mount', async () => {
      const template = { id: '123', type: 'Section', children: [] }
      const promise = Promise.resolve(template)
      props.getFormTemplate.mockReturnValue(promise)
      const wrapper = mount(<Lookup {...props} />)
      await promise
      expect(wrapper.state().template).toEqual(template)
    })
  })
})
