/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import axios from 'axios'
import MockAdapter from 'axios-mock-adapter'
import {
  formTemplate,
  lookupURL,
  RANGE_LOWER_BOUND_KEY_PREFIX
} from '../form-template-builder'

describe('formTemplate', () => {
  const boName = 'Foo'
  let axiosMock

  beforeEach(() => {
    axiosMock = new MockAdapter(axios)
  })

  const testTemplateBuilder = (
    controlDesc,
    inputAttributes,
    outputGadgets,
    values
  ) => {
    it(`should create formbot template for ${controlDesc}`, async () => {
      axiosMock.onGet(lookupURL(boName)).reply(200, inputAttributes)
      if (values) {
        axiosMock.onGet(lookupURL(boName) + '/values').reply(200, values)
      }
      const template = await formTemplate(boName)
      expect(template).toEqual({
        type: 'Section',
        id: 'ROOT',
        children: outputGadgets
      })
    })
  }

  testTemplateBuilder(
    'default (text) control',
    [
      {
        name: 'myField',
        label: 'My Field',
        control: {}
      }
    ],
    [
      {
        type: 'Text',
        formKey: 'myField',
        label: 'My Field',
        id: expect.anything(),
        details: {}
      }
    ]
  )

  testTemplateBuilder(
    'datepicker control',
    [
      {
        name: 'myDate',
        label: 'My Date',
        control: {
          datePicker: true
        }
      }
    ],
    [
      {
        type: 'Date',
        formKey: 'myDate',
        label: 'My Date',
        id: expect.anything(),
        details: {}
      }
    ]
  )

  testTemplateBuilder(
    'ranged datepicker control',
    [
      {
        name: 'myDateRange',
        label: 'My Date Range',
        control: {
          datePicker: true,
          ranged: true
        }
      }
    ],
    [
      {
        type: 'Date',
        formKey: `${RANGE_LOWER_BOUND_KEY_PREFIX}_myDateRange`,
        label: 'My Date Range From',
        fieldSize: 'medium',
        id: expect.anything(),
        details: {}
      },
      {
        type: 'Date',
        formKey: 'myDateRange',
        fieldSize: 'medium',
        label: 'My Date Range To',
        id: expect.anything(),
        details: {}
      }
    ]
  )

  testTemplateBuilder(
    'hierarchical multiselect control',
    [
      {
        name: 'myHierarchy',
        label: 'My Hierarchy',
        control: {
          multiselect: true,
          hierarchical: true
        }
      }
    ],
    [
      {
        type: 'Tree',
        formKey: 'myHierarchy',
        label: 'My Hierarchy',
        id: expect.anything(),
        details: {
          nodes: ['foo', 'bar'],
          valueKey: 'value',
          gadgetClassName: 'tree'
        }
      }
    ],
    {
      myHierarchy: ['foo', 'bar']
    }
  )

  it('should only make one request for values even with multiple attributes that expect values', async () => {
    // Tree attributes expect values (nodes), so we will pass two of them to test this
    const attributes = [
      {
        name: 'tree1',
        label: 'Tree 1',
        control: {
          multiselect: true,
          hierarchical: true
        }
      },
      {
        name: 'tree2',
        label: 'Tree 2',
        control: {
          multiselect: true,
          hierarchical: true
        }
      }
    ]
    const values = {
      tree1: ['a', 'b', 'c'],
      tree2: [1, 2, 3]
    }
    const expectedGadgets = [
      {
        type: 'Tree',
        formKey: 'tree1',
        label: 'Tree 1',
        id: expect.anything(),
        details: {
          nodes: ['a', 'b', 'c'],
          valueKey: 'value',
          gadgetClassName: 'tree'
        }
      },
      {
        type: 'Tree',
        formKey: 'tree2',
        label: 'Tree 2',
        id: expect.anything(),
        details: {
          nodes: [1, 2, 3],
          valueKey: 'value',
          gadgetClassName: 'tree'
        }
      }
    ]

    axiosMock.onGet(lookupURL(boName)).reply(200, attributes)
    let valuesRequestedCounter = 0
    axiosMock.onGet(lookupURL(boName) + '/values').reply(() => {
      valuesRequestedCounter++
      return [200, values]
    })
    const template = await formTemplate(boName)
    expect(template).toEqual({
      type: 'Section',
      id: 'ROOT',
      children: expectedGadgets
    })
    expect(valuesRequestedCounter).toBe(1)
  })

  it('should return null for a failed form definition fetch', async () => {
    axiosMock.onGet(lookupURL(boName)).networkError()
    const template = await formTemplate(boName)
    expect(template).toBeUndefined()
  })

  it('should gracefully handle error on fetch form values', async () => {
    const attributes = [
      {
        name: 'myTree',
        label: 'My Tree',
        control: {
          multiselect: true,
          hierarchical: true
        }
      }
    ]
    const expectedGadgets = [
      {
        type: 'Tree',
        formKey: 'myTree',
        label: 'My Tree',
        id: expect.anything(),
        details: {
          gadgetClassName: 'tree',
          valueKey: 'value',
          nodes: undefined
        }
      }
    ]
    axiosMock.onGet(lookupURL(boName)).reply(200, attributes)
    axiosMock.onGet(lookupURL(boName) + '/values').networkError()
    const template = await formTemplate(boName)
    expect(template).toEqual({
      type: 'Section',
      id: 'ROOT',
      children: expectedGadgets
    })
  })
})
