/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import KfsUtils from './utils.js'
import Link from '../components/general/link.jsx'
import React from 'react'
import { flatten, keys, last, map, sortBy, values } from 'lodash'

export function convertLink (
  link,
  type,
  backdoorIdAppender,
  linkWrapper,
  group
) {
  let target = null
  if (
    (link.linkType !== 'kfs' && link.linkType !== 'report') ||
    link.newTarget
  ) {
    target = '_blank'
  }

  const url =
    link.linkType === 'rice' ? backdoorIdAppender(link.link) : link.link
  const key = link.navLinkId
  let linkComponent = (
    <Link
      key={key}
      url={url}
      label={link.label}
      className='list-group-item'
      target={target}
      click={stayOnPage}
    />
  )
  return linkWrapper
    ? linkWrapper(link, linkComponent, key, group)
    : linkComponent
}

export function convertLinks (links, type, backdoorId, linkWrapper, group) {
  if (!links) {
    return ''
  }

  const backdoorIdAppender = KfsUtils.buildBackdoorIdAppender(backdoorId)
  return links.map((link, i) =>
    convertLink(link, type, backdoorIdAppender, linkWrapper, group)
  )
}

export function addHeading (links, type) {
  let newLinks = []
  if (links.length > 0) {
    newLinks = newLinks
      .concat([<h4 key={type + '_label'}>{type}</h4>])
      .concat(links)
  }
  return newLinks
}

export function findLongestLinkSectionCount (linkGroups) {
  const linkCounts = sortBy(
    map(linkGroups, ({ links }) => {
      return getLinkCounts(links)
    }),
    'linkCount'
  )
  return last(linkCounts)
}

export function getLinkCounts (links) {
  const childCount = flatten(values(links)).length
  const headerCount = keys(links).length
  return {
    linkCount: childCount + headerCount,
    headerCount
  }
}

export function determineMenuWidth (linkCount, headingCount) {
  // 1400px is the width at which links in 3rd column start to clip (unzoomed)
  let width = 1
  let mq = window.matchMedia('screen and (min-width: 1400px)')
  if (linkCount > 36 - headingCount) {
    width = mq.matches ? 3 : 2
  } else if (linkCount > 18 - headingCount) {
    width = 2
  }
  return width
}

export const menuWidthToClassName = {
  1: 'menu-small',
  2: 'menu-medium',
  3: 'menu-large'
}

export function determineMenuWidthClass (linkCount, headingCount) {
  return menuWidthToClassName[determineMenuWidth(linkCount, headingCount)]
}

export function localStorageGet (key) {
  let item = null
  try {
    item = localStorage.getItem(key)
  } catch (err) {
    // Ignore the error, some browsers don't support localStorage
  }
  return item
}

export function localStorageRemove (key) {
  try {
    localStorage.removeItem(key)
  } catch (err) {
    // Ignore the error, some browsers don't support localStorage
  }
}

export function localStorageSet (key, value) {
  try {
    localStorage.setItem(key, value)
  } catch (err) {
    // Ignore the error, some browsers don't support localStorage
  }
}

const SidebarUtils = {
  addHeading,
  determineMenuWidth,
  determineMenuWidthClass,
  findLongestLinkSectionCount,
  localStorageGet,
  localStorageRemove,
  localStorageSet,
  menuWidthToClassName
}

export default SidebarUtils
