/*
 * The Kuali Financial System, a comprehensive financial management system for higher education.
 *
 * Copyright 2005-2018 Kuali, Inc.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import classnames from 'classnames'
import installReactA11y from '../../utils/react-a11y'
import KfsUtils from '../../utils/utils'
import Link from '../link/link.jsx'
import React from 'react'
import styles from './style.css'
import { ToastContainer } from 'react-toastify'
import { URLS } from '../../utils/constants'
import {
  addClassToElement,
  getFirstElementBySelector,
  replaceWindowLocation,
  setStyleRule
} from '../../utils/dom_utils'

class Header extends React.Component {
  constructor () {
    super()
    this.state = { preferences: {}, user: {}, environment: {}, backdoorId: '' }
  }

  componentDidMount () {
    const {
      BACKDOOR,
      SYSTEM_ENVIRONMENT,
      INSTITUTION_PREFERENCES,
      USER
    } = URLS.API.SYS

    KfsUtils.apiCall(USER, {})
      .then(result => {
        this.setState({ user: result.data })
      }).catch(error => {
        console.error(error.response.status, error.message)
      })

    KfsUtils.apiCall(INSTITUTION_PREFERENCES, {})
      .then(result => {
        this.setState({ preferences: result.data })
      }).catch(error => {
        console.error(error.response.status, error.message)
      })

    KfsUtils.apiCall(SYSTEM_ENVIRONMENT, {})
      .then(result => {
        this.setState({ environment: result.data })

        if (
          this.state.environment &&
          this.state.environment.prodMode === false
        ) {
          const body = getFirstElementBySelector('.body')
          addClassToElement(body, 'test-env')
          setStyleRule(
            document.getElementById('test-header'),
            'display',
            'block'
          )
          installReactA11y()
        }
      }).catch(error => {
        console.error(error.response.status, error.message)
      })

    KfsUtils.apiCall(BACKDOOR, {})
      .then(result => {
        this.setState({ backdoorId: result.data.backdoorId })
      }).catch(error => {
        console.error(error.response.status, error.message)
      })
  }

  handleBackDoorEnter = e => {
    if (e.charCode == 13) {
      document.getElementById('backdoorLoginButton').click()
    }
  }

  backdoorLogin = event => {
    event.preventDefault()
    const backdoorEl = document.getElementById('backdoorId')
    const backdoorId = backdoorEl.value

    if (backdoorId !== '') {
      KfsUtils.apiCall(URLS.API.SYS.BACKDOOR_LOGIN, {
        method: 'POST',
        data: { backdoorId }
      }).then(result => {
        backdoorEl.value = ''
        replaceWindowLocation(KfsUtils.getUrlPathPrefix())
      }).catch(error => {
        console.error(error.response.status, error.message)
      })
    }
  }

  backdoorLogout = () => {
    KfsUtils.apiCall(URLS.API.SYS.BACKDOOR_LOGOUT, {})
      .then(_result => {
        replaceWindowLocation(KfsUtils.getUrlPathPrefix())
      })
      .catch(error => {
        console.error(error.response.status, error.message)
      })
  }

  openAboutModal = versions => {
    let aboutHtml = ''
    aboutHtml += '<div class="content about">'
    aboutHtml += '    <div class="modal-header">'
    aboutHtml += '        <h2>ABOUT</h2>'
    aboutHtml +=
      '        <button type="button" data-remodal-action="close" class="close remodal-close">'
    aboutHtml += '          <span aria-hidden="true">&times;</span>'
    aboutHtml += '        </button>'
    aboutHtml += '    </div>'
    aboutHtml += '    <ul class="versions">'
    Object.keys(versions).map(key => {
      aboutHtml += '    <li>' + key + ': ' + versions[key] + '</li>'
    })
    aboutHtml += '    </ul>'
    aboutHtml += '</div>'
    const remodalContent = getFirstElementBySelector('.remodal-content')
    remodalContent.innerHTML = aboutHtml
    $('[data-remodal-id=modal]').remodal().open()
    getFirstElementBySelector('button', remodalContent).focus()
  }

  render () {
    let logoutButton
    if (this.state.backdoorId && this.state.backdoorId !== '') {
      logoutButton = (
        <button
          data-test-id='backdoor-logout'
          type='button'
          className='btn btn-default'
          onClick={this.backdoorLogout}
        >
          Logout
        </button>
      )
    }

    let prefs = this.state.preferences
    let menuLinks = []
    if (prefs.menu) {
      menuLinks = prefs.menu.map(function (link, i) {
        var target = '_blank'
        return (
          <li key={i}>
            <Link
              data-test-id={`header-menu-link-${link.label}`}
              url={link.link}
              label={link.label}
              target={target}
            />
          </li>
        )
      })
    }

    if (prefs.versions) {
      menuLinks.push(
        <li key='about-item'>
          <a
            role='button'
            tabIndex={0}
            data-test-id='header-about'
            onClick={this.openAboutModal.bind(null, prefs.versions)}
            onKeyDown={event => {
              event.preventDefault()
              const key = parseInt(event.keyCode || event.which)
              if (key === 13 || key === 32) {
                this.openAboutModal(prefs.versions)
              }
            }}
          >
            About
          </a>
        </li>
      )
    }

    let logoUrl = prefs.logoUrl
    if (
      logoUrl &&
      logoUrl.indexOf('data:') !== 0 &&
      !logoUrl.startsWith('http')
    ) {
      logoUrl = KfsUtils.getUrlPathPrefix() + logoUrl
    }

    let backdoorIdAppender = KfsUtils.buildBackdoorIdAppender(
      this.state.backdoorId
    )
    let actionListLink = backdoorIdAppender(prefs.actionListUrl)
    let docSearchLink = backdoorIdAppender(prefs.docSearchUrl)
    let testEnv = <div />

    if (this.state.environment && !this.state.environment.prodMode) {
      testEnv = (
        <div id='test-header' className={styles.testHeader}>
          <div className={styles.testInfo}>
            <div className={classnames(styles.testHeaderColumn, styles.firstColumn)}>This is a test environment</div>
            <div data-test-id='impersonation-label' className={classnames(styles.testHeaderColumn, styles.middleColumn)}>
              {this.state.backdoorId
                ? 'Impersonating: ' + this.state.backdoorId
                : ''}&nbsp;
            </div>
            <div className={classnames(styles.testHeaderColumn, styles.lastColumn)}>
              <input
                id='backdoorId'
                type='text'
                placeholder='Back Door ID'
                onKeyPress={this.handleBackDoorEnter}
              />
              <button
                type='button'
                id='backdoorLoginButton'
                className='btn btn-default'
                onClick={this.backdoorLogin}
              >
                Login
              </button>
              {logoutButton}
            </div>
          </div>
        </div>
      )
    }

    let actionListConnector =
      actionListLink && actionListLink.indexOf('?') !== -1 ? '&' : '?'

    return (
      <div>
        <ToastContainer
          closeButton={false}
          hideProgressBar={true}
          style={{
            width: 'auto'
          }}
        />
        <div id='skipmenu'>
          <a href='#sidebar' className={styles.skippy}>
            Skip to menu
          </a>
          {document.getElementById('lookup') !== null ||
          document.getElementById('workarea') !== null ? (
            <a href='#view_div' className={styles.skippy}>
              Skip to main
              </a>
            ) : (
              ''
            )}
        </div>
        <div className={classnames('navbar-header', styles.navbarHeader)}>
          <a className={classnames('navbar-brand', styles.navbarBrand)} href='#d'>
            <img src={logoUrl} alt='kuali logo' height='35px' />
            <span className={styles.logoRight}>Financials</span>
          </a>
        </div>
        <nav className='collapse navbar-collapse'>
          <ul className='nav navbar-nav pull-right'>
            <li>
              <a
                data-test-id='header-action-list'
                href={
                  prefs.remoteViewUrl +
                  '?url=' +
                  escape(
                    actionListLink +
                      actionListConnector +
                      'documentTargetSpec=KFS:_top,*:_blank&routeLogTargetSpec=*:_self'
                  ) +
                  '&title=Action List'
                }
                onClick={stayOnPage}
              >
                <span className='glyphicon glyphicon-ok-sign' />Action List
              </a>
            </li>
            <li>
              <a
                data-test-id='header-doc-search'
                href={
                  prefs.remoteViewUrl +
                  '?url=' +
                  escape(
                    docSearchLink +
                      '&documentTargetSpec=KFS:_top,*:_blank&routeLogTargetSpec=*:_self&showSuperUserButton=false'
                  ) +
                  '&title=Doc Search'
                }
                onClick={stayOnPage}
              >
                <span className='glyphicon glyphicon-search' />Doc Search
              </a>
            </li>
            <li className='dropdown'>
              <a
                data-test-id='header-user'
                href='#d'
                className='dropdown-toggle'
                data-toggle='dropdown'
                title={this.state.user.principalName}
              >
                <span className='glyphicon glyphicon-user' />
                {this.state.user.firstName}&nbsp;
                <span className='caret' />
              </a>
              <ul className='dropdown-menu pull-right'>
                <li>
                  <a
                    data-test-id='header-sign-out'
                    href={prefs.signoutUrl}
                    onClick={stayOnPage}
                  >
                    Sign Out
                  </a>
                </li>
              </ul>
            </li>
            <li className='dropdown'>
              <a
                data-test-id='header-menu'
                href='#d'
                id='nbAcctDD'
                className='dropdown-toggle'
                data-toggle='dropdown'
              >
                <i className='icon-user' />
                <span className='glyphicon glyphicon-menu-hamburger' />Menu&nbsp;
                <span className='caret' />
              </a>
              <ul className='dropdown-menu pull-right'>{menuLinks}</ul>
            </li>
          </ul>
        </nav>
        {testEnv}
      </div>
    )
  }
}

export default Header
